require 'rails_helper'

RSpec.describe "Evaluations", :type => :request do
  before do
    @user = create(:examiner)
    login_as @user
  end

  describe "GET /claims/:claim_id/evaluations/:id/edit" do
    context "when a user makes a change" do
      before do
        @evaluation_spec = create(:diabetes_spec)
        @claim = create(:claim)
        @claim.contentions << create(:contention)
        @evaluation = Evaluation.new(claim: @claim, evaluation_spec: @evaluation_spec, user: @user)
        @claim.contentions.first.evaluations << @evaluation
        visit edit_claim_evaluation_path(@claim, @claim.contentions.first.evaluations.first)
      end

      xit "changes the evaluation last update time", js: true do
        last_update = find("#evaluation-updated").text
        choose 'hasDiabetes_yes'
        sleep 62
        expect(find("#evaluation-updated")).to_not have_content last_update
      end
    end

    context "when a claim has suggested exams" do
      before do
        @evaluation_spec = create(:suggested_exam_spec)
        @claim = create(:claim)
        @claim.contentions << create(:contention, history: "This or That")
        @evaluation = Evaluation.new(claim: @claim, evaluation_spec: @evaluation_spec, user: @user)
        @claim.contentions.first.evaluations << @evaluation
      end

      context "when those exams don't exist" do
        before do
          visit edit_claim_evaluation_path(@claim, @claim.contentions.first.evaluations.first)
        end

        it "doesn't shows suggested exams" do
          expect(page).to_not have_content "Suggested Exams"
        end
      end

      context "when they do" do
        before do
          EvaluationSpec.create(spec: suggested_exam_spec('Alpha', 'Alpha Exam'), active: true)
          EvaluationSpec.create(spec: suggested_exam_spec('Beta', 'Beta Exam'), active: true)
          visit edit_claim_evaluation_path(@claim, @claim.contentions.first.evaluations.first)
        end

        it "shows suggested exams" do
          expect(page).to have_content 'Suggested Exams'
          expect(page).to have_content 'Alpha Exam'
          expect(page).to have_content 'Beta Exam'
        end

        # return all evaluations whose spec has a given title
        def eval_where_title(title)
          Evaluation.all.select {|e| e.evaluation_spec.title == title}
        end

        it "creates exams when selected" do
          choose 'yesno_yes'
          check "_suggested_Alpha"
          l = Evaluation.all.length
          click_button 'Review'
          expect(Evaluation.all.length).to eq l+1
          expect(eval_where_title('Alpha Exam').length).to eq 1
          expect(eval_where_title('Beta Exam').length).to eq 0
        end
      end
    end

    context "when a claim has an unsatisfied dependency" do
      before do
        @evaluation_spec = create(:dependent_spec)
        @claim = create(:claim)
        @claim.contentions << create(:contention, history: "This or That")
        @evaluation = Evaluation.new(claim: @claim, evaluation_spec: @evaluation_spec, user: @user)
        @claim.contentions.first.evaluations << @evaluation
      end

      it "should allow the user to save as draft" do
          visit edit_claim_evaluation_path(@claim, @evaluation)
          click_button 'Save as draft'
          expect(page).to have_content "Saved as draft"
      end

      it "should not allow the user to review" do
          visit edit_claim_evaluation_path(@claim, @evaluation)
          choose 'yesno_yes'
          click_button 'Review'
          expect(page).to have_content 'This evaluation requires a parent evaluation with body_system "endocrine" but none was found'
      end
    end

    context "when a claim has a satisfied dependency" do
      before do
        @claim = create(:claim)
        @claim.contentions << create(:contention, history: "This or That")

        @satisfied_spec = create(:dependent_spec)
        @satisfied_eval = Evaluation.new(claim: @claim, evaluation_spec: @satisfied_spec, user: @user)
        @claim.contentions.first.evaluations << @satisfied_eval

        @diabetes_spec = create(:diabetes_spec)
        @diabetes_eval = Evaluation.new(claim: @claim, evaluation_spec: @diabetes_spec, completed_at: Time.now, user: @user)
        @claim.contentions.first.evaluations << @diabetes_eval

        @satisfied_eval.primary_evaluation_id = @diabetes_eval.id
        @satisfied_eval.save
      end

      it "should allow the user to save as draft" do
        visit edit_claim_evaluation_path(@claim, @satisfied_eval)
        click_button 'Save as draft'
        expect(page).to have_content "Saved as draft"
      end

      it "should allow the user to review" do
        visit edit_claim_evaluation_path(@claim, @satisfied_eval)
        choose 'yesno_yes'
        click_button 'Review'
        expect(page).to have_content 'Please review'
      end
    end

    context "when a claim has an incomplete satisfied dependency" do
      before do
        @claim = create(:claim)
        @claim.contentions << create(:contention, history: "This or That")

        @satisfied_spec = create(:dependent_spec)
        @satisfied_eval = Evaluation.new(claim: @claim, evaluation_spec: @satisfied_spec, user: @user)
        @claim.contentions.first.evaluations << @satisfied_eval

        @diabetes_spec = create(:diabetes_spec)
        @diabetes_eval = Evaluation.new(claim: @claim, evaluation_spec: @diabetes_spec, user: @user)
        @claim.contentions.first.evaluations << @diabetes_eval

        @satisfied_eval.primary_evaluation_id = @diabetes_eval.id
        @satisfied_eval.save
      end

      it "should allow the user to save as draft" do
        visit edit_claim_evaluation_path(@claim, @satisfied_eval)
        click_button 'Save as draft'
        expect(page).to have_content "Saved as draft"
      end

      it "should not allow the user to review" do
          visit edit_claim_evaluation_path(@claim, @satisfied_eval)
          choose 'yesno_yes'
          click_button 'Review'
          expect(page).to have_content 'This evaluation requires a parent evaluation with body_system "endocrine" but none was found'
      end
    end

    context "when a claim has a contention that has an evaluation" do
      before do
        @evaluation_spec = create(:diabetes_spec)
        @claim = create(:claim)
        @claim.contentions << create(:contention)
        @evaluation = Evaluation.new(claim: @claim, evaluation_spec: @evaluation_spec, user: @user)
        @claim.contentions.first.evaluations << @evaluation
      end

      context "which is completed" do
        it "redirects to the show url" do
          @evaluation.update_attributes({ completed_at: Time.now, doc: build(:diabetes_doc).doc })
          get edit_claim_evaluation_path(@claim, @evaluation)
          expect(response).to redirect_to(claim_evaluation_path(@claim, @evaluation))
          expect(request.flash[:notice]).to eq("Evalauation is already completed")
        end
      end

      context "when viewing an empty form" do
        before do
          visit edit_claim_evaluation_path(@claim, @claim.contentions.first.evaluations.first)
        end

        it "should show the evaluation spec title and version" do
          expect(page).to have_content @evaluation.evaluation_spec.title
          expect(page).to have_content "Version: #{@evaluation.evaluation_spec.version}"
        end

        it "should not indicate that the form is valid" do
          expect(page).to have_no_content "Validates"
        end
      end

      context "when saving as a draft" do
        it "should save the information and reload the form" do
          visit edit_claim_evaluation_path(@claim, @claim.contentions.first.evaluations.first)
          fill_in 'Test Number', with: '123'
          expect(page.has_select?('primary_evaluation_id', :selected => '--none--')).to eq false
          page.select('--none--', :from => 'primary_evaluation_id')
          click_button 'Save as draft'
          expect(page).to have_content "Saved as draft"
          expect(page).to have_no_content "Validation errors"
          expect(page).to have_content 'Diabetes'
          expect(find_field('Test Number').value).to eq "123"
          expect(page.has_select?('primary_evaluation_id', :selected => '--none--')).to eq true
          visit claim_contention_path(@claim, @claim.contentions.first)
          expect(page).to have_content "Draft"
        end
      end

      context "when viewing a draft" do
        before do
          @evaluation.update_attributes({ doc: build(:diabetes_doc).doc })
          visit edit_claim_evaluation_path(@claim, @claim.contentions.first.evaluations.first)
        end

        it "renders partial dates correctly" do
          expect(page).not_to have_checked_field('fastingPlasmaGlucoseOccurence1_estimate')
          expect(page).to have_checked_field('fastingPlasmaGlucoseOccurence1_unknown')

          expect(page).to have_checked_field('fastingPlasmaGlucoseOccurence2_estimate')
          expect(page).not_to have_checked_field('fastingPlasmaGlucoseOccurence2_unknown')
        end

      end

       context "when submitting an evaluation for review" do
          it "displays the evaluation, the complete button and the pdf button" do
            visit claim_contention_path(@claim, @claim.contentions.first)
            click_link @evaluation_spec.title
            expect(page).to have_content 'Diabetes'
            expect(page).to have_content 'Does the veteran currently have diabetes mellitus?'
            choose 'hasDiabetes_yes'
            check 'fastingPlasmaGlucoseTest'
            select '2014', from: 'fastingPlasmaGlucoseOccurence1_year'
            select 'February', from: 'fastingPlasmaGlucoseOccurence2_month'
            select '2013', from: 'fastingPlasmaGlucoseOccurence2_year'
            choose 'testRadio_2'
            select 'Third', from: 'testSelect'
            fill_in 'Test Number', :with => '6.3'
            click_button 'Review'
            expect(page).to have_content 'Please review your application, check for errors, and hit complete when you are satisfied.'
            expect(page).to have_text 'Does the veteran currently have diabetes mellitus?'
            expect(page).to have_content 'Date: February 2013'
            expect(page).to have_selector(:link_or_button, 'Complete')
            expect(page).to have_link 'PDF'
          end
        end

      context "when starting an evaluation" do
        context "when number is invalid" do
          it "should error appropriately" do
            visit claim_contention_path(@claim, @claim.contentions.first)
            click_link @evaluation_spec.title
            expect(page).to have_content 'Diabetes'
            expect(page).to have_content 'Does the veteran currently have diabetes mellitus?'
            choose 'hasDiabetes_yes'
            check 'fastingPlasmaGlucoseTest'
            select '2014', from: 'fastingPlasmaGlucoseOccurence1_year'
            select 'February', from: 'fastingPlasmaGlucoseOccurence2_month'
            select '2013', from: 'fastingPlasmaGlucoseOccurence2_year'
            choose 'testRadio_2'
            select 'Third', from: 'testSelect'
            fill_in 'Test Number', :with => 'invalid number duder'
            click_button 'Review'
            expect(page).to have_content "Invalid number"
          end
        end

        context "when all required information is provided and reviewed" do
          before do
            @evaluation = @claim.contentions.first.evaluations.first
            visit claim_contention_path(@claim, @claim.contentions.first)
            click_link @evaluation_spec.title
            expect(page).to have_content 'Diabetes'
            expect(page).to have_content 'Does the veteran currently have diabetes mellitus?'
            choose 'hasDiabetes_yes'
            check 'fastingPlasmaGlucoseTest'
            select '2014', from: 'fastingPlasmaGlucoseOccurence1_year'
            select 'February', from: 'fastingPlasmaGlucoseOccurence2_month'
            select '2013', from: 'fastingPlasmaGlucoseOccurence2_year'
            choose 'testRadio_2'
            select 'Third', from: 'testSelect'
            fill_in 'Test Number', :with => '6.3'
            click_button 'Review'
          end

          it "should allow the examnier to complete the evaluation with all required information and send to VBMS" do
            fill_in 'VHA User Electronic Signature', with: 'Signature'
            fill_in 'VHA User VistA Access Code', with: '12455555'
            fill_in 'VHA User VistA Verify Code', with: '67444441'
            click_button 'Complete'
            expect(page).to have_content "Evaluation completed"
            expect(page).to have_content "Claim: #{@claim.patient_name} #{@claim.created_at.to_date}"
            expect(page).to have_content "SSN: #{@claim.masked_ssn}"
            visit claim_contention_path(@claim, @claim.contentions.first)
            expect(page).to have_content "Completed"
            visit claim_evaluation_path(@claim, @evaluation)
            expect(page).to have_no_content "Signature"
            expect(page).to have_no_content "12455555"
            expect(page).to have_no_content "67444441"
          end

          context "when VistA credentials are missing" do
            it "should show an error message to the user" do
              click_button 'Complete'
            end
          end
        end

        context "when required information is omitted" do
          it "should show an error message and ask the user to correct the page" do
            visit claim_contention_path(@claim, @claim.contentions.first)
            click_link @evaluation_spec.title
            choose 'hasDiabetes_yes'
            check 'fastingPlasmaGlucoseTest'
            select '2014', from: 'fastingPlasmaGlucoseOccurence1_year'
            click_button 'Review'
            expect(page).to have_content "This form contains the following errors"
            expect(page).to have_checked_field('fastingPlasmaGlucoseTest')
            expect(page).to have_checked_field('hasDiabetes_yes')
            expect(page.find_field("fastingPlasmaGlucoseOccurence1_year").value).to eq "2014"
            expect(page).to have_content "This field is required"
          end
        end

        context "when submitting a previously-invalid form" do
          it "should not show old error messages" do
            visit edit_claim_evaluation_path(@claim, @claim.contentions.first.evaluations.first)
            choose 'hasDiabetes_yes'
            click_button 'Save as draft'
            expect(page).to have_content "Saved as draft"
            click_button 'Review'
            expect(page).to have_content 'This form contains the following errors'
            expect(page).to have_no_content 'Saved as draft'
            check 'fastingPlasmaGlucoseTest'
            select '2014', from: 'fastingPlasmaGlucoseOccurence1_year'
            select 'February', from: 'fastingPlasmaGlucoseOccurence2_month'
            select '2013', from: 'fastingPlasmaGlucoseOccurence2_year'
            choose 'testRadio_2'
            select 'Third', from: 'testSelect'
            fill_in 'Test Number', :with => '6.3'
            click_button 'Save as draft'
            expect(page).to have_content 'Validates'
            expect(page).to have_no_content 'Validation errors'
          end
        end

        context "when completing an evaluation with conditional fields that are conditional on other conditional fields" do
          before do
            @conditional_evaluation_spec = create(:conditional_spec)
            @claim.contentions << create(:contention, name: "Broken arm")
            @claim.contentions.last.evaluations << Evaluation.new(claim: @claim, evaluation_spec: @conditional_evaluation_spec, user: @user)
            visit edit_claim_evaluation_path(@claim, @claim.contentions.last.evaluations.first)
          end

#          it "should reset conditional fields that are conditional on other conditional fields when parent conditionals get reset", js: true do
#            expect(page).to have_content "First"
#            expect(page).to have_no_content "Second"
#            expect(page).to have_no_content "Third"
#            choose "first_yes"
#            expect(page).to have_content "First"
#            expect(page).to have_content "Second"
#            expect(page).to have_no_content "Third"
#            choose "second_yes"
#            expect(page).to have_content "First"
#            expect(page).to have_content "Second"
#            expect(page).to have_content "Third"
#            choose "third_yes"
#            fill_in "third_textarea", with: "Some text"
#            choose "first_no"
#            expect(page).to have_content "First"
#            expect(page).to have_no_content "Second"
#            expect(page).to have_no_content "Third"
#            choose "first_yes"
#            expect(page).to have_content "First"
#            expect(page).to have_content "Second"
#            expect(page).to have_no_checked_field("second_yes")
#            expect(page).to have_no_content "Third"
#            choose "second_yes"
#            expect(page).to have_content "First"
#            expect(page).to have_content "Second"
#            expect(page).to have_content "Third"
#            expect(page).to have_no_checked_field "third_yes"
#            expect(page).to have_field("third_textarea", with: "")
#          end

#          it "should allow for markdown fields to be conditional", js: true do
#            expect(page).to have_no_content "Some markdown text should only appear when Yes is clicked."
#            choose "first_yes"
#            expect(page).to have_content "Some markdown text should only appear when Yes is clicked."
#          end
        end

        context "when a second evaluation is added to the claim" do
          before do
            @conditional_evaluation_spec = create(:conditional_spec)
            @claim.contentions << create(:contention, name: "Broken arm")
            @claim.contentions.last.evaluations << Evaluation.new(claim: @claim, evaluation_spec: @conditional_evaluation_spec, user: @user)
            visit edit_claim_evaluation_path(@claim, @claim.contentions.first.evaluations.first)
          end

          it "should have a primary_evaluation_id selection option for the eval not currently being edited" do
            expect(page.has_select?('primary_evaluation_id', :options =>  ['Select a primary evaluation', 'Conditional test', '--none--'])).to eq true
          end

          it "should save the selection for primary_evaluation_id and reload the form" do
            expect(page.has_select?('primary_evaluation_id', :selected => 'Conditional test')).to eq false
            page.select('Conditional test', :from => 'primary_evaluation_id')
            click_button 'Save as draft'
            expect(page.has_select?('primary_evaluation_id', :selected => 'Conditional test')).to eq true
          end
        end
      end

      context "when a contention has an evaluation with a spec with disabled fields" do
        before do
          @claim.contentions.first.evaluations << Evaluation.new(claim: @claim, evaluation_spec: create(:disabled_spec), user: @user)
        end

        it "show show the form with disabled fields" do
          visit edit_claim_evaluation_path(@claim, @claim.contentions.first.evaluations.last)
          expect(page.find("#sampleYesNo_yes").disabled?).to be_truthy
          expect(page.find("#sampleYesNo_no").disabled?).to be_truthy
          expect(page.find("#sampleCheckbox").disabled?).to be_truthy
          expect(page.find("#sampleDate").disabled?).to be_truthy
          expect(page.find("#sampleTextarea").disabled?).to be_truthy
          expect(page.find("#sampleString").disabled?).to be_truthy
          expect(page.find("#sampleRadio_1").disabled?).to be_truthy
          expect(page.find("#sampleRadio_2").disabled?).to be_truthy
          expect(page.find("#sampleRadio_3").disabled?).to be_truthy
          expect(page.find("#sampleSelect").disabled?).to be_truthy
          expect(page.find("#samplePartialDate_day").disabled?).to be_truthy
          expect(page.find("#samplePartialDate_month").disabled?).to be_truthy
          expect(page.find("#samplePartialDate_year").disabled?).to be_truthy
          expect(page.find("#samplePartialDate_unknown").disabled?).to be_truthy
          expect(page.find("#samplePartialDate_estimate").disabled?).to be_truthy
        end
      end
    end

    context "when you don't own a spec" do
      before do
        @user2 = create(:user)
        @evaluation_spec = create(:diabetes_spec)
        @claim = create(:claim)
        @claim.contentions << create(:contention)
        @evaluation = Evaluation.new(claim: @claim, evaluation_spec: @evaluation_spec, user: @user2)
        @claim.contentions.first.evaluations << @evaluation
      end

      it "should not let you edit it" do
        visit edit_claim_evaluation_path(@claim, @claim.contentions.first.evaluations.last)
        expect(page).to have_content "You are not authorized to access this page"
        expect(page.current_path).to eq root_path
      end
    end
  end

  describe "PUT /claims/:claim_id/evaluations/:id" do
    context "when a claim has a contention that has an evaluation" do
      before do
        @evaluation_spec = create(:diabetes_spec)
        @claim = create(:claim)
        @claim.contentions << create(:contention)
        @evaluation = Evaluation.new(claim: @claim, evaluation_spec: @evaluation_spec, user: @user)
        @claim.contentions.first.evaluations << @evaluation
      end

      context "which is completed" do
        it "redirects to the show url" do
          @evaluation.update_attributes({ completed_at: Time.now, doc: build(:diabetes_doc).doc })
          put claim_evaluation_path(@claim, @evaluation)
          expect(response).to redirect_to(claim_evaluation_path(@claim, @evaluation))
          expect(request.flash[:notice]).to eq("Evalauation is already completed")
        end
      end
    end
  end

  describe "GET /claims/:claim_id/evaluations/:id" do
    before do
      @evaluation_spec = create(:diabetes_spec)
      @claim = create(:claim)
      @claim.contentions << create(:contention)
      @evaluation = Evaluation.new(claim: @claim, evaluation_spec: @evaluation_spec, doc: build(:diabetes_doc).doc, user: @user)
      @claim.contentions.first.evaluations << @evaluation
      visit claim_evaluation_path(@claim, @claim.contentions.first.evaluations.first)
    end

    it "should show the evaluation title and version" do
      expect(page).to have_content @evaluation.evaluation_spec.title
      expect(page).to have_content "Version: #{@evaluation.evaluation_spec.version}"
    end

    it "should show the form" do
      expect(page).to have_content "Fasting plasma glucose"
      expect(page).to have_no_content "Random plasma glucose"
      expect(page).to have_content "1 January 2014"
      expect(page).to have_content "2 January 2014"
      expect(page).to have_content "This checkbox SHOULD show up."
      expect(page).to have_no_content "This checkbox should NOT show up."

      page.within('.formrow', text: 'Fasting Date 1') do
        expect(page).not_to have_content('Unknown')
        expect(page).to have_content('Estimate')
      end

      page.within('.formrow', text: 'Fasting Date 2') do
        expect(page).to have_content('Unknown')
        expect(page).not_to have_content('Estimate')
      end
    end

    context "renders the field using conditionalOn logic" do
      it "on boolean fields"  do
        expect(page).to have_selector('label', text: "Visible Checkbox")
        expect(page).to have_no_selector('label', text: "Hidden Checkbox")
      end
    end

    it "should link to a PDF and XML version of the evaluation" do
      expect(page).to have_link 'PDF', href: claim_evaluation_path(@claim, @claim.contentions.first.evaluations.first, format: :pdf)
      expect(page).to have_link 'XML', href: claim_evaluation_path(@claim, @claim.contentions.first.evaluations.first, format: :xml)
      expect(page).to have_link 'Specification', href: admin_evaluation_spec_path(@claim.contentions.first.evaluations.first.evaluation_spec)
    end
  end
end
