# == Schema Information
#
# Table name: providers
#
#  id             :integer          not null, primary key
#  npi            :string           not null
#  name           :string           not null
#  physician_name :string           not null
#  content        :json
#  created_at     :datetime         not null
#  updated_at     :datetime         not null
#  facility_id    :integer
#

require 'rails_helper'

RSpec.describe Provider, :type => :model do

  context "Associations", cpp: true do
    it { should belong_to(:facility) }

    it { should have_one(:visn).through(:facility) }

    it { should have_and_belong_to_many(:medical_specialties) }

    it { should have_many(:referrals) }
    it { should have_many(:consultations).through(:referrals) }
    it { should have_many(:providers_users).dependent(:destroy) }
    it { should have_many(:users).through(:providers_users) }
  end

  context "methods" do
    let (:address_attributes) {
      {"city"=>"SomeCity",
       "address_2"=>"",
       "telephone_number"=>"333-555-0000",
       "fax_number"=>"333-556-0000",
       "state"=>"MD",
       "postal_code"=>"21045",
       "address_1"=>"123 SOME PLACE",
       "country_code"=>"US",
       "country_name"=>"United States",
       "address_type"=>"DOM",
       "address_purpose"=>"LOCATION"}
    }
    let(:address) {[address_attributes]}
    let(:address_with_zip_plus_4) {[address_attributes.merge("postal_code"=>"210456666")]}
    let(:taxonomy) {[{"state"=>"MD",
        "code"=>"0000C0700X",
        "primary"=>true,
        "license"=>"XX#0000",
        "desc"=>"SomeDescription"}]}
    let(:visn) { Visn.first }
    let(:facility) { create(:facility, visn: visn) }
    # facility2 must have different VISN for visn_id params query test to pass
    let(:facility2) { create(:facility, visn: Visn.second) }
    let!(:provider)  { create(:provider, content: {addresses: address, taxonomies: taxonomy}) }
    let!(:provider_1_in_second_visn) { create(:provider, content: {addresses: address, taxonomies: taxonomy}, facility: facility2) }
    let!(:provider_2_in_second_visn) { create(:provider, content: {addresses: address, taxonomies: taxonomy}, facility: facility2) }
    let!(:provider_with_zip_plus_4)  { create(:provider, content: {addresses: address_with_zip_plus_4, taxonomies: taxonomy}, facility: facility2) }

    it "should return a hash with address parameters", cpp: true do
      expect(provider.address_attrs).to include(address: "123 SOME PLACE ", city: "SomeCity", state: "MD", zip: "21045")
    end

    it "should return address", cpp: true do
      expect(provider.address).to start_with address.first["address_1"]
      expect(provider.address).to end_with address.first["address_2"]
    end

    it "should return formatted address", cpp: true do
      expect(provider.formatted_address_string).to eq "123 SOME PLACE , SomeCity, MD 21045"
    end

    it "should return city", cpp: true do
      expect(provider.city).to eq address.first["city"]
    end

    it "should return state", cpp: true do
      expect(provider.state).to eq address.first["state"]
    end

    it "should return zip", cpp: true do
      expect(provider.zip).to eq address.first["postal_code"]
    end

    it "should return formatted zip", cpp: true do
      expect(provider_with_zip_plus_4.zip).to eq "21045-6666"
    end

    it "should return phone no", cpp: true do
      expect(provider.phone).to eq address.first["telephone_number"]
    end

    it "should return fax no", cpp: true do
      expect(provider.fax).to eq address.first["fax_number"]
    end

    it "should return medical speciality", cpp: true do
      expect(provider.medical_specialty_names).to eq taxonomy.first["desc"]
    end

    # SMELL: goverly generic names 'provider1', etc. make the logic of this test hard to understand at first glance.
    it "should return filters based on visn_id params query", cpp: true do
      result1 = Provider.filter({visn_id: provider.facility.visn_id})
      result2 = Provider.filter({visn_id: facility2.visn_id})
      
      expect(result1.first).to eq(provider)
      expect(result2).to match_array([provider_1_in_second_visn, provider_2_in_second_visn, provider_with_zip_plus_4])
    end

    it "should return filters based on name params query", cpp: true do
      result = Provider.filter({name: provider_1_in_second_visn.name})
      expect(result.count).to eq(1)
      expect(result.first).to eq(provider_1_in_second_visn)
    end

    it "should return filters based on physician_name params query", cpp: true do
      result = Provider.filter({physician_name: provider_2_in_second_visn.physician_name})
      expect(result.count).to eq(1)
      expect(result.first).to eq(provider_2_in_second_visn)
    end

    it "should return all providers without filter", cpp: true do
      expect(Provider.filter).to match_array([provider, provider_1_in_second_visn, provider_2_in_second_visn,
                                              provider_with_zip_plus_4])
    end
  end
end
