# == Schema Information
#
# Table name: claims
#
#  id                :integer          not null, primary key
#  completed_at      :datetime
#  created_at        :datetime
#  updated_at        :datetime
#  patient_ssn       :string
#  file_number       :string
#  first_name        :string
#  middle_name       :string
#  last_name         :string
#  date_of_birth     :date
#  facility_name     :string
#  facility_number   :string
#  exam_request_data :text
#  vbms_claim_id     :string
#

require 'rails_helper'

RSpec.describe Claim, :type => :model do

  context "Associations", cui: true do
    it { should have_many(:contentions).
         dependent(:destroy)
    }
    it { should have_many(:evaluations).
         dependent(:destroy)
    }
    it { should have_many(:examinations) }
    it { should have_many(:exam_requests) }
    it { should have_many(:examination_schedules).
         through(:examinations)
    }
  end

  context "#masked_ssn" do
    let(:claim) { build(:claim, patient_ssn: '111223333') }
    it "masks area and group codes", cui: true do
      expect(claim.masked_ssn).to eq "XXX-XX-3333"
    end
  end

  context "#patient_name" do
    let(:claim) { create(:claim, first_name: 'First', middle_name: 'Middle', last_name: 'Last') }
    it "returns a full name when all names are present", cui: true do
      expect(claim.patient_name).to eq "First Middle Last"
    end

    it "returns a full name when only first and last are present", cui: true do
      claim.update_attribute(:middle_name, nil)
      expect(claim.patient_name).to eq "First Last"
    end
  end

  context "auto_strip_attributes" do
    let(:claim) { create(:claim, first_name: 'A  ', middle_name: 'Longish ', last_name: 'Name    ', patient_ssn: '111223333 ', file_number: '111 ', date_of_birth: '04/01/1980 ') }

    %w(first_name middle_name last_name patient_ssn file_number date_of_birth).each do |field|
      it "strips trailing whitespace from #{field}", cui: true do
        expect(claim[field]).not_to end_with ' '
      end
    end
  end

  context "#create_from_exam_request_xml" do
    before do
      @claim = Claim.create_from_exam_request_xml(File.read("spec/support/exam_request_sample.xml"))
    end

    it "should set the basic claim information", cui: true do
      expect(@claim.first_name).to eq "Gladys"
      expect(@claim.last_name).to eq "Pippenwanker-Wienerdoodle"
      expect(@claim.middle_name).to eq "Gertrude"
      expect(@claim.patient_ssn).to eq "111995588"
      expect(@claim.file_number).to eq "987654"
      expect(@claim.date_of_birth).to eq Date.parse("1945-05-29")
      expect(@claim.facility_name).to eq "Bugtussle VA"
      expect(@claim.facility_number).to eq "123"
    end
  end
end
