require 'rails_helper'

RSpec.feature "Validation errors are highlighted" do
  let!(:evaluation_spec) { FactoryBot.create(:evaluation_spec) }
  let!(:admin) { FactoryBot.create(:admin) }
  let!(:claim) { FactoryBot.create(:claim) }
  let!(:contention) { FactoryBot.create(:contention, claim: claim) }
  let!(:contention_detail) {FactoryBot.create(:contention_detail, contention: contention, name: "knee hurts")}
  let!(:evaluation) { FactoryBot.create(:evaluation, claim: claim) }
  let!(:diagnosis_modifier) { FactoryBot.create(:diagnosis_modifier) }
  let!(:diagnosis) { FactoryBot.create(:diagnosis, diagnosis_modifier_id: diagnosis_modifier) }
  let!(:site) { FactoryBot.create(:site) }
  let!(:exam_request) { FactoryBot.create(:exam_request, claim: claim, identifier: "lsuejh", site: site )}
  let!(:major_system) { FactoryBot.create(:major_system) }
  let!(:minor_system) { FactoryBot.create(:minor_system, major_system: major_system) }
  let!(:notification_log) { FactoryBot.create(:notification_log)}
  let!(:user) { FactoryBot.create(:user) }

  
  context "Evaluation spec page" do
    before do
      login_as(admin)
    end
    scenario "on edit: when form is saved empty" do
      visit edit_admin_evaluation_spec_path(evaluation_spec)

      fill_in 'evaluation_spec_spec', :with => ""
      click_button "Save"
      
      expect(find('.alerts.alert-box.alert').text).to eq "Error updating spec: Could not find required `form` attribute in false"
      expect(find('label.form_with_errors').text).to match(/Spec/)
      expect(find('form textarea')[:class]).to match(/form_with_errors/) 
    end

    scenario "on edit: when form is empty & then previewed" do
      visit edit_admin_evaluation_spec_path(evaluation_spec)

      fill_in 'evaluation_spec_spec', :with => ""
      click_button "Preview"
      
      expect(find('.alerts.alert-box.alert').text).to eq "Invalid Spec: Could not find required `form` attribute in false"
      expect(find('label.form_with_errors').text).to match(/Spec/)
      expect(find('form textarea')[:class]).to match(/form_with_errors/) 
    end

    scenario "on new: when form is saved empty" do
      visit new_admin_evaluation_spec_path(evaluation_spec)

      fill_in 'evaluation_spec_spec', :with => ""
      click_button "Create evaluation"
      
      expect(find('.alerts.alert-box.alert').text).to eq "Error creating spec: Could not find required `form` attribute in false"
      expect(find('label.form_with_errors').text).to match(/Create a new Evaluation/)
      expect(find('form textarea')[:class]).to match(/form_with_errors/) 
    end
  end

  context "Claims page" do
    before do
      login_as(admin)
    end
    scenario "on new: when form is saved empty" do
      visit new_claim_path

      click_button "Create claim"
      
      error_explanations = ["First name can't be blank","Last name can't be blank","Patient ssn can't be blank","File number can't be blank"]
      expect(all('#error_explanation li').length).to eq 4

      all('#error_explanation li').each do |err|
        expect(error_explanations).to include(err.text)
      end
      
      assert_selector('label.form_with_errors', :count => 4)
      all('label.form_with_errors').each do |l|
        expect(['First name (required)', 'Last name (required)', 'Patient Social Security Number (required)', 'File number (required)']).to include(l.text)
      end

      assert_selector('input.form_with_errors', :count => 4)
      all('input.form_with_errors').each do |l|
        expect(['claim_first_name', 'claim_last_name','claim_patient_ssn','claim_file_number']).to include(l[:id])
      end
    end
  end

  context "Contentions page" do
    before do
      login_as(admin)
    end
    scenario "on new: when form is saved empty" do
      visit new_claim_contention_path(claim)

      click_button "Create contention"
      
      expect(find('.alerts.alert-box.alert').text).to eq "Name can't be blank"
      expect(find('label')[:class]).to match(/form_with_errors/)
      expect(find('span.required')[:class]).to match(/form_with_errors/)
      expect(find('input#contention_name')[:class]).to match(/form_with_errors/)
    end
  end

  context "Password reset form" do
    scenario "when email is left blank" do
      visit new_user_password_path

      click_button "Send me reset password instructions"

      expect(find('#error_explanation h2').text).to eq "1 error prohibited this user from being saved:"
      expect(find('label.form_with_errors').text).to eq "Email"
      expect(find('.field_with_errors input')[:class]).to  match(/form_with_errors/)
    end
  end

  context "User sign up" do
    scenario "when all fields are blank" do
      visit new_user_registration_path

      click_button "Sign up"

      expect(find("#error_explanation h2").text).to eq "4 errors prohibited this user from being saved:"

      assert_selector('#error_explanation li', :count => 4)
      all('#error_explanation li').each do |m|
        expect(["First name can't be blank","Last name can't be blank","Email can't be blank","Password can't be blank"]).to include(m.text)
      end
      
      assert_selector('.field_with_errors label', :count => 4)
      all('.field_with_errors label').each do |l|
        expect(['First name', 'Last name', 'Email', 'Password', 'Password confirmation']).to include(l.text)
      end
      expect(find('label.form_with_errors').text).to eq 'Password confirmation'

      assert_selector('input.form_with_errors', :count => 5)
      all('.field input').each do |i|
        expect(i[:class]).to match(/form_with_errors/) 
      end
    end
  end

  context "User sign in" do
    scenario "with password is left blank" do
      visit new_user_session_path

      fill_in "Email", with: "samantha@smith.com"
      fill_in "Password", with: ""
      click_button ""

      expect(find(".alerts.alert-box.alert").text).to eq "Invalid email address or password."
      assert_selector('label.form_with_errors', :count => 2)
      all('label.form_with_errors').each do |l|
        expect(['Email','Password']).to include l.text
      end

      assert_selector('input.form_with_errors', :count => 2)

    end
  end

  context "Resend unlock instructions" do
    scenario "with email left blank" do
      visit new_user_unlock_path

      fill_in "Email", with: ""
      click_button "Resend unlock instructions"

      expect(find(".field_with_errors label").text).to eq "Email"
      expect(find(".field_with_errors input")[:class]).to  match(/form_with_errors/)
    end
  end

  context "Diagnoses pages:" do
    before do
      login_as(admin)
    end
    scenario "on Diagnosis new page: when label is left blank" do
      visit new_diagnosis_path

      fill_in "Name", with: "name1"
      fill_in "Label", with: ""
      fill_in "Code", with: "code1"
      click_button "Save"

      expect(find('#error_explanation li').text).to eq "* Label can't be blank"
      assert_selector('.field_with_errors label', :count => 1)
      expect(find('.field_with_errors label').text).to eq "Label"
      assert_selector('input.form_with_errors', :count => 1)
    end
  end

  context "Diagnosis Modifier pages:" do
    before do
      login_as(admin)
      FactoryBot.create(:diagnosis_modifier, name: "nm2")
    end
    scenario "on Diagnosis Modifier new page: when name is left blank" do
      visit new_diagnosis_modifier_path

      fill_in "Name", with: ""
      click_button "Save"

      expect(find('#error_explanation li').text).to eq "* Name can't be blank"
      assert_selector('.field_with_errors label', :count => 1)
      expect(find('.field_with_errors label').text).to eq "Name"
      assert_selector('input.form_with_errors', :count => 1)
    end

    scenario "on Diagnosis Modifier edit page: when name is not unique" do
      visit edit_diagnosis_diagnosis_modifier_path(diagnosis, diagnosis_modifier)

      fill_in "Name", with: "nm2"
      click_button "Update Fields"

      expect(find('#error_explanation li').text).to eq "* Name has already been taken"
      assert_selector('.field_with_errors label', :count => 1)
      expect(find('.field_with_errors label').text).to eq "Name"
      assert_selector('input.form_with_errors', :count => 1)
    end
  end

  context "Evaluation Template pages:" do
    before do
      login_as(admin)
    end
    scenario "on new page: when name/label is left blank" do
      visit new_evaluation_template_path

      fill_in "Name", with: ""
      fill_in "Label", with: ""
      click_button "Save"

      expect(all('#error_explanation li')[0].text).to eq "* Name can't be blank"
      expect(all('#error_explanation li')[1].text).to eq "* Label can't be blank"
      assert_selector('.field_with_errors label', :count => 2)
      expect(all('.field_with_errors label')[0].text).to eq "Name"
      expect(all('.field_with_errors label')[1].text).to eq "Label"
      assert_selector('input.form_with_errors', :count => 2)
    end
  end

  context "Evaluation pages:" do
    before do
      login_as(admin)
    end
    scenario "on new page: when evaluation_spec is left blank" do
      visit "/claims/#{claim.id}/evaluations/new?contention_id=#{contention.id}"

      click_button "Add evaluation"

      expect(find('.alert.alert-box.alerts').text).to eq "Error adding evaluation to contention: Evaluation spec can't be blank"
      expect(find('label.form_with_errors').text).to eq "Select new evaluation"
      assert_selector('label.form_with_errors', :count => 1)
      
    end
  end

  context "Exam Request actions" do
    before do
      login_as(admin)
    end
    scenario "select contentions page: " do
      visit "/exam_request/get_clarification?er_id=lsuejh"

      click_button "Submit"

      expect(find('.alert.alert-box.alerts').text).to eq "Please select at least one contention."
      expect(find('td.form_with_errors').text).to eq "knee hurts"
      assert_selector('td.form_with_errors', :count => 1)
    end
  end


  context "Major System actions" do
    before do
      login_as(admin)
    end
    scenario "new page: " do
      visit "/major_systems/new"

      click_button "Save"

      assert_selector('#error_explanation li', :count => 2)
      assert_selector('.field_with_errors label', :count => 2)
      assert_selector('input.form_with_errors', :count => 2)
    end

    scenario "edit page: " do
      visit "/major_systems/#{major_system.id}/edit"

      fill_in "Label", with: ""
      click_button "Update Fields"
      
      assert_selector('#error_explanation li', :count => 1)
      expect(find('.field_with_errors label').text).to eq "Label"
      assert_selector('input.form_with_errors', :count => 1)
    end
  end

  context "Minor System actions" do
    before do
      login_as(admin)
    end
    scenario "new page: " do
      visit "/minor_systems/new"

      click_button "Save"

      assert_selector('#error_explanation li', :count => 2)
      assert_selector('.field_with_errors label', :count => 2)
      assert_selector('input.form_with_errors', :count => 2)
    end

    scenario "edit page: " do
      visit "/minor_systems/#{minor_system.id}/edit"

      fill_in "Name", with: ""
      click_button "Update Fields"
      
      assert_selector('#error_explanation li', :count => 1)
      expect(find('.field_with_errors label').text).to eq "Name"
      assert_selector('input.form_with_errors', :count => 1)
    end
  end

  context "Notifcation Log " do
    before do
      login_as(admin)
    end
    scenario "new page: " do
      visit "/notification_logs/new"

      click_button "Save"

      fail "No form present; Saves successfully"
    end

    scenario "edit page: " do
      visit "/notification_logs/#{notification_log.id}/edit"
      #no form present
      click_button "Save"
    end
  end

  context "Person pages " do
    before do
      login_as(admin)
    end
    scenario "new page: " do
      visit "/user/new"

      fill_in "First Name:", with: ""
      fill_in "Last Name:", with: "last name"
      fill_in "Password:", with: "password"

      click_button "Create User"

      expect(find('.alert.alert-box.alerts').text).to eq "- Email can't be blank- First name can't be blank"
      assert_selector('label.form_with_errors', count: 2)
      assert_selector('input.form_with_errors', count: 2)
    end

    scenario "index page: " do
      visit "/user"

      click_button "Select"

      expect(find('.alert.alert-box.alerts').text).to eq "Please select a site."
      assert_selector('label.form_with_errors', count: 1)
      expect(find('label.form_with_errors').text).to eq 'Select a site for the list of users'
      assert_selector('select.form_with_errors', count: 1)
    end
    
    scenario "show page: " do
      visit "/user/#{user.id}"

      lower_select_button = all('input.button')[1]
      lower_select_button.click

      expect(find('.alert.alert-box.alerts').text).to eq "Please select a site."
      assert_selector('label.form_with_errors', count: 1)
      expect(find('label.form_with_errors').text).to eq 'Choose a site to view and update roles'
      assert_selector('select.form_with_errors', count: 1)
    end
  end

  context "Contention rejection pages " do
    before do
      login_as(admin)
    end
    scenario "new : " do
      contention.update_attributes(:resolved_at => Time.now)
      visit new_claim_contention_rejection_path(claim, contention)

      click_button "Reject"

      expect(find('label h3.form_with_errors').text).to eq "Why are you rejecting this claim?"
      assert_selector('textarea.form_with_errors', :count => 1)
    end
  end

  context "Reports " do
    before do
      login_as(admin)
    end

    scenario "Exam Request by Date" do
      visit reports_exam_requests_by_date_path

      fill_in "exam_request_start_date", with: ""
      fill_in "exam_request_end_date", with: ""
      click_button "Search"

      list_of_labels = all('label.form_with_errors')
      expect(find('.alert.alert-box.alerts').text).to eq "Please enter a valid date range for Examination Requests"
      expect([list_of_labels[0].text,list_of_labels[1].text]).to include('Start date:', 'End date:')
      assert_selector('input.form_with_errors', :count => 2)
    end

    scenario "Insufficient Exam Report" do
      visit reports_insufficient_exam_report_path

      fill_in "start_date", with: ""
      fill_in "end_date", with: ""
      click_button "Get Report"

      list_of_labels = all('label.form_with_errors')
      expect(find('.alert.alert-box.alerts').text).to eq "Please provide a valid date range and select a report type (CSV or Plain text)."
      expect([list_of_labels[0].text,list_of_labels[1].text,list_of_labels[2].text,list_of_labels[3].text]).to include('Start date', 'End date', 'CSV', 'Text')
      assert_selector('input.form_with_errors', :count => 2)
    end

    #failing : no validation checks when form is empty
    scenario "Pending Exam" do
      visit reports_pending_exam_path

      fail "Pending Exam: no validation checks/feedback for user."
    end
  end

  context "Site pages" do
    before do
      login_as(admin)
    end

    scenario "New page" do
      visit new_site_path

      fill_in "name", with: "name 1"
      fill_in "address", with: ""
      fill_in "city", with: ""
      fill_in "state", with: "CA"
      fill_in "country", with: "USA"

      click_button 'Create Site'

      list_of_labels = all('label.form_with_errors')
      expect(find('.alert.alert-box.alerts').text).to match(/Address can\'t be blank.*City can\'t be blank/)
      expect([list_of_labels[0].text,list_of_labels[1].text]).to include('Address:', 'City:')
      assert_selector('input.form_with_errors', :count => 2)
    end
  end

  context "Symptoms pages" do
    before do
      login_as(admin)
    end

    scenario "New page" do
      visit new_symptom_path

      fill_in "symptom_name", with: "name 1"
      fill_in "symptom_label", with: ""

      click_button 'Save'

      expect(find('#error_explanation li').text).to match(/Label can\'t be blank/)
      expect(find('.field_with_errors label').text).to eq "Label"
      assert_selector('input.form_with_errors', :count => 1)
    end

    #failing spec: edit page rendered without @diagnoses_not_yet_added defined
    scenario "Edit page" do
      symptom = Symptom.create(name: "sym1", label: "label1")
      visit edit_symptom_path(symptom)

      fill_in "symptom_label", with: ""

      click_button 'Update Fields'

    end
  end

  context "Vbms r fact group pages" do
    before do
      login_as(admin)
    end

    scenario "New page" do
      visit new_vbms_r_fact_group_path

      click_button 'Save'

      fail "No validation checks: new group can be created with all input as blank"
    end

    scenario "Edit page" do
      vrfg = VbmsRFactGroup.create
      visit edit_vbms_r_fact_group_path(vrfg)

      click_button 'Save'

      fail "No validation checks: new group can be edited with all input as blank"
    end
  end


end
