# frozen_string_literal: true
#
# This rake task is for manually adding users until
# SSOi/SSOe are fully deployed at all sites.
#
# Input is read from spreadsheet (xls) expecting one user per row
# There are 7 columns per row expected as follows:
#
# Index 0 (Column A): First Name               (mandatory)
# Index 1 (Column B): Last Name                (mandatory)
# Index 2 (Column C): Email Address            (mandatory)
# Index 3 (Column D): Password                 (optional)
# Index 4 (Column E): Roles                    (optional)
# Index 5 (Column F): NPI for provider         (optional, non_vha role only)
# Index 6 (Column G): Send Confirmation Email  (optional, default to true)

require 'awesome_print'

namespace :admin do
  desc 'Add users to database from input file'

  task add_users: :environment do

    require 'spreadsheet'
    require 'fileutils'
    require 'logger'
    require 'active_support'
    require 'action_mailer'
    require 'set'

    ARGV.shift # remove task name from args

    # command line usage / feedback / error handling
    if ARGV.count < 1
      # added rake command directly; couldn't properly output current task
      abort "USAGE: rake admin:add_users <spreadsheet_file1> [<spreadsheet_file2> ...]"
    end
    ARGV.each do |filename_arg|
      unless File.exists? filename_arg
        abort "Spreadsheet file name #{filename_arg} not found."
      end
    end

    ROLES = %w[
                medical_assistant
                supervisor
                examiner
                administrator
                quality_assurance
                supervisory_clinician
                clinician
                triage
                scheduling
                site_user
                vha_cc
                non_vha
    ].freeze
    ROLES_SET = ROLES.to_set.freeze
    SITES_SET = Site.distinct.pluck(:name).to_set.freeze
    NPI_SET   = Provider.pluck(:npi).uniq.to_set.freeze

    CUI_LOGIN_EMAIL = ENV['APP_LOGIN_URL'] || 'https://deap-cui-tester-deapcuitester.c9users.io/sign_in'

    # constant names for spreadsheet columns within a row e.g. row[xxx_COL]
    FIRST_NAME_COL      = 0
    LAST_NAME_COL       = 1
    EMAIL_COL           = 2
    PASSWORD_COL        = 3
    ROLES_COL           = 4
    NPI_COL             = 5
    SEND_EMAIL_FLAG_COL = 6


    email_msg_func = -> (password_field) do
      return <<-END_EMAIL_MSG
        You are receiving this email because your account for the Clinician User Interface is now ready!

        Please login at: #{CUI_LOGIN_EMAIL}

        Login: your email address
        Password: #{password_field}
      END_EMAIL_MSG
    end

    # test input file(s) existence, readability before processing
    total_added = 0

    #setup logger
    log = ActiveSupport::Logger.new("log/admin-add-users.log")
    start_time = Time.now.strftime('%H:%M:%S')
    log.info "--Starting @ #{start_time}"

    
    ARGV.each do |filename_arg|
      book    = Spreadsheet.open filename_arg
      sheet0  = book.worksheet 0
      current_sheet_count = 0

      sheet0.each_with_index do |row, row_indx|
        # Index 0 (Column A): First Name               (mandatory)
        # Index 1 (Column B): Last Name                (mandatory)
        # Index 2 (Column C): Email Address            (mandatory)
        # Index 3 (Column D): Password                 (optional)
        # Index 4 (Column E): Roles                    (optional)
        # Index 5 (Column F): NPI for provider         (optional, non_vha role only)
        # Index 6 (Column G): Send Confirmation Email  (optional, default to true)

        # throw out first non-empty row, which should contain column labels
        next if row_indx == 0

        puts "Processing #{row[ROLES_COL]} - #{row[EMAIL_COL]} ..."
        ap row

        if [FIRST_NAME_COL, LAST_NAME_COL, EMAIL_COL].any? {|col_indx| row[col_indx].blank? }
          message = "----Error in row ##{row_indx + 1}: missing mandatory field (first name, last name, email) in columns A thru C"
          puts message
          log.info message
          next
        end


        # check if email address already exists
        if User.exists?(email: row[EMAIL_COL])
          log.info "----User with email #{row[EMAIL_COL]} (row ##{row_indx + 1}, col C in spreadsheet) already exists."
          next
        end

        # check for presence of roles and inclusion in list of valid roles.
        # Roles are comma-separated, with whitespace after commas being discarded.
        if row[ROLES_COL].present?
          role_tokens = row[ROLES_COL].split(/\,\s*/).map(&:downcase).to_set
          unless role_tokens.subset? ROLES_SET
            log.info "----Error in row ##{row_indx + 1}: unknown role in column E"
            log.info "\t\t\tAllowed roles: #{ROLES}"
            next
          end
        end

        # check for presence of NPI and inclusion in list of valid sites
        if row[NPI_COL].present?
          row[NPI_COL] = row[NPI_COL].to_s.strip.gsub(/\.0$/, '')
          unless role_tokens.include? "non_vha"
            log.info "----Error in row ##{row_indx + 1}: only non_vha users can have an assigned NPI."
            next
          end
        end


        # IMPORTANT: All passwords must be at least 12 chars long
        new_password = "Cui!#{Array.new(8){rand(36).to_s(36)}.join}"



        user = User.create(
          first_name:           row[FIRST_NAME_COL],
          last_name:            row[LAST_NAME_COL],
          email:                row[EMAIL_COL].downcase,
          password:             new_password,
          roles:                (role_tokens || []).to_a,
          authorization_state:  'authorized',
          user_preference:      UserPreference.new,
          npi:                  row[NPI_COL]
        )

        begin
          user.save!
          log.info "Account created for #{user.email}."
        rescue Exception => e
          log.info "-----ERROR: for Row #{row_indx} - #{row[EMAIL_COL]} - #{e}"
        end

        # checking for empty cells (which Spreadsheet explicitly maps to nil) or
        # case-insensitive truthy values, and in either case (empty or truthy) we send an email.
        email_flag = row[SEND_EMAIL_FLAG_COL]
        if (email_flag.nil? || ['TRUE', 'YES', 'Y'].include?(email_flag.to_s.upcase))
          UserMailer.welcome_email(row[EMAIL_COL], email_msg_func.call(new_password)).deliver_now
          log.info "----Email sent for above account " + row[EMAIL_COL]
          log.info "------Password: #{new_password}"
        end

        current_sheet_count += 1
      end
      log.info "----Added #{current_sheet_count} users from #{filename_arg}"
      total_added += current_sheet_count
    end

    log.info "---Total Users added: #{total_added}"
    log.info "--Stopping @ #{Time.now.strftime('%H:%M:%S')}\n"
  end

  task generate_users: :environment do

    ARGV.each {|a| task a.to_sym do ; end }
    count = ARGV[1]
    site = ARGV[2]
    roles  = ARGV[3]

    if count == nil or count == "" or site == nil or roles == nil
      puts "Please make sure you are invoking the command as specified below:"
      puts ""
      puts "rake admin:generate_users <number of users> <site> <roles>"
      puts ""
    else
      DEFAULT_AUTHORIZATION_STATE = "authorized".freeze
      ROLES = ["medical_assistant",
             "supervisor",
             "examiner",
             "administrator",
             "quality_assurance",
             "supervisory_clinician",
             "clinician",
             "triage",
             "scheduling",
             "site_user",
             "app_admin",
             "site_admin"]
      site_from_db = Site.find_by_name(site)

      role_tokens = roles.split(",")

      original_role_tokens = role_tokens.clone
      role_tokens.reject! { |n| !ROLES.include?  n.lstrip.rstrip }

      if (original_role_tokens.length < 1 or original_role_tokens.length != role_tokens.length)
        puts "Invalid Roles"
        puts "Please select from the list ::: " + ROLES.to_s
      elsif (site_from_db == nil)
        puts "Invalid site ::: " + site
        puts "Please select one from the list ::: [QTC, VES, VetFed, Atlanta VA Medical Center, Chicago Medical Center]"
      else
        $i = 1
        $num = count.to_i

        File.open('lib/tasks/admin/users.txt', 'w') do |file|
         while $i <= $num  do

          first_name = "FName" + (SecureRandom.random_number(1000)).to_s
          last_name = "LName" + (SecureRandom.random_number(1000)).to_s
          email = "test" + (SecureRandom.random_number(100000)).to_s + "@gmail.com"
          password = "CUI-#{Array.new(6){rand(36).to_s(36)}.join}"

          u = User.find_by_email(email)
          if u.present?
            puts email + " already exists!!!"
          else
             new_user = User.create(
                     :first_name           => first_name,
                     :last_name            => last_name,
                     :email                => email,
                     :password             => password,
                     :roles                => role_tokens,
                     :authorization_state  => DEFAULT_AUTHORIZATION_STATE
                   )

             new_user.add_site(site_from_db)

             site_roles = new_user.get_site_roles(site_from_db)

             site_roles.triage = false
             if (role_tokens.include? 'triage')
               site_roles.triage = true
             end
             site_roles.admin = false
             if (role_tokens.include? 'admin')
               site_roles.admin = true
             end
             site_roles.scheduling = false
             if (role_tokens.include? 'scheduling')
               site_roles.scheduling = true
             end
             site_roles.clinician = false
             if (role_tokens.include? 'clinician')
               site_roles.clinician = true
             end
             site_roles.super_clinician = false
             if (role_tokens.include? 'supervisory_clinician')
               site_roles.super_clinician = true
             end
             site_roles.qa = false
             if (role_tokens.include? 'quality_assurance')
              site_roles.qa = true
             end
             site_roles.save

            puts("Generated and Created...#$i" )

            file.puts first_name + "  "  + last_name + "  " + email + "  " + password

          end
          $i +=1
         end
        end
        puts "**********************************"
        puts "The following file has been generated:"
        puts "Location ::: /lib/tasks/admin"
        puts "File     ::: users.txt "
        puts "**********************************"
      end
    end
  end












end
