# lib/ntm/training_material.rb
# NOTE: supports the /news route
#       also known as the news and training material website (NTM)
#       intent to be dynamic always presenting the user with the
#       most uptodate information
#
# NOTE: Training material files are stored in pairs with the same basenamee
#       and different file extensions.  The material is presented to the user
#       alphabetic sequence according to the basename.  Recommend using a
#       naming convention like "NNNNN_stuff.[txt|*]" where NNNNN is a numeric
#       sequence number.  Consider this example pair:
#
#         00010_create_referral.txt
#         00010_create_referral.mov
#
#       Notice how the name only differs in the extension.  The *.txt file
#       a text description of the training resource (aka material) - in this
#       case it is a video.  Note the ".mov" extension.  The actual training
#       material can be of ANY filetype EXCEPT "*.txt" because that would
#       conflict with the file used to store the description.

# Used in development and testing
require_relative 'ntm_storage/filesystem_access.rb'

# For production and UAT-level testing
require_relative 'ntm_storage/s3_access.rb'

class TrainingMaterial

  # Used to select which files to process as News articles.
  # NOTE: file_path is a Pathname object.
  FILE_SELECTER = Proc.new do |file_path|
    (!file_path.directory?) &&
    ('.txt' != file_path.extname.downcase) # Can not be a '*.txt' file
  end
  def initialize(user_group)
    @user_group   = user_group

    if NTM_S3_ENABLED
      # Access training materials from AWS S3 (Simple Storage System)
      root_path = Pathname.new('/s3')
      storage_system = NtmStorage::S3_Access
    else
      # Access training materials from local filesystem
      root_path = Rails.root + 'public'
      storage_system = NtmStorage::FileSystem_Access
    end

    base_path = root_path + ENV['NTM_ROOT_DIR'] + ENV['NTM_TM_DIR']

    @storage = storage_system.new(
                @user_group, 
                base_path, 
                TrainingMaterial::FILE_SELECTER
              )
   
  end # def initialize


  def latest
    @storage.latest
  end
  alias :first  :latest
  alias :newest :latest

  def oldest
    @storage.oldest
  end
  alias :last :oldest

  def each_pair(&block)
    @storage.each_pair do |file_path, file_content|
      block.call(file_path, file_content)
    end
  end
  alias :each :each_pair

  def keys
    @storage.keys
  end

  def [](key)
    @storage[key]
  end


end # class TrainingMaterial