# lib/ntm/storage/filesystem.rb

module NtmStorage
  # This class only does read-only access hence the "_Access" in its name
  class FileSystem_Access

    def initialize(user_group, base_path, selecter)
      @user_group = user_group
      @base_path  = base_path

      build_the_paths
      load_the_files(selecter)
    end

    # Establish the paths to the storage locations
    def build_the_paths
      @group_path = @base_path + ENV["NTM_#{@user_group}_DIR"]
    end

    # Establishes the structures to access the filenames (aka locations)
    # but does not read the actual content of the files.
    def load_the_files(selecter)
      paths  =  @base_path.children.select  {|c| selecter.call(c) }
      paths += @group_path.children.select  {|c| selecter.call(c) }
      @paths = Hash.new
      paths.each {|c| @paths[c.basename.to_s] = c}
      @filenames = @paths.keys.sort
    end

    # reverse the access keys
    def reverse_keys!
      @filenames.reverse!
    end

    # returns the date and content of the latest news article
    def latest
      key       = @filenames.first
      return path_and_content(key)
    end
    alias :first  :latest
    alias :newest :latest

    # returns the oldest news article's date and content
    def oldest
      key = @filenames.last
      return path_and_content(key)
    end
    alias :last :oldest

    # returns each news article's data and content in the order
    # newest to oldest
    def each_pair( &block )
      @filenames.each do |key|
        block.call( path_and_content(key) )
      end
    end

    # reutns an array of strings used as the keys to obtain the news articles
    def keys
      @filenames
    end

    # returns the news article content for a specific key as a string
    def [](key)
      retrieve_content(key)
    end

    private

    # returns the path and content for a specific file key
    # as an array where the path object is the first element and the content
    # is the next 
    def path_and_content(key)
      return @paths[key], retrieve_content(key)
    end

    # content by definition is always in a *.txt file.
    # keys that do not end with '.txt' are used by TrainingMateral class
    # to seperate the description (in the *.txt file - the content) with
    # the actual training material file path in the @paths object
    def retrieve_content(key)
      # TODO: cache content in Redis with a TTL of 60 minutes
      key_parts     = key.split('.')
      extension     = key_parts.last.downcase
      content_path  = @paths[key]

      unless 'txt' == extension
        key_parts[key_parts.size-1] = 'txt' # replace the file extension
        new_key = key_parts.join('.')
        content_path = content_path.parent + new_key
      end

      content_path.read
    end

  end # class FileSystem_Access
end # module NtmStorage