# lib/audit_trail.rb

module AuditTrail
  def audit_trail
    @audit_hash = Hash.new
    return(@audit_hash) if versions.empty?
    build_audit_hash
  end


  # The Consultation model may make use of the same kind of structure that
  # ReferralNote uses.  It might be called ConsultationNote or maybe we might
  # turn ReferralNote into a polymorphic model called UserNote and use it for
  # both Referral and Consultation.  Regardless of which way we go we are using
  # a duck typing concept to give both Referral and Consultation a "user_notes" method
  # that encapsulates their sub-table access.
  def merge_audit_trail_with_notes
    @audit_hash ||= audit_trail

    notes_hash = Hash.new

    # NOTE: that 'user' and 'notes' are both fields in the 'content'
    #       attribute of the ReferralNote model.  'user' is a string pbject.
    #       It is a name of a user and not a User.id
    user_notes.each do | note |
      key             = note.created_at
      changed_by      = note.user
      changed_because = 'create'
      notes_hash[key] = { 'changed_by'      => changed_by,
                          'changed_because' => changed_because,
                          'fields_changed'  => {
                            'notes' => [nil, note.notes]
                          }
                        }
    end
    @audit_hash.merge! notes_hash
  end


  def humanize_the_id_field_values(table_to_field_xref: {
                'referral_type_id'    => :title,
                'referral_reason_id'  => :title,
                'referral_status_id'  => :name
              })
    @audit_hash ||= audit_trail

    table_ids     = table_to_field_xref.keys
    klass_array   = table_ids.map{|tid| tid.gsub('_id','').camelize.constantize}

    @audit_hash.each_pair do |key, value|
      value['fields_changed'].each_pair do | field_name, field_values|
        if klass_index = table_ids.index(field_name) # NOTE: Yes! assignment in condition is intended.
          old_value = nil
          new_value = nil
          old_id    = field_values.shift
          new_id    = field_values.shift
          old_value = klass_array[klass_index].find(old_id).send(table_to_field_xref[field_name]) unless old_id.nil?
          new_value = klass_array[klass_index].find(new_id).send(table_to_field_xref[field_name]) unless new_id.nil?
          @audit_hash[key]['fields_changed'][field_name] = [old_value, new_value]
        end
      end
    end

    return @audit_hash
  end

  private
  ##################################################
  ## Support for #audit_trail

  def build_audit_hash
    versions.each do |version|
      insert_new_audit_entry(version)
    end
    return @audit_hash
  end


  def insert_new_audit_entry(version)
    changed_by      = version.whodunnit.nil? ? "unknown" : User.find(version.whodunnit).name
    changed_because = version.event
    key             = version.created_at
    @audit_hash[key] = Hash.new

    @audit_hash[key] =  {   'changed_by'      => changed_by,
                            'changed_because' => changed_because,
                            'fields_changed'  => {}
                        }

    version.changeset.each_pair do |column_name, column_values|
      if 'content' == column_name
        @audit_hash[key]['fields_changed'].merge! extract_changes_from_content(column_values)
      else
        @audit_hash[key]['fields_changed'][column_name] = column_values
      end
    end
  end


  def extract_changes_from_content(content_field_values)
    content_hash  = Hash.new
    change_array  = HashDiff.diff content_field_values.first, content_field_values.last
    change_array.each do |change|
      change_reason     = change.shift # NOTE: one of "~-+"
      change_field_name = change.shift
      content_hash[change_field_name] = case change_reason
        when '~' # update
          change                # NOTE: [old_value, new_value]
        when '-' # destroy
          [change.shift, nil]
        when '+' # create
          [nil, change.shift]
      end
    end
    return content_hash
  end

  ##################################################
  ## Support for #audit_trail

end # module AuditTrail