# System Environment Variables

## Environment Variables NOT set in .env* files

There are three environment variables which need to be set within the terminal/console/shell environment and not in a `dotenv` managed `.env*` file.  RAILS_ENV, RACK_ENV and DEPLOY_ENV shall be set outside of any `.env*` file.

### RAILS_ENV

The Rails library makes extensive use of the RAILS_ENV variable.  Within the Rails application the value of the variable can be access using `Rails.env` in addition to the common Ruby `ENV['RAILS_ENV']` means.  Also a Rails application can conditionally access the value using query methods.  For example `Rails.env.development?` returns true when `'development' == ENV['RAILS_ENV']` which is in a word wonderful.

| RAILS_ENV Value | Meaning |
|-------------------|----------------------|
| development | Adds tools and configurations for use by a software developer |
| production  | Minimize extranious tools, utilities and configurations to that specific to the application |
| test        | Something between development and production |

The full database name includes a base prefix component (ENV['DBNAME']) and the operational environment value (ENV['RAILS_ENV']) allowing the same database management system to host development, test and production databases.

The default value for RAILS_ENV is development.


### RACK_ENV

Typically the value of the RACK_ENV and RAILS_ENV are the same.  They do not have to be.  RACK_ENV is used by any Rack-based web server to establish the environment (set of middleware) configured as part of the overall web server.  Common values for RACK_ENV are:

| RACK_ENV Value | Default Rack middleware Loaded |
|-------------------|----------------------|
| development | CommonLogger, ShowExceptions, and Lint |
| deployment  | CommonLogger |
| none        | no extra middleware added |

The default value for RACK_ENV is development.


### DEPLOY_TYPE

The value of the DEPLOY_TYPE variable specifies the kind (perhaps even hostname) of the deployed application.

| DEPLOY_TYPE Value | Description |
|-------------------|----------------------|
| dev         | GovCloud deployment for a development environment |
| localhost   | local workstation deployment; typically used only fpr developer workstations |
| prep        | GovCloud deployment of a staging environment |
| prod        | GovCloud deployment of a production environment |
| test        | GovCloud deployment of a test environment |

The default value for DEPLOY_TYPE is localhost.


## .env* Files

The [dotenv gem](https://github.com/bkeepers/dotenv/blob/master/README.md) manages the values of system environment variables (SEV) in this application.  This file documents each SEV used by this application in terms of its use and value types.

The files that contain the SEVs values are shown in the following sections.

> *IMPORTANT:  DO NOT* set the values of RACK_ENV, RAILS_ENV and DEPLOY_TYPE within a `.env*` file.  Doing so *may* result in unanticipated consequences.  Always set these three variables within your terminal/console window as necessary to accomplish your task.


## Convention w/r/t `dotenv`

The `dotenv` gem does not require the use of the 'export' prefix in front of a VARIABLE=value statement.  However, since the `.env*` files are used by Ruby programs and sourced by `bash` shell scripts, this system will *ALWAYS USE* the 'export' prefix on a variable assignment.

All system environment variables shall be documented in the `doc/system_environment_variables.md` file.


## Base Defaults

| File Name | Description |
|-------------------|----------------------|
| .env              | The base-level file containing default values |

The `.env` file contains the default values for every system environment variable that requires a value.  A value for every environment variable used in this system shall be defined in at least one of the `.env*` files.


## RAILS_ENV-based Files

The system environment variable *RAILS_ENV* is used to direct the loading of variables whose values are are different from the default values specified in the `.env` base file.

| File Name | Description |
|-------------------|----------------------|
| .env.development  | Loaded when RAILS_ENV is 'development' to over-ride .env |
| .env.production   | Loaded when RAILS_ENV is 'production' to over-ride .env |
| .env.test         | Loaded when RAILS_ENV is 'test' to over-ride .env |


## DEPLOY_TYPE-based Files

The *DEPLOY_TYPE* system environment variable is used to specify/select a set of system environment variable values that are used by a specific deployment host machine.  For example any variable that is used to specify an application URL.  The URL's value will depend upon which host machine to which the application is deployed.

| File Name | Description |
|-------------------|----------------------|
| .env.sqa          | Used for deployment to an SQA server |
| .env.uat          | Used for deployment to a UAT server |
| .env.deploy       | Used for a localhost docker deployment on a developer's workstation |

The `.env.deploy` file is special.  It is used by developer's who chose to deploy the application to their workstation (localhost) using docker containers.


## Local Over-rides

Any system environment variable can be over-ridden by a "*.local" file.  These files are intended for use on a developer's workstation or on specific host machines that support the application.  The purpose of the local over-ride is to provide values that are required by the installation *BUT ARE NOT* checked into the source code management repositories.


| File Name | Description |
|-------------------|----------------------|
| .env.local        | (Not checked in to the repository) Used by developers to over-ride values on their local workstations |


### RAILS_ENV - based local over-rides

| File Name | Description |
|-------------------|----------------------|
| .env.development.local  | Loaded when RAILS_ENV is 'development' to over-ride .env |
| .env.production.local   | Loaded when RAILS_ENV is 'production' to over-ride .env |
| .env.test.local         | Loaded when RAILS_ENV is 'test' to over-ride .env |


### DEPLOY_TYPE - based local over-rides

| File Name | Description |
|-------------------|----------------------|
| .env.sqa.local          | Used for deployment to an SQA server |
| .env.uat.local          | Used for deployment to a UAT server |
| .env.deploy.local       | Used for a localhost docker deployment on a developer's workstation |


## Loading Conventions

The `.env*` files are processed (loaded) in specific orders depending on whether they are being used within the RAILS application or within a BASH shell script.


### RAILS .env* File Loading Order

According to the documentation the [dotenv gem](https://github.com/bkeepers/dotenv/blob/master/README.md) takes the *first* value set for a veriable as its "official" value.  So within the rails application for a RAILS_ENV == 'development' and DEPLOY_TYPE unset, the files are loaded in this order:

* .env.development.local
* .env.local
* .env.development
* .env


#### Existing Variable Values in the RAILS Application

In the RAILS application the `dotenv` gem loads the variables, assigns and maintains their value based upon the first value set.  This is contrary to normal thinking.  Consider this `.env` file:

    export VARIABLE=1
    export VARIABLE=2
    export VARIABLE=3

In the RAILS application the value of ENV['VARIABLE'] will be 1 not 3.

However if the RAILS application were launched like this using the same `.env` file:

    VARIABLE=4 rails server

The value of ENV['VARIABLE'] inside the application will be 4 because it was the first value set.


### BASH .env* File Loading Order

When using the files from a shell script (bash for example) the value of a variable is the *last* value that was set.  In this case within a shell script for RAILS_ENV == 'development' and DEPLOY_TYPE unset the files need to be *sourced* in this order:

* .env
* .env.development
* .env.local
* .env.development.local

Source the file `bin/setup_environment` within a BASH shell script to have the application's system environment variables set for the proper environment/host as specified by the values of RAILS_ENV and DEPLOY_TYPE.  The script `bin/setup_environment` takes care of sourcing the `.env*` files in the proper order.


#### Existing Variable Values in a BASH Script

In a `bash` script the variable values are assigned based upon the *last* value set.  This is the normal way of thinking about variables.  Consider this `.env` file:

    export VARIABLE=1
    export VARIABLE=2
    export VARIABLE=3

In a `bash` script the value of ENV['VARIABLE'] will be 3 not 1.

If the `bash` script were launched like this using the same `.env` file:

    VARIABLE=4 my_bash_script_using_env_files

The value of ENV['VARIABLE'] inside the script will still be 3 because it was the last value set.

There is an circumstance in which this will make a significant difference.  The same conditions with the script being launched like this:

    RAILS_ENV=production DEPLOY_TYPE=sqa my_bash_script_using_env_files

By sourcing the `bin/setup_environment` file within `my_bash_script_using_env_files` the initial values of RAILS_ENV and DEPLOY_TYPE are used to establish which `.env*` files are to be processed.  If one of those files changes the value of either RAILS_ENV or DEPLOY_TYPE then there is a possibility that things will get messed up.  So the moral of the story is *do not set the values of RAILS_ENV and DEPLOY_TYPE within a `.env*` file.*


## Special Values used by `dotenv` and `bash`

The `dotenv` gem supports the use of two special values that are consistent with `bash` capabilities.


### Assigning a Value Based Upon Another Variable

The `odtebv` gem allows access to existing values of a system environment variable (VARIABLE) within a `.env*` file using '$VARIABLE' which is consistent with the `bash` shell.  Consider this snippit from a `.env` file:

    export DOCKER_HOST_WEB_PORT=$PUMA_PORT

The value of the DOCKER_HOST_WEB_PORT will be set to the value of PUMA_PORT at the time that the assignment to DOCKER_HOST_WEB_PORT is made.  If the value of PUMA_PORT is changed sometime after the assignment to DOCKER_HOST_WEB_PORT that future assignment *WILL NOT* change the previously set value of DOCKER_HOST_WEB_PORT.

> IMPORTANT: The `bash` convention of ${VARIABLE} is NOT supported by the `dotenv` gem.


### Assigning a Value Based Upon the Results of a Shell Command

There may be times when it is necessary to access a shell command to obtain a value for a variable within a `.env*` file.  Both `dotenv` and bash follow the same syntax using '$(some shell command)' to obtain the value (STDOUT) of a shell command.  For example consider a `.env*` file that has this entry:

    export VARIABLE=$(hostname)

In both the RAILS and BASH systems the shell command "hostname" will be executed and its return value (STDOUT) will be used as the value for assignment to VARIABLE.

The shell command can be more complex than a single command.  Consider this example from a developer's `.env.local` file:

    export APP_VERSION=$(git describe --abbrev=0 --tags)
    export GIT_BRANCH=$(git branch | fgrep '*' | cut -d' ' -f 2)
    export MY_VERSION=$USER@$(hostname) [$RAILS_ENV] $GIT_BRANCH/$(git rev-parse --short HEAD) CM: $APP_VERSION

With those two lines in a developer's `.env.local` file a `RAILS_ENV=development rails console` session will show something like:

```
Loading development environment (Rails 4.2.3)
[1] pry(main)> ENV['APP_VERSION']
=> "CUI-2.2.0.EOS10-Int"
[2] pry(main)> ENV['GIT_BRANCH']
=> "PP-558"
[3] pry(main)> ENV['MY_VERSION']
=> "dvanhoozer@42Six-00485.local [development] PP-558/90c02e53 CM: CUI-2.2.0.EOS10-Int"
[4] pry(main)>
```


## Naming Conventions

System environment variables need to be named in such a way where their application and function is obvious upon inspection of their name.  System environment variables which are used together within the same feature should all begin with the same phrase (namespaced) such as is done with the EMAIL and FEEDBACK features.


## Variable Name (Sorted)

[A](#a) . [B](#b) . [C](#c) . [D](#d) . [E](#e) . [F](#f) . [G](#g) . [H](#h) . [I](#i) . [J](#j) . [K](#k) . [L](#l) . [M](#m) . [N](#n) . [O](#o) . [P](#p) . [Q](#q) . [R](#r) . [S](#s) . [T](#t) . [U](#u) . [V](#v) . [W](#w) . [X](#x) . [Y](#y) . [Z](#z)


### A
| Variable Name     | Description |
|-------------------|----------------------|
| APP_HOST | Set the HTTP header to localhost to resolve WASA report finding |
| ACTIVE_JOB_QUEUE_PREFIX | A string to identify the background jobs submitted by this application |
| ACTIVE_JOB_URL | The data store where background jobs are submitted |
| ALWAYS_MINIFY_ASSETS | If true, JS/CSS assets get auto-minified regardless of Rails env |
| APP_LOGIN_URL | The fully qualified URL for the login page for this application |
| APP_URL | The fully qualified base URL for this application.  APP_LOGIN_URL should start with the value of this variable |
| APP_VERSION | Used by the CM to set the application version for display within the web pages |

### B
| Variable Name     | Description |
|-------------------|----------------------|
| BUNDLE_GEMFILE | The file name that the `bundle` command uses to retrieve the gem names for inclusion into the application.  Default is "Gemfile" |

### C
| Variable Name            | Description |
|--------------------------|----------------------|
| CUI_DISABLE_UNDER_REVIEW | |
| CUI_SUPPORT_INBOX        | Email address that support requests are sent to |
| CLAIMS_API_URL           | The url portion that claims are posted to |

### D
| Variable Name     | Description |
|-------------------|----------------------|
| db - database feature | This applications makes use of the standard Rails object relational mapping (ORM) system ActiveRecord.  Access to the relational database management system (RDBMS) is configured via the file `config/database.yml` using variables whose names start with DB. |
| DBADAPTER | The Rails ORM (ActiveRecord) database adapter to use for accessing the DBMS.  Example: 'pg' |
| DBHOST | The host name or IP address of the DBMS server.  Can also be a socket reference. |
| DBMIGRATE | true/false invokes rake db:migrate task on application start |
| DBNAME | The base name of the database within the DBMS server used.  The RAILS_ENV is automatically used as a suffix in the `config/database.yml` file |
| DBPASS | The password of the `DBUSER` for accessing the database |
| DBPORT | The TCP port to use when contacting the DBMS server via TCP. |
| DBRESET | true/false invokes rake db:resettask on application start |
| DBSEED | true/false invokes rake db:seed task on application start |
| DBUSER | The user account to use when contacting the DBMS server |
| DEPLOY_TYPE | Not set in any `.env*` files.  Used to specify which `.env*` is used for a specific type of deployment.  (eg. sqa, uat, docker) |
| DEVELOPER_TESTING | true/false Adds "Developer Testing" banner to top of every page |
| DISPLAY_FACILITY_DROPDOWNS | true/false Determines whether Facility dropdowns are visible on Consultation and Referral filter options panels |
| DOCKER_HOST_WEB_PORT | Used with the `web_docker-compose.yml` file to specify the TCP port to be exposed to the host machine from inside the docker image. |

The `rake prep:db` task used by the `rake app:start` task makes use of the DBMIGRATE, DBRESET and DBSEED variables to control its execution of the `db:migrate, db:seed and db:reset` tasks.  See the file `lib/tasks/prep.rake` for details.

* `db:reset` destroys all data in a database
* `db:migrate` chances the schema of a database.  It can also change the content of the database as well based upon the new migrations that are being applied.
* `db:seed` task modifies the content of a database.

This integration of system environment variables with rake tasks was specifically implemented to support the deployment of the docker containers.

### E
| Variable Name     | Description |
|-------------------|----------------------|
| email feature | This application sends (never receives) eMail messages to users for various purposes.  The outgoing eMail is configured using the variables with names beginning EMAIL_*  eMail can also be sent to notify developers/maintainers of user feedback messages.  The configuration of the eMail services supporting the feedback feature is set by variables with names beginning FEEDBACK_EMAIL_* which parrallels the variables defined as EMAIL_*  This was done to provide for the fexlibility of having two eMail routes out of the application.  Typically the FEEDBACK_EMAIL_* variables are set to their $EMAIL_* counterparts. |
| EMAIL_DOMAIN | FIXME: undocumented |
| EMAIL_HOST | This is the host name or IP address of the eMail server |
| EMAIL_PASS | This is the password for the $EMAIL_USER account on the eMail server |
| EMAIL_PORT | This is the TCP port to use when accessing the eMail server |
| EMAIL_SANITIZE | FIXME: undocumented |
| EMAIL_USER | This is the user account for accessing the eMail server |
| ENABLE_FEEDBACK | true/false enables the user feedback feature which is based upon the [pointless_feedback gem](https://github.com/vigetlabs/pointless-feedback/blob/master/README.md)|
| ENABLE_FEEDBACK_EMAIL | true/false enables notification of user feedback via eMail |
| ENABLE_FEEDBACK_HIPCHAT | true/false enables notification of user feedback via hipchat |
| ENABLE_FEEDBACK_SLACK | true/false enables notification of user feedback via slack |
| ENABLE_OPENSSL_INITIALIZER | true/false enables initialization of openSSL |
| ENABLE_PHASE_2 | FIXME: too generic; undocumented |

### F
| Variable Name     | Description |
|-------------------|----------------------|
| feedback feature | The feedback feature allows application users to send messages to the developers and maintainers of the application.  This feature is controlled by ENABLE_FEEDBACK.  It has the ability to provide notifications to the developers/maintainers via three different routes: eMail, hipchat and slack.  The notifications routes are controled by ENABLE_FEEDBACK_EMAIL, ENABLE_FEEDBACK_HIPCHAT and ENABLE_FEEDBACK_SLACK |
| FEEDBACK_EMAIL_DOMAIN | Typically set to $EMAIL_DOMAIN |
| FEEDBACK_EMAIL_HOST | Typically set to $EMAIL_HOST |
| FEEDBACK_EMAIL_PASS | Typically set to $EMAIL_PASS |
| FEEDBACK_EMAIL_PORT | Typically set to $EMAIL_PORT |
| FEEDBACK_EMAIL_SANITIZE | true/false Used to scrub HTML from eMail messages. |
| FEEDBACK_EMAIL_SUBJECT | The prefix phrase to use on an eMail subject line for eMail notifications of user feedback.  The topic of the feedback message is appeded to this value to build the complete subject line|
| FEEDBACK_EMAIL_TO_CUI | This is the eMail address to which user feedback notification messages are sent for CUI. |
| FEEDBACK_EMAIL_TO_CPP | This is the eMail address to which user feedback notification messages are sent for CPP. |
| FEEDBACK_EMAIL_USER | The user account name of the eMail server used to send feedback message notifications.  Typically set to $EMAIL_UESR |
| FEEDBACK_HIPCHAT_ROOM | The hipchat room name to which feedback message notifications are sent |
| FEEDBACK_HIPCHAT_SERVER | The host name or IP address of the hipchat server to receive user feedback message notifications |
| FEEDBACK_HIPCHAT_TOKEN | The hipchat special access authentication token used to grant outside access to a hipchat server |
| FEEDBACK_IM_GATEWAY | The distributed Ruby URL of the instant messaging gateway used to provided feedback message notifications to both hipchat and slack. |
| FEEDBACK_SLACK_CHANNEL | The channel (e.g. room) into which user feedback message notifications are posted on the slack server. |
| FEEDBACK_SLACK_USERNAME | The user name of the account authorized to post feedback notification messages to the slack server |
| FEEDBACK_SLACK_WEBHOOK_URL | This is the URL used by the IM Gateway for sending user feedback notification messages to a slack server |

### G
### H
| Variable Name     | Description |
|-------------------|----------------------|
| HOME | Intrinsic variable of the OS (not set in any `.env*` file) Used in the `rake db:backup and db:restore` tasks. The root directory of the user within the workstation/host computer. |


### I
| Variable Name     | Description |
|-------------------|----------------------|

### J
### K
### L
| Variable Name     | Description |
|-------------------|----------------------|
| LOG_LEVEL | The level of Rails logging to use in the application; values are DEBUG, INFO, WARNING, ERROR |

### M
| Variable Name     | Description |
|-------------------|----------------------|
| MVI_API_URL       | This is the endpoint for the VA Master Veteran Index web-service to retrieve veteran/patient identity info

### N
| Variable Name     | Description |
|-------------------|----------------------|
| NTM_ENABLED    | A truthy thing (true or false) to enable or disable the /news route |
| NTM_S3_ENABLED | A truthy thing to enable or disable access to S3 for data |
| NTM_S3_TOKEN   | The security token used to access S3 |
| NTM_S3_BUCKET | The S3 buckect containing NTM data |
| NTM_ROOT_DIR  | The sub-directory within the S3 bucket for NTM data |
| NTM_NEWS_DIR  | The sub-directory within the NTM_ROOT_DIR for news articles |
| NTM_TM_DIR    | The sub-directory within the NTM_DOOR_DIR for training material |
| NTM_CUI_DIR   | The sub-directory under news and tm directories that is specific to CUI users |
| NTM_CPP_DIR   | The sub-directory under news and tm directories that is specific to CPP users |

### O
### P
| Variable Name     | Description |
|-------------------|----------------------|
| PHASE_2_BODY_SYSTEMS | FIXME: too generic; undocumented |
| POSTGRES_DB | Used within the `db_docker-compose.yml` file to specify the name of the database to use on the DBMS server.  Must be set to the value of $DBNAME |
| POSTGRES_PASSWORD | Used within the `db_docker-compose.yml` file to specify the user account password.  Must be set to the value of $DBPASS |
| POSTGRES_USER | Used within the `db_docker-compose.yml` file to specify the user account fpr accessomg tje DB<S server.  Must be set to the value of $DBNAME |
| puma feature | The [puma gem](https://github.com/puma/puma/blob/master/README.md) implements the web server selected by the application developers for its speed and stability. It is configured using environment variables with names that start PUMA_* Puma is the most widely used web server for Rails applications on the internet.  Its popularity and its stability have resulted in the Rails team choosing Puma as their standard web server beginning with Rails version 5.1 |
| PUMA_BIND | Over-rides the values of PUMA_PROTOCOL, PUMA_HOST and PUMA_PORT as we;; as PUMA_SSL_CERT_PATH and PUMA_SSL_KEY_PATH when the protocol is "ssl://" - PUMA_BIND is the golden source for all its components.  Use Ruby's URI modeul to extract its components if they are needed individually - See `config/puma.rb` for details |
| PUMA_DAEMONIZE | true/false Set to false for all web containers; Set to true for all other web servers |
| PUMA_HOST | The host name or IP address of this web server to which the Puma process should bind.  Typically the value is 0.0.0.0 meaning all IP address and names of this server |
| PUMA_MAX_THREADS | The maximum number of threads to run per Puma worker.  Typically 4 |
| PUMA_MIN_THREADS | The minimum number of threads to run per Puma worker.  Typically 1 |
| PUMA_PORT | The TCP port through which the Puma application server is reached. |
| PUMA_PROTOCOL | The puma web server can be reached by various network connetions.  The most typical is TCP.  The value for this variable is "tcp://" in normal configurations |
| PUMA_REDIRECT_LOGS | when a true-ish value STDOUT and STDERR are sent to files in the Rails.root/log directory |
| PUMA_SSL_CERT_PATH | When the PUMA_PROTOCOL is "ssl://" this variable is a relative path to the cert.file from Rails.root; it is only used to build PUMA_BIND |
| PUMA_SSL_KEY_PATH | When the PUMA_PROTOCOL is "ssl://" this variable is a relative path to the cert.file from Rails.root; it is only used to build PUMA_BIND |
| PUMA_WORKERS | The number of workers for the Puma master process to start.  Typically 1 per CPU (not core) in the web server. |
| PUMA_WORKER_TIMEOUT | The number of seconds to wait before the Puma master process restarts a stalled Puma worker. |

### Q
### R
| Variable Name     | Description |
|-------------------|----------------------|
| RACK_ENV | (Not set in any `.env*` file) Should have the same value as RAILS_ENV|
| RAILS_ENV | (Not set in any `.env*` file) Establishes the operating environment; typical valures are: development, test, production |

### S
| Variable Name                 | Description |
|-------------------------------|----------------------|
| SSOI_CALLBACK_URL             | Callback URL provided to the Single Sign On - Internal team to post the response back to the CUI application |
| SSOI_ISSUER                   | Issuer on the Single Sign On - Internal side |
| SSOI_IDP_TARGET_URL           | Target URL that CUI application posts the SAML request to |
| SSOI_IDP_CERT_FINGERPRINT     | Finger print generated based on the Single Sign On - Internal certificate |
| SSOI_NAME_IDENTIFIER_FORMAT   | Name identifier provided by the Single Sign On - Internal implementation team |
| SSOI_CERTIFICATE              | Certificate provided by the Single Sign On - Internal impelentation team  |
| SSOI_PRIVATE_KEY              | Environment private key  |
| SAML_RELAY                 | FIXME: undocumented  |
| SANITIZE                   | FIXME: too generic; undocumented |
| SANITIZED_EMAIL            | FIXME: bad namespace; how does this differ from EMAIL_SANITIZE? ; undocumented |
| SECRET_KEY_BASE            | FIXME: undocumented |
| SEND_NOTIFICATIONS_TO_DAS  | true/false, flag that enables/prevents notifications from being sent to DAS

### T

| Variable Name                 | Description |
|-------------------------------|----------------------|
| TEST_STRESS_MAX_REFERRALS     | Count of the entries in the referrals database table |
| TEST_STRESS_MAX_CONSULTATIONS | Count of the entries in the consultations database table |
| TEST_STRESS_APP_URL           | The base URL (eg. http://localhost:4567) of the application |
| TEST_STRESS_SIGN_IN           | The URI of the sign_in page (eg. /sign_in) |
| TEST_STRESS_USERNAME          | The account for the application sign_in |
| TEST_STRESS_PASSWORD          | The password for the sign_in account |
| TEST_STRESS_COUNT             | The number of concurrent users (threads) to use during the test |
| TEST_STRESS_RAMPUP            | The number of seconds to take to get from zero to the total number of threads |
| TEST_STRESS_LOOPS             | How many times should each thread (user) repeat the test sequence |
| TEST_STRESS_PAGES             | The number of pages to access during the test |
| TEST_STRESS_DURATION          | The number of seconds to run the stress test |

The directory `test/stress` contains at least one program (cpp_jmeter.rb) that creates an XML test plan file for use with the Apache Foundation's testing tool `jMeter`.  `jMeter` makes use of a test plan expressed in an XML file format for testing web sites.  The test plan generated by `cpp_jmeter.rb` can be used to do load testing against the application from either a single host.  The test plan generated can also be used with the [flood.io](http://flood.io/) cloud-based load testing service.


### U
| Variable Name              | Description |
|----------------------------|----------------------|
| URL_FOR_NPI_LOOKUP         | The URL to use for accessing the NPI Registry maintained by the Health and Human Services Agency.  FIXME: bad namespace; feature name should come first - consider: NPI_LOOKUP_URL |

### V
| Variable Name                 | Description |
|-------------------------------|----------------------|
| VBMS_UPLOADER_ENV             | FIXME: undocumented |
| VIA_ACTIVE_FOR_CPP            | Setting to toggle using VIA web-service |
| VIA_ACTIVE_FOR_CUI            | Setting to activate the CUI Vista login page |
| VIA_API_URL                   | Base URL for accessing various VIA web-services |
| VIA_HOST                      | Hostname to be added to /etc/hosts for VIA web-services |
| VIA_IP                        | IP address for VIA web-services /etc/hosts file entry|
| VIA_API_ROOT_ENDPOINT         | Endpoint root used when making requests through VIA web-service|
| VIA_API_XML_NAMESPACE         | XML namespace used when parsing VIA web-service XML documents|
| VIA_CONS_APP_TOKEN            | Consuming app token used to authenticate with VIA web-service |
| VIA_CONS_APP_PASS             | Consuming app password used to authenticate VIA web-service |
| VIA_REQ_APP                   | Requesting app used in VIA web-service |
| VLER_DAS_CACERT               | DAS Certificate Authority Path (e.g: /website/ssl/vler-das/cachain.crt)
| VLER_DAS_CERT                 | CUI Certificate Path (e.g: /website/ssl/www.dev-cui.DNS   .crt)
| VLER_DAS_CIPHER_SUITES        | FIXME: undocumented |
| VLER_DAS_ENABLE_NOTIFICATIONS | true/false, flag that enables/prevents notifications from being sent to DAS
| VLER_DAS_EXAM_MANAGEMENT_URL  | DAS Exam Management Endpoint URL (e.g: https://das-sqa.DNS   /ecrud/v1/core/examManagement)
| VLER_DAS_EXAM_RESPONSE_URL    | DAS Exam Resonse Endpoint URL (e.g: https://das-sqa.DNS   /ecrud/v1/core/examResponse/transform)
| VLER_DAS_KEYFILE              | CUI Certificate Key (e.g: /website/ssl/cui.DNS   .key)
| VLER_DAS_KEYPASS              | FIXME: undocumented |
| VLER_DAS_URL                  | FIXME: undocumented |
| VX_API_BASE_URL               | This is the Base URL for Veterans Exchange web-service|
| VX_API_AUTH_URL               | The URL used to establish a session for further communication with VistA |
| VX_API_CREDENTIALS            | The access and verify codes needed for establishing connection VistA |

### W
### X
| Variable Name     | Description |
|-------------------|----------------------|
| XCERTIFICATE | FIXME: too generic; undocumented |
| XIDP_CERT_FINGERPRINT | FIXME: too generic; undocumented |
| XPRIVATE_KEY | FIXME: too generic; undocumented |

### Y
### Z
