#!/usr/bin/env bash
# bin/docker_cui_create.sh
#
# NOTE: This script makes use of the common *NIX utilities: awk, m4

awk_path=$(which awk)
m4_path=$(which m4)

for utility in "awk:$awk_path" "m4:$m4_path" ; do
  echo $utility
  utility_name=`echo $utility | cut -f 1 -d':'`
  utility_path=`echo $utility | cut -f 2 -d':'`

  if [ "" = "$utility_path" ]; then
    echo "SYSTEM CONFIGURATION ERROR: The common *NIX utility '$utility_name' is missing"
    exit 1
  fi
done

# save current working directory
here=`pwd`

my_dir="$( cd "$( dirname "${BASH_SOURCE[0]}" )" && pwd )"
source $my_dir/check_app_root

export DOCKER_DIR=$APP_ROOT/config/docker
export TMP_DIR=$APP_ROOT/tmp

cd $APP_ROOT

function usage {
  echo
  echo "Usage: '$0' container+ [options]"
  echo
  echo "OPTIONS"
  echo "  -h --help   Show this content and exit"
  echo
  echo "Make sure that your RAILS_ENV variable is set to the desired environment value."
  echo "If it is not set (for example via the command line) it will be defaulted to development."
  echo "Even through the .env* files will be process you can not rely upon RAILS_ENV being set in"
  echo "the .env or .env.local file.  You can do this:"
  echo "     RAILS_ENV=production $0"
  echo
  echo "You must specify at least one docker container to be built."
  echo "You can specify all of them at once like this:"
  echo "     RAILS_ENV=production $0 db redis web"
  echo
  echo "Notice that only a space is used to separate the container types.  Do not specify"
  echo "a container type more than once.  If you do an error message will be displayed."
  echo
  echo -n "The following containers have been defined: "
  cd $DOCKER_DIR && ls -1 *_docker-compose.yml | awk -F '_' '{printf " %s", $1}'
  echo
  echo
  cd $here
  exit 0
}


if [ $# -eq 0 ]; then
  usage
fi


if `echo $@ | grep "\-h" 1>/dev/null 2>&1`
then
  usage
fi


servers=""
errors=0

for server in $@ ; do
  # ignore any variable that contains a dash; goal is to remove options
  if `echo $server | grep '\-' 1>/dev/null 2>&1`
  then
    continue
  fi

  if `echo $servers | grep $server 1>/dev/null 2>&1`
  then
    echo "ERROR: $server has been specified more than once."
    let errors++
  else
    if [ -f $DOCKER_DIR/${server}_docker-compose.yml ]; then
      servers="$servers $server"
    else
      echo "ERROR: Invalid container type; ${server}_docker-compose.yml does not exist."
      let errors++
    fi
  fi
done

if [ $errors -gt 0 ]; then
  usage
fi


source $my_dir/setup_environment


cp -f $DOCKER_DIR/*_dockerfile .


for server in $servers ; do
  echo
  echo "Building and launching the application containers from ${server}_docker-compose.yml ..."

  m4 $DOCKER_DIR/${server}_docker-compose.yml > ${server}_docker-compose.yml

  command="docker-compose -f ${server}_docker-compose.yml -p CPP up -d --build"
  echo "Command: $command"
  `$command`

done

# Go back to where you started
cd $here

echo
echo "Done."
echo


