module VIA_API
  module Models

    # NOTE: These properties are used by the database-backed Veteran model.
    #       In order to seemlessly swap out the datastore backends this model
    #       support these as properties.
    #
    #  id                          :integer          not null, primary key
    #  ssn                         :string(10)       not null
    #  first_name                  :string
    #  middle_name                 :string
    #  last_name                   :string
    #  other_health_insurance_name :string
    #  content                     :json
    #  created_at                  :datetime         not null
    #  updated_at                  :datetime         not null
    #  other_health_insurance_id   :integer
    #
    #  autowire_content_fields :gender, :date_of_birth, :address, :city, :state, :zip


    class Veteran < ::EnhancedHash
    # access by:                  API uses:
      property :id
      property :ssn
      property :first_name,     from: :name, transform_with: ->(v) { split_name(v)[0] }
      property :middle_name,    from: :name, transform_with: ->(v) { split_name(v)[1] }
      property :last_name,      from: :name, transform_with: ->(v) { split_name(v)[2] }
      property :other_health_insurance_name
      property :other_health_insurance_id

      property :gender
      property :date_of_birth,  from: :dob, transform_with: ->(v) { formatted_dob(v) }

      # The home address and home phone tags are present only in the patient_to tags in the FindPatientService responses
      property :address,        from: :home_address, transform_with: ->(v) { v[:street_address1] }
      property :city,           from: :home_address, transform_with: ->(v) { v[:city] }
      property :county,         from: :home_address, transform_with: ->(v) { v[:county] }
      property :state,          from: :home_address, transform_with: ->(v) { v[:state] }
      property :zip,            from: :home_address, transform_with: ->(v) { v[:zipcode] }
      property :phone,          from: :home_phone, transform_with: ->(v) { v[:number] }

      # Removing these fields is triggering an error as the web-service
      # returns these fields.

      property :name
      property :age
      property :patient_name

      # SMELL: Are these really necessary for the Veteran class or are they some kind
      #        state information required by the VIA API?

      property :mpi_pid
      property :mpi_checksum
      property :local_pid
      property :confidentiality,
                                  coerce: VIA_API::Models::Veteran::Confidentiality
      property :patient_flags,
                                  coerce: VIA_API::Models::Veteran::PatientFlags

      property :demographics
      property :address,        from: :demographics, transform_with: ->(v) {v[:demographic_set_to][:addresses][:address_to][:street_address1]}
      property :city,           from: :demographics, transform_with: ->(v) {v[:demographic_set_to][:addresses][:address_to][:city]}
      property :county,         from: :demographics, transform_with: ->(v) {v[:demographic_set_to][:addresses][:address_to][:county]}
      property :state,          from: :demographics, transform_with: ->(v) {v[:demographic_set_to][:addresses][:address_to][:state]}
      property :zip,            from: :demographics, transform_with: ->(v) {v[:demographic_set_to][:addresses][:address_to][:zipcode]}
      property :phone,          from: :demographics, transform_with: ->(v) {[v[:demographic_set_to][:phones][:phone_num_to][0][:area_code],
                                                                            v[:demographic_set_to][:phones][:phone_num_to][0][:exchange],
                                                                            v[:demographic_set_to][:phones][:phone_num_to][0][:number]].join('-') }
      property :ethnicity
      property :marital_status
      property :restricted
      property :service_connected
      property :sc_percent
      property :inpatient
      property :needs_means_test
      property :cmor_site_id
      property :active_insurance
      property :is_test_patient
      property :has_insurance
      property :patient_type
      property :is_veteran
      property :is_locally_assigned_mpi_pid
      property :sites
      property :team
      property :cwad
      property :admit_timestamp
      property :location




      def address_attrs
        {:address=>address, :city=>city, :state=>state, :zip=>zip}
      end


      def full_name
        [first_name, middle_name, last_name].join(' ')
      end
      alias_method :patient_name, :full_name

      def formatted_ssn
        "#{ssn.slice(0,3)}-#{ssn.slice(3,2)}-#{ssn.slice(5,4)}"
      end


      # FIXME: This method is coupled to the other_health_insurance database table.
      #        Does VISTA have this kind of information?
      def has_other_health_insurance?
        case other_health_insurance.title.strip.downcase
          when 'yes'  # SMELL: tight coupling to the contents of the static text table
            true
          when 'no'
            false
          else
            raise "Unknown"
        end
      end


      # When using VISTA, consultation numbers should appear in the consultation
      # and this method should not be necessary. (generate_consultation_number should
      # only get called if VIA_ACTIVE_FOR_CPP is false.)
      def generate_consultation_number
        raise ViaApiError.new("Consultations from VISTA should not have auto-generated consultation numbers.")
      end


      class << self

        def filtered_veterans(params, vista_session)
          if params[:from_date].present? && params[:to_date].present?
            from_date = Date.strptime(params[:from_date], "%m/%d/%Y")
            to_date   = Date.strptime(params[:to_date],   "%m/%d/%Y") + 1
            date_filter = { created_at: ( from_date .. to_date ) }
          else
            date_filter = {created_at: ''}
          end

          veteran_filter_params = {
            ssn:        params[:ssn].gsub(/\D/,''),
            first_name: params[:first_name],
            last_name:  params[:last_name]
          }.merge(date_filter).delete_if{ |k,v| v.to_s.strip.blank? }

          if veteran_filter_params[:ssn].present?
            veterans = VIA_API::FindPatientService.mpi_lookup(params[:ssn], vista_session)
          elsif veteran_filter_params[:first_name].present? || veteran_filter_params[:last_name].present?
            # TODO: Add the VIA service call that returns the veterans based on
            #       first_name and last_name
            raise ViaApiError.new("Filtering on first name and last name is currently not available")
          else
            raise ViaApiError.new("Valid search parameters are not available")
          end

          return veterans
        end

        # Duplicates functionality of ActiveRecord Cpp::Local::Veteran#find_by_ssn
        def find_by_ssn(ssn, vista_session)
          veteran_array = filtered_veterans({ssn: ssn}, vista_session)
          veteran_array.first if veteran_array.present?
        end

        # Return an array of name components in the order first, middle, last
        # api_name = last_name,first_name middle_name
        def split_name(api_name)
          raise "Name is empty" if api_name.nil? || api_name.empty?
          raise "Name has no comma: #{api_name}" unless api_name.include?(',')
          name_parts = api_name.split(',')
          last_name = name_parts.first || ''
          first_name_parts = name_parts.last.split(' ')
          first_name   = first_name_parts.first || ''
          middle_name = first_name_parts.last || ''

          return [ first_name, middle_name, last_name]
        end

        # Returns date of birth in yyyy-mm-dd
        # api_dob = yyyymmdd.000000
        def formatted_dob(api_dob)
          "#{api_dob.slice(4,2)}/#{api_dob.slice(6,2)}/#{api_dob.slice(0,4)}"
        end
      end # class << self

    end # class Veteran < EnhancedHash

  end
end # module VIA_API
