# app/via_api/emr_service.rb
module VIA_API
  module EmrService

    class << self

      ##############################################################
      # Retrieves appointments from VISTA, for the patient passed in.
      # patient - VIA_API::Models::Veteran object representing patient to
      #   search for appointments on.
      # vista_session - ??
      # options - basic filtering options:
      #   :future_only (boolean) => if true then remove all appointments before current date
      #   :active_status_only (boolean) => if true then do not return appointments
      #     with status 'NO-SHOW', 'CANCELLED BY CLINIC', or 'CANCELLED BY PATIENT', or
      #     with purpose other than 'C&P', '10-10', or 'SCHEDULED VISIT.
      ##############################################################
      def get_appointments(patient, vista_session , options = {})

        VIA_API.validate_vista_session(vista_session)
        request  = EMR_CLIENT.build_request(:get_appointments, message: get_appointments_request(patient, vista_session)).body
        request  = VIA_API.add_ser_prefix(request)
        response  = EMR_CLIENT.call(:get_appointments, xml: request, soap_action: false)

        # Savon's inbuilt response.body or response.to_hash
        # is unable to parse the response accurately

        xml = response.to_xml

        # Parsing the XML using Nokogiri
        doc = Nokogiri::XML(xml)

        fault_node = doc.xpath '//ns2:TaggedAppointmentArrays/fault', 'ns2' => ENV['VIA_API_XML_NAMESPACE']
        if fault_node.present?
          raise ViaApiFaultError.new(fault_node.text)
        end

        # Appointments as Nokogiri elements
        appts = doc.xpath "//appointmentTO"

        appointments = []

        # Building an array of appointment hashes
        appts.each do |raw_appt|
          hashified_appt = VIA_API.x2h(raw_appt)["appointmentTO"].deep_symbolize_keys!
          appointment = VIA_API::Models::Appointment.new(hashified_appt)
          appointments << appointment unless appointment.ignored_appointment?(options)
        end

        return appointments

      end # def get_appointment

      def match(ssn, vista_session)
        VIA_API.validate_vista_session(vista_session)
        request  = EMR_CLIENT.build_request(:match, message: match_request(ssn,vista_session)).body
        request  = VIA_API.add_ser_prefix(request)
        response  = EMR_CLIENT.call(:match, xml: request, soap_action: false)

        data = response.body

        patient_raw_data = data.dig(:match_response, :tagged_patient_array, :patients, :patient_to)
        patient_raw_data_count = data.dig(:match_response, :tagged_patient_array, :count)
        fault_message = data.dig(:match_response, :tagged_patient_array, :fault, :message)

        if fault_message.present?
          raise ViaApiFaultError.new(fault_message)
        end

        veterans = []
        if patient_raw_data_count.to_i > 1
          patient_raw_data.each do |vet|
            veterans << VIA_API::Models::Veteran.new(vet)
          end
        else
          veterans = [VIA_API::Models::Veteran.new(patient_raw_data)]
        end

        return veterans
      end

      def match_request(ssn, vista_session)

        request = {}

        request[:queryBean] = {
          provider: {
            name:           vista_session[:user_name],
            userId:         vista_session[:duz],
            loginSiteCode:  vista_session[:site_id]
          },
          target:             ssn,
          requestingApp:        ENV['VIA_REQ_APP'],
          consumingAppToken:    ENV['VIA_CONS_APP_TOKEN'],
          consumingAppPassword: ENV['VIA_CONS_APP_PASS']
        }

        return request
      end

      def select(patient, vista_session)
        VIA_API.validate_vista_session(vista_session)
        request  = EMR_CLIENT.build_request(:select, message: select_request(patient,vista_session)).body
        request  = VIA_API.add_ser_prefix(request)
        response  = EMR_CLIENT.call(:select, xml: request, soap_action: false)

        data = response.body

        patient_raw_data = data.dig(:select_response, :patient_to)
        fault_message = data.dig(:select_response, :fault, :message)

        if fault_message.present?
          raise ViaApiFaultError.new(fault_message)
        end

        veteran = nil
        if !patient_raw_data.nil?
          veteran = VIA_API::Models::Veteran.new(patient_raw_data)
        end
        return veteran
      end

      def select_request(patient,vista_session)
        request = {}

        request[:queryBean] = {
          provider: {
            name:               vista_session[:user_name],
            userId:             vista_session[:duz],
            loginSiteCode:      vista_session[:site_id]
          },
          patient: {
            localPid:           patient.local_pid,
            localSiteId:        vista_session[:site_id]
          },
          requestingApp:        ENV['VIA_REQ_APP'],
          consumingAppToken:    ENV['VIA_CONS_APP_TOKEN'],
          consumingAppPassword: ENV['VIA_CONS_APP_PASS']
        }
        return request
      end

      def get_appointments_request(patient, vista_session)

        request = {}

        request[:queryBean] = {
          provider: {
            userId:         vista_session[:duz],
            name:           vista_session[:user_name],
            loginSiteCode:  vista_session[:site_id]
          },
          patient: {
            localPid:     patient.local_pid,
            localSiteId:  vista_session[:site_id],
            mpiPid:       patient.mpi_pid
          },
          requestingApp:  ENV['VIA_REQ_APP'],
          consumingAppToken:    ENV['VIA_CONS_APP_TOKEN'],
          consumingAppPassword: ENV['VIA_CONS_APP_PASS']
        }

        return request
      end # def get_appointment_request

      ##############################################################
      def get_consults_for_patient(patient, vista_session)

        VIA_API.validate_vista_session(vista_session)
        request  = EMR_CLIENT.build_request(:get_consults_for_patient, message: get_consults_for_patient_request(patient, vista_session)).body
        request  = VIA_API.add_ser_prefix(request)
        response  = EMR_CLIENT.call(:get_consults_for_patient, xml: request, soap_action: false)

        data = response.body

        consult_raw_data = data.dig(:get_consults_for_patient_response, :tagged_consult_arrays,
                                   :arrays, :tagged_consult_array, :consults, :consult_to)
        consult_raw_data_count = data.dig(:get_consults_for_patient_response, :tagged_consult_arrays,
                                   :arrays, :tagged_consult_array, :count)
        fault_message = data.dig(:get_consults_for_patient_response, :tagged_consult_arrays, :fault, :message)

        if fault_message.present?
          raise ViaApiFaultError.new(fault_message)
        end

        consults = []

        if consult_raw_data_count.to_i > 1
          consult_raw_data.each do |consult|
            consult[:timestamp] = VIA_API::parse_vista_date(consult[:timestamp]).strftime(TIMESTAMP_FORMAT)
            consults << VIA_API::Models::Consultation.new(consult)
          end
        else
          consult_raw_data[:timestamp] = VIA_API::parse_vista_date(consult_raw_data[:timestamp]).strftime(TIMESTAMP_FORMAT)
          consults = [VIA_API::Models::Consultation.new(consult_raw_data)]
        end

        return consults

      end # def get_consults_for_patient


      def get_consults_for_patient_request(patient, vista_session)

        request = {}

        request[:queryBean] = {
          provider: {
            userId:         vista_session[:duz],
            name:           vista_session[:user_name],
            loginSiteCode:  vista_session[:site_id]
          },
          patient: {
            localPid: patient.local_pid,
            localSiteId: vista_session[:site_id],
            mpiPid: patient.mpi_pid
          },
          requestingApp:  ENV['VIA_REQ_APP'],
          consumingAppToken:    ENV['VIA_CONS_APP_TOKEN'],
          consumingAppPassword: ENV['VIA_CONS_APP_PASS']
        }

        return request
      end # def get_consult_for_patient_request

    end # class << self
  end # module EmrService
end # module VIA_API
