# == Schema Information
#
# Table name: symptoms
#
#  id              :integer          not null, primary key
#  name            :string
#  label           :string
#  created_at      :datetime         not null
#  updated_at      :datetime         not null
#  minor_system_id :integer
#  position        :integer
#

class Symptom < ActiveRecord::Base
  include ContentManagerHelper
  has_many :symp_diag_relations, dependent: :delete_all
  has_many :diagnoses, through: :symp_diag_relations
  has_many :dbq_informations, through: :symptoms_dbq_informations, dependent: :destroy
  has_many :symptoms_dbq_informations
  belongs_to :minor_system
  validates_presence_of :name
  validates_presence_of :label
  validates_uniqueness_of :name

  before_save :set_name
  before_update :set_name

  scope :order_by_position, -> () { includes(:diagnoses).order("position ASC" ) }

  FIELDS_FOR_RESULTSET = {
    0 => 'symptoms.name',
    1 => 'symptoms.label'
  }

  class << self
    def filter(params={})

      query = Symptom.all

      unless params[:search][:value].blank?
        query = filter_using_symptom(params, query)
      end

      return query
    end

    def filter_using_symptom(cp, q)
      q.where('name ilike :search OR label ilike :search', :search => "%#{cp[:search][:value]}%")
    end
  end

  def html_dbq_info
    dbq_information_ids.join(", ")
  end 

  def build_html(the_diagnosis_style_to_use='yes_no')   # the other value is 'checkbox'
    case the_diagnosis_style_to_use
      when 'checkbox'
        @shtml = base_html.clone
        @shtml.gsub!('[Symptom Name]', self.name)
        @shtml.gsub!('[Symptom Label]', self.label)
        @shtml.gsub!('[dbq-assoc]', html_dbq_info)
        @shtml.gsub!('[Associated Diagnoses *1]', self.diagnoses_codes_string)
        @shtml.gsub!('[Associated Diagnoses *2]', self.diagnoses_names_string)
      when 'yes_no'
        @shtml = the_html_for_a_yes_no_style_symptom
      else
        raise 'An invalid symptom style was provided'
    end
    return @shtml
  end

  def diagnoses_codes_string
    code_str = ""
    self.diagnoses.each do |d|
      code_str += ", " + d.code
    end
    code_str.sub!(", ", "")
    return code_str.html_safe
  end

  def diagnoses_names_string
    names_str = ""
    self.diagnoses.each do |d|
      names_str += ", diagnosis-" + d.name
    end
    names_str.sub!(", ", "")
    return names_str.html_safe
  end

  def base_html
    SYMPTOM_HTML
  end

  SYMPTOM_HTML = '<div id="individual_symptom_container-[Symptom Name]" class="container symptom individual" data-cui-dbq-assoc="[dbq-assoc]">
  <input id="symptom-[Symptom Name]" name="symptom-[Symptom Name]" type="checkbox" value="[Associated Diagnoses *1]" visibility_dependents="" bump_dependents="[Associated Diagnoses *2]" onchange="Set_Dependent_Visibility_Based_On(this);Bump_Dependent_Elements_Based_On(this);">
  <label for="symptom-[Symptom Name]">
	  [Symptom Label]
  </label>
  </div>'

  def the_html_for_a_yes_no_style_symptom
    the_yes_choice_value_string = "unassigned"
    if( self.diagnoses_codes_string != "" )
      the_yes_choice_value_string = self.diagnoses_codes_string
    end

    return  '<div id="individual_symptom_container-' + self.name + '" class="container symptom individual" data-cui-dbq-assoc="' + html_dbq_info + '">'  +
              '<input   id="symptom-' + self.name + '-yes"'                                                                                   +
                      ' type="radio"'                                                                                                           +
                      ' name="symptom-' + self.name + '-yes_no"'                                                                              +
                      ' visibility_dependents="diagnosis_modifiers-' + self.name + '"'                                                          +
#                      ' onchange="Set_Dependent_Visibility_Based_On(this);"'                                                                    +
#cs                      ' value="' + self.diagnoses_codes_string + '">Yes '                                                                                         +
                      ' value="' + the_yes_choice_value_string + '">Yes '                                                                                         +
              '<input   id="symptom-' + self.name + '-no"'                                                                                    +
                      ' type="radio"'                                                                                                           +
                      ' name="symptom-' + self.name + '-yes_no"'                                                                              +
                      ' visibility_dependents="symptom-' + self.name + '"'                                                          +
#                      ' onchange="Set_Dependent_Visibility_Based_On(this);"'                                                                    +
#                      ' value="' + self.code + '">No '                                                                                          +
                      ' value="">No '                                                                                                           +
               ' - ' + self.label                                                                                                               +
#              '<div id="diagnosis_modifiers-' + self.name + '" class="container lab-results inner small-12 columns" style="display: none">'     +
# 	              '[HTML for associated diagnosis modifiers]'                                                                                     +
#              '</div>'                                                                                                                          +
            '</div>'
  end


  def set_name
    self.name = clean_name(self.name)
  end

end
