# == Schema Information
#
# Table name: referral_statuses
#
#  id                  :integer          not null, primary key
#  name                :string
#  code                :string
#  update_description  :string
#  referral_queue      :string
#  filterable_by_roles :string
#

class ReferralStatus < ActiveRecord::Base
  has_many :referrals

  # constant used to designate hardcoded status code name for 'VA Review'
  VA_REVIEW_CODE = 'REVIEW_PENDING'

  # status codes visible for non-VHA users
  NON_VHA_STATUS_CODES = ['ASSIGNED', 'REVIEW_PENDING', 'INFORMATION_NEEDED', 'ACCEPTED', 'COMPLETE']

  # workflow phases where non-VA users can upload documents
  NON_VA_ADD_DOC_PHASES = ['ASSIGNED', 'ACCEPTED', 'INFORMATION_NEEDED']

  STAT_REPORT_IGNORE = ['ACCEPTED', 'COMPLETE']

  EXPIRED_REPORT = ['COMPLETE']

  # Constants used to send emails to Community Providers
  # Referral is assigned to a Community Provider
  ASSIGNED = 'ASSIGNED'.freeze

  # VA user requests Community Provider for more information
  NEED_INFO = 'INFORMATION_NEEDED'.freeze


  class << self

    # get hardcoded review phase, and keeping phase-specific biz logic contained in model
    def va_review_phase
      find_by_code VA_REVIEW_CODE
    end

  end

  # 1/17/17: fixed bugs in code as per PP-245 pull request comment suggestion by Dewayne.
  def filterable_by_user?(current_user)
    result = false
    filterable_by_roles.split(',').map{|s| s.strip}.each do |role|
      result ||= current_user.has_role?(role)
    end
    return result
  end

  # 1/17/17: fixed bugs in code as per PP-245 pull request comment suggestion by Dewayne.
  def add_to_referral_queue?(current_user)
    result = false
    referral_queue.split(',').map{|s| s.strip}.each do |role|
      result ||= current_user.has_role?(role)
    end
    return result
  end

  def non_vha_add_document_phase?
    NON_VA_ADD_DOC_PHASES.include? code.upcase
  end

  def vha_valid_for_approve?
    [VA_REVIEW_CODE].include? code.upcase
  end

  def visible_for_non_vha?
    NON_VHA_STATUS_CODES.include? code.upcase
  end

end
