# == Schema Information
#
# Table name: referral_appointments
#
#  id               :integer          not null, primary key
#  content          :json
#  created_at       :datetime         not null
#  updated_at       :datetime         not null
#  referral_id      :integer
#  appointment_time :datetime
#  added_to_cprs_at :datetime
#  added_to_cprs_id :integer
#

class ReferralAppointment < ActiveRecord::Base
  has_paper_trail

  belongs_to :referral
  belongs_to :added_to_cprs_user, class_name: 'User', foreign_key: 'added_to_cprs_id', validate: true

  validates_presence_of :appointment_type, message: "Please add a appointment description"
  validates_presence_of :appointment_time, message: "Please select a date and time for the appointment."

  validate :valid_appointment_time?, :valid_referral_provider?

  autowire_content_fields :appointment_type, :scheduled_by, :book_notes, :purpose, :appointment_end_time, :veteran_id

  scope :needs_reminder, -> {where("appointment_time >= ? AND appointment_time <= ?",
                      (Time.now - 3.days).beginning_of_day, (Time.now - 1.days).end_of_day) }

  scope :all_by_vet, ->(veteran_id) do where("content ->>'veteran_id' = ?", veteran_id.to_s) end
  scope :all_by_scheduler, ->(scheduler) do where("content ->>'scheduled_by' = ?", scheduler.to_s) end
  scope :all_within_time, ->(start_time, interval) do where("appointment_time > ? AND appointment_time < ?", (start_time - interval), (start_time + interval)) end

  APPOINTMENT_TIME_STEP = 15
  APPOINTMENT_TIME_FORMAT = '%m/%d/%Y %H:%M %p'
  VET_TIME_CONFLICT_MESSAGE = 'the selected appointment time conflicts with another appointment for this veteran.  Please select another time.'.freeze

  def added_to_cprs?
    added_to_cprs_at.present?
  end

  def added_to_cprs_username
    added_to_cprs_user.try! :name
  end

  def vista_appointment?
    false
  end

  # returns an array of fields that appear in 'Details' section of appointment datatable
  def details
    [
       "Scheduled By: " + scheduled_by,
      "Specialty: " + referral.provider.medical_specialties.pluck(:title).join(', ')
    ]
  end


  protected

  def valid_appointment_time?
    return unless appointment_time
    valid_vet = true
    veteran_id = (referral.content['veteran_id'].nil? ? self.veteran_id : referral.content['veteran_id']) rescue nil
    if veteran_id
      vet_conflicts = ReferralAppointment.all_within_time(appointment_time, ReferralAppointment::APPOINTMENT_TIME_STEP.minutes).all_by_vet(veteran_id).to_a +
                      ReferralAppointment.all_within_time(appointment_time, ReferralAppointment::APPOINTMENT_TIME_STEP.minutes).select do  |ra| ra.referral.veteran_id.eql? veteran_id end
      valid_vet = vet_conflicts.reject {|ra| self.id.eql?(ra.id)}.empty?
      errors.add('Appointment time is not valid,', VET_TIME_CONFLICT_MESSAGE) unless valid_vet

    else
      #if we don't have a vet we cannot have a conflict.
    end
    valid_vet
  end

  def valid_referral_provider?
    return true unless referral.provider_id.nil?
    errors.add(:base, "Please select a Provider on the Referral Details tab before creating an appointment.")
    false
  end

end # class ReferralAppointment < ActiveRecord::Base
#load('./app/models/referral_appointment.rb')