# == Schema Information
#
# Table name: referrals
#
#  id                   :integer          not null, primary key
#  coordinator_id       :integer          not null
#  referral_number      :string           not null
#  authorization_number :string           not null
#  content              :json
#  created_at           :datetime         not null
#  updated_at           :datetime         not null
#  consultation_id      :integer
#  referral_status_id   :integer
#  referral_type_id     :integer
#  referral_reason_id   :integer
#  provider_id          :integer
#

# The Referral model is currently backed by a database record.
# However in the objective system it will be some kind of
# ActiveModel backed by a web-service.

class Referral < ActiveRecord::Base
  has_paper_trail
  include AuditTrail

  belongs_to :consultation
  belongs_to :coordinator, class_name: 'User', foreign_key: 'coordinator_id', validate: true
  belongs_to :referral_status
  belongs_to :referral_type
  belongs_to :referral_reason
  belongs_to :provider

  has_many :facility, through: :provider
  has_many :visn,     through: :provider
  # has_many :veteran,  through: :consultation

  has_many :referral_appointments
  has_many :referral_approvals
  has_many :referral_documents
  has_many :referral_notes

  validates_presence_of :referral_type, :coordinator_id
  validates :authorization_number, presence:true, uniqueness: true

  autowire_content_fields :diagnosis_codes, :procedure_codes, :length_of_stay, :length_of_procedure,
    :treatment_frequency, :number_of_visits, :cost_estimate, :status_message, :veteran_id

  scope :completed_recent_only, -> { joins(:referral_status).joins(:consultation) \
    .where("referral_statuses.code <> ? OR consultations.valid_to > ?", 'COMPLETE', Time.now - 30.days)
  }

  scope :non_vha_visible_statuses, -> { joins(:referral_status).where("referral_statuses.code in (?) ",
    ReferralStatus::NON_VHA_STATUS_CODES) }

  scope :stat, -> { includes(:consultation).joins(:referral_status)
          .where("consultations.content ->> 'urgent_priority' = ? AND referral_statuses.code NOT IN (?)", 'true', ReferralStatus::STAT_REPORT_IGNORE)
          .references(:consultations)
  }

  scope :expired, -> { includes(:consultation).joins(:referral_status)
          .where("consultations.valid_to < ? AND referral_statuses.code NOT IN (?)",DateTime.now, ReferralStatus::EXPIRED_REPORT)
          .references(:consultations)
  }

  # Drop referrals whose associated consultation period of validity has expired over 30 days ago.

  scope :completed_recent_only_nonvha, -> {joins(:consultation).where("consultations.valid_to > ?", Time.now - 30.days) }

  scope :with_new_appointments, -> { includes(:referral_appointments)
          .where("referral_appointments.id is NOT NULL AND referral_appointments.added_to_cprs_at is NULL")
          .references(:referral_appointments)
  }

  scope :medical_record_letters, -> { includes(:referral_appointments)
          .references(:referral_appointments)
          .joins(:referral_status)
          .where("referral_appointments.appointment_time <= ? AND referral_statuses.code in (?)",
                      (Time.now - 3.days).beginning_of_day, ReferralStatus::NON_VA_ADD_DOC_PHASES)
  }

  # Currently looking for appointments that are scheduled for 1-3 days from current date to send reminder emails
  scope :needs_reminder, -> { includes(:referral_appointments)
          .references(:referral_appointments)
          .joins(:referral_status)
          .where("referral_appointments.appointment_time >= ? AND referral_appointments.appointment_time <= ? AND referral_statuses.code in (?)",
                      (Time.now - 3.days).beginning_of_day, (Time.now - 1.days).end_of_day , ReferralStatus::NON_VA_ADD_DOC_PHASES)
  }

  # Ordering referrals by provider name using ProviderDataPresenter business logic:
  # 1. Use Referral provider name if referral provider exists, or ...
  # 2. Use Consultation ordering provider name if ordering provider exists, or ...
  # 3. Use 'requesting_provider_contact_name' field of Consultation data record.
  scope :order_by_provider_name, -> (descending=false) { joins(:consultation)
    .joins('left join providers as ordering_providers on ordering_providers.id = consultations.ordering_provider_id')
    .references(:provider)
    .includes(:provider)
    .order("(case when providers.id is not null then providers.name else " +
           "  (case when ordering_providers.id is not null then ordering_providers.name " +
                   "else consultations.content ->> 'requesting_provider_contact_name' end)" +
           " end) #{'DESC' if descending}")
  }

  # Ordering referrals by "VISN:facility" using ProviderDataPresenter business logic:
  # 1. Use facility and VISN corresponding to referral provider if referral provider exists, or ...
  # 2. Use facility and VISN corresponding to consultation ordering provider name if ordering provider exists, or ...
  # 3. Field is null.
  scope :order_by_visn_facility, -> (descending=false) { joins(:consultation)
    .joins('left join providers as ordering_providers on ordering_providers.id = consultations.ordering_provider_id')
    .references(:provider)
    .includes(:provider)
    .joins('left join facilities as rp_facilities on rp_facilities.id = providers.facility_id')
    .joins('left join facilities as op_facilities on op_facilities.id = ordering_providers.facility_id')
    .order("(case when providers.id is not null then ( rp_facilities.visn_id, rp_facilities.name) else " +
           "  (case when ordering_providers.id is not null then (op_facilities.visn_id, op_facilities.name) " +
                   "else null end)" +
           " end) #{'DESC' if descending}")
  }


  VALID_FILTER_PARAM_KEYS = {
    referral: [
      :authorization_number,
      :coordinator_id,
      :created_at,
      :from_date,
      :referral_type_id,
      :status,
      :to_date,
      :medical_record_letters
    ],

    consultation: [
      :care_category_id,
      :consultation_number,
      :stat_data,
      :expired_data,
      :new_appointments,
      :first_name,
      :last_name,
      :ssn,
      :dob
    ],

    provider: [
      :provider_id,
      :facility_id,
      :visn_id,
    ]
  }

  after_save do
    if referral_status_id_changed? && skip_callbacks != true
      notes_changes = humanize_the_id_field_values.values.last
      status = notes_changes["fields_changed"]["referral_status_id"].last
      ReferralNote.create(user: notes_changes["changed_by"], notes: 'Status set to '+ status, referral: self)
    end
  end

  # Get veteran from associated consultation.
  # current_user is passsed in, in case the veteran needs to be looked up through VISTA.
  def veteran(vista_session)
    consultation.veteran(vista_session)
  end

  def use_ordering_provider
    provider.blank?
  end


  # Display all documents if user is VHA; display only approved documents
  # or self-submitted documents if user is non-VHA.  Order by most recent,
  # and make sure empty list gets returned if no documents match.
  def visible_documents_for_referral(user)
    referral_documents.includes(:referral_document_type, :uploader).order('updated_at DESC').select {|d| d.visible?(user) }
  end

  def deleted_visible_documents_for_referral(user)
    referral_documents.only_deleted.includes(:referral_document_type, :uploader).order('deleted_at DESC').select {|d| d.visible?(user) }
  end

  def status_name
    referral_status.name
  end


  def referral_details_dropdown
    "##{authorization_number} #{created_at.strftime(TIMESTAMP_FORMAT)} #{referral_type.title} #{consultation.care_category.title}"
  end


  # shorthand way to reference the referral_notes table
  def user_notes
    referral_notes
  end

  # Arel mappings for resultset field names, based on SQL (not ActiveRecord)
  # table and column names.  Array fields are added as multiple ORDER BY parameters.
  FIELDS_FOR_RESULTSET = {
    vha: {
      0 => ['consultations.veteran_first_name', 'consultations.veteran_middle_name', 'consultations.veteran_last_name'],
      1 => 'referral_types.title',
      2 => 'consultations.veteran_ssn',
      3 => 'referrals.created_at',
      4 => 'referral_statuses.code',
      5 => 'users.first_name'
    },
    non_vha: {
      0 => ['consultations.veteran_first_name', 'consultations.veteran_middle_name', 'consultations.veteran_last_name'],
      1 => 'referral_types.title',
      2 => 'referrals.created_at',
      3 => 'referral_statuses.code',
      4 => 'users.first_name',
      5 => 'referrals.authorization_number',
      6 => 'consultations.consultation_number'
    }
  }

  # a list of all referral datatable column indices
  COLUMNS = {
    patient_name: '0',
    referral: '1',
    ssn: '2',
    date: '3',
    status: '4',
    coordinator: '5',
    provider: '6',
    visn_facility: '7'
  }.freeze


  # column names containing custom ordering scopes
  CUSTOM_FIELD_COLUMNS = {
    vha: [COLUMNS[:provider], COLUMNS[:visn_facility]],
    non_vha: []
  }

  #######################################################################
  ###
  ##    The Eigen class
  #

  class << self

    include DataTablePaginator

    # Remove any parameters which are not needed by this model's filter function and/or
    # whose value is blank.

    def clean_the_params(params)
      return params if params.blank?

      selected_param_keys = (VALID_FILTER_PARAM_KEYS.values.flatten.map(&:to_s) & params.keys).
        select{ |key| !params[key].blank? }

      clean_params = params.select {|key| selected_param_keys.include? key }.symbolize_keys

      # TODO: make sure to validate user input of either a from_date or both but not just a to_date
      clean_params[:created_at] = DateHelpers.create_range(
                  clean_params[:from_date],
                  clean_params.has_key?(:to_date) ? clean_params[:to_date] : clean_params[:from_date]
            )  if clean_params.has_key?(:from_date)

      # NOTE: allows user to enter ssn like: xxx-xx-xxxx
      clean_params[:ssn].gsub!(/\D/,'') if clean_params.has_key? :ssn

      return clean_params
    end # def clean_the_params(params)


    ########################################################################
    # Returns an instance of the class ActiveRecord_Relation (AREL)

    def filter(params={}, user)
      filter_params = clean_the_params(params)
      filter_params ||= {}

      # Start with basic "all possible visible records for user" query,
      # then join on all associated tables needed for ORDER BY clause (sorting by row)
      query = basic_filter(user)
                .joins(:consultation).joins(:coordinator)
                .joins(:referral_type).joins(:referral_status)

      unless filter_params.blank?
        %w[   filter_using_self
              filter_using_consultation
              filter_using_provider
        ].map(&:to_sym).each do |filter_method|
          query = send(filter_method, filter_params, query)
        end
      end
      return query
    end

    # Default filtering that always applies when viewing referrals, without
    # any additional filtering based on user-selected parameters.
    def basic_filter(user)
      if user.is_non_vha?
        query_without_provider = all.non_vha_visible_statuses.completed_recent_only_nonvha
        provider_filter = {provider_id: user.providers.pluck(:id)}
        return filter_using_provider(provider_filter, query_without_provider)
      else
        return all.completed_recent_only
      end
    end

    ########################################################################
    # Add query components that deal with attributes that are in the Referral model.

    def filter_using_self(f_params, query = all)
      return query if (VALID_FILTER_PARAM_KEYS[:referral] & f_params.keys).empty?

      # NOTE: supports case insensitive starts_with?

      if f_params.has_key? :authorization_number
        query = query.where("referrals.authorization_number ilike ?",
          "#{f_params[:authorization_number]}%")
      end

      if f_params.has_key? :coordinator_id
        query = query.where('coordinator_id = ?',
          f_params[:coordinator_id])
      end

      if (f_params.has_key? :from_date) || (f_params.has_key? :to_date)
        from_date = Date.strptime(f_params[:from_date],'%m/%d/%Y').beginning_of_day
        to_date = Date.strptime(f_params[:to_date],'%m/%d/%Y').end_of_day
        query = query.where('consultations.created_at BETWEEN ? AND ?', from_date, to_date)
      end

      if f_params.has_key? :referral_type_id
        query = query.where('referral_type_id = ?',
          f_params[:referral_type_id])
      end

      if f_params.has_key? :status
        query = query.where('referral_status_id in (?)',
          f_params[:status])
      end

      if f_params.has_key? :medical_record_letters
        query = query.medical_record_letters
      end

      return query
    end # def filter_using_self(f_params, query = all)


    ########################################################################
    # Add query components that deal with attributes that are an inherent
    # part of the main record and any static text sub-records.

    def filter_using_consultation(f_params, query = all)
      return query if (VALID_FILTER_PARAM_KEYS[:consultation] & f_params.keys).empty?

      if f_params.has_key? :care_category_id
        query = query.where('consultations.care_category_id = ?',
          f_params[:care_category_id])
      end

      # NOTE: supports case insensitive starts_with?

      if f_params.has_key? :consultation_number
        query = query.where("consultations.consultation_number ilike ?",
          "#{f_params[:consultation_number]}%")
      end

      if f_params.has_key? :stat_data
        query = query.stat
      end

      if f_params.has_key? :new_appointments
        query = query.with_new_appointments
      end

      if f_params.has_key? :expired_data
        query = query.expired
      end

      if f_params.has_key? :first_name
        query = query.where("consultations.veteran_first_name ilike ?", "#{f_params[:first_name]}%")
      end

      # NOTE: supports case insensitive starts_with?

      if f_params.has_key? :last_name
        query = query.where("consultations.veteran_last_name ilike ?", "#{f_params[:last_name]}%")
      end

      # NOTE: supports searching for last 4 digits
      #       This is an IMPORTANT Personal Privacy issue when
      #       dealing with people over the phone - they do not
      #       want to give out their full SSN because other
      #       people may be in the room eavesdropping.

      if f_params.has_key? :ssn
        query = query.where("consultations.veteran_ssn like ?", "%#{f_params[:ssn]}")
      end

      if f_params.has_key? :dob
        query = query.where("consultations.veteran_dob like ?", "%#{f_params[:dob]}")
      end

      return query
    end # def filter_using_self(fp, query = all)


    ########################################################################
    # Add query components that deal with attributes that are in the Provider model.
    # FIXME: when the ordering physician requirement gets settled.
    # SMELL: The facility/visn components need to be clarified as to whether they are
    #        based upon the ordering physician or the referral provider.

    def filter_using_provider(f_params, query = all)
      return query if (VALID_FILTER_PARAM_KEYS[:provider] & f_params.keys).empty?

      query = query.includes(:facility, :visn)

      # NOTE: this next where clause filters on the provider_id of the
      #       referral - not the ordering physician.

      # FIXME: when the ordering physician/provider
      #        requirement gets settled.

      if f_params.has_key? :provider_id
        query = query.where("referrals.provider_id in (?)",
                f_params[:provider_id])
      end

      # NOTE: The geolocation queries are based upon
      #       the referral's provider not the ordering
      #       physician.

      # FIXME: when the ordering physician requirement
      #        gets settled.

      # NOTE: no reason to search the facility since we
      #       already have the provider.

      unless  f_params.has_key?(:provider_id)

        if f_params.has_key? :facility_id
          query = query.where('facilities.id = ?',
                  f_params[:facility_id]).references(:facilities)
        end

        # NOTE: no reason to search for the region since we
        #       already have the facility.

        unless  f_params.has_key?(:facility_id)

          if f_params.has_key? :visn_id
            query = query.where('visns.id = ?',
                  f_params[:visn_id]).references(:visns)
          end

        end # unless  f_params.has_key?(:facility_id)

      end # unless  f_params.has_key?(:provider_id)

      return query
    end # def filter_using_provider(f_params, query = all)


    # Returns a paginated list of referrals, based on the DataTables-based
    # params passed in:
    # pagination_params[:draw] -- sequential number used for asynchronous ordering
    # pagination_params[:start] -- offset of query resultset
    # pagination_params[:length] -- length of resultset
    # pagination_params[:order] -- a hash-of-hashes structure containing a collection of
    #   orderings, with each value being a hash with values for "column"
    #   (sorting column) and "dir" (direction)
    def get_paginated_list(referrals, user, pagination_params)
      if user.is_vha_cc?
        referral_to_fields_func = -> (referral) do
          provider_data = ProviderDataPresenter.new(referral)
          {
            id:            referral.id,
            patient_name:  referral.consultation.veteran_full_name,
            referral:      referral.referral_type.title,
            ssn:           referral.consultation.formatted_ssn,
            date:          referral.created_at.strftime(DATE_FORMAT),
            status:        referral.referral_status.name,
            coordinator:   referral.coordinator.name,
            provider:      provider_data.name,
            visn_facility: provider_data.facility.try(:visn_region_prepended_name) || 'N/A'
          }
        end
        ordered_referrals = apply_ordering_to_query(
          referrals, FIELDS_FOR_RESULTSET[:vha], pagination_params[:order]
        )
        ordered_referrals = apply_vha_custom_ordering(ordered_referrals, pagination_params[:order])
      else
        referral_to_fields_func = -> (referral) do
          {
            id:                   referral.id,
            patient_name:         referral.consultation.veteran_full_name,
            referral:             referral.referral_type.title,
            date:                 referral.created_at.strftime(DATE_FORMAT),
            status:               referral.referral_status.name,
            coordinator:          referral.coordinator.name,
            authorization_number: referral.authorization_number,
            consultation_number:  referral.consultation.consultation_number
          }
        end
        ordered_referrals = apply_ordering_to_query(
          referrals, FIELDS_FOR_RESULTSET[:non_vha], pagination_params[:order]
        )
      end
      return datatables_json_for_query_using_func(
        ordered_referrals, referral_to_fields_func, pagination_params
      )
    end


    # Handles custom ordering for VHA users when the column number is one of:
    # COLUMNS[:provider] or COLUMNS[:visn_facility].
    def apply_vha_custom_ordering(referrals, ordering_params)
      custom_ordering_params = (ordering_params || {}).select do |k,v|
        v['column'].present? && (CUSTOM_FIELD_COLUMNS[:vha].include? v['column'].to_s)
      end
      # if custom ordering column exists, then take first column and order based
      if custom_ordering_params.present?
        custom_ordering = custom_ordering_params.values.first
        if (custom_ordering['column'] == COLUMNS[:provider].to_s)
          referrals = referrals.order_by_provider_name(custom_ordering['dir'].to_s.downcase == 'desc')
        elsif (custom_ordering['column'] == COLUMNS[:visn_facility].to_s)
          referrals = referrals.order_by_visn_facility(custom_ordering['dir'].to_s.downcase == 'desc')
        end
      end
      return referrals
    end


    # Returns an array of referral status with counts of the number of referrals that have that status.
    # The status that are returned are based upon the role of the current user - where vha or non-vha
    def group_by_status_and_count(current_user)
      status_role           = current_user.is_vha_cc? ? 'vha_cc' : (current_user.is_non_vha? ? 'non_vha' : '')
      status_count_hash     = Referral.group(:referral_status_id).count
      raw_referral_status_array = ReferralStatus.all.collect do |rs|
        [   rs.name, {
                id:     rs.id,
                count:  status_count_hash[rs.id]
            }
        ] if rs.referral_queue.include?(status_role) && status_count_hash.keys.include?(rs.id)
      end
      referral_status_array = raw_referral_status_array.compact.to_h
    end

    def status_count(current_user, status)
      Referral.where(referral_status_id: status).count
    end

    def stat_count(current_user)
      if current_user.is_non_vha?
        self.non_vha_visible_statuses.stat.count
      else
        self.completed_recent_only.stat.count
      end
    end

    def expired_count(current_user)
      completed_recent_only.expired.count
    end

    def with_new_appointments_count
      with_new_appointments.count
    end

    def med_letter_count(current_user)
      medical_record_letters.count
    end

=begin

    def filtered_referrals(params)
      get_referrals(params)
      referral_filter_params = {referral_status_id: params[:status].try(:map, &:to_i), referral_type_id: params[:referral_type_id].to_i, authorization_number: params[:authorization_number]}.delete_if{|k,v| v.blank? or v == 0}
      if (referral_filter_params.nil? or referral_filter_params.empty?)
        return @filtered_ref
      else
        return @filtered_ref.where(referral_filter_params)
      end
    end

    def get_referrals(params)
      query = self.joins(provider: :facility)
      if !(params[:visn_id].nil? or params[:visn_id].empty?) && !(params[:facility_id].nil? or params[:facility_id].empty?) && !(params[:provider_id].nil? or params[:provider_id].empty?)
        @filtered_ref = query.where('referrals.provider_id=?', params[:provider_id]).where('providers.facility_id=?',params[:facility_id])
      elsif !(params[:visn_id].nil? or params[:visn_id].empty?) && !(params[:facility_id].nil? or params[:facility_id].empty?) && (params[:provider_id].nil? or params[:provider_id].empty?)
        @filtered_ref = query.where('providers.facility_id=?',params[:facility_id])
      elsif !(params[:visn_id].nil? or params[:visn_id].empty?) && (params[:facility_id].nil? or params[:facility_id].empty?) && !(params[:provider_id].nil? or params[:provider_id].empty?)
        @filtered_ref = query.where('referrals.provider_id=?',params[:provider_id])
      elsif !(params[:visn_id].nil? or params[:visn_id].empty?) && (params[:facility_id].nil? or params[:facility_id].empty?) && (params[:provider_id].nil? or params[:provider_id].empty?)
        @filtered_ref = query.where('facilities.visn_id=?',params[:visn_id])
      else
        @filtered_ref = Referral.all
      end
    end

=end


    ########################################################################
    # Provide data for download based upon the last filter action.
    # NOTE: The options hash configures the CSV.generate method

    def to_csv(a_hash, user, options = {})
      an_arel = filter(a_hash, user)

      result = CSV.generate(options) do |csv|
        csv << ["Patient Name", "Referral", "SSN", "Date", "Status", "Coordinator", "Provider", "VISN:Facility"]

        an_arel.each do |referral|
          csv <<  [
                    referral.consultation.veteran_full_name,
                    referral.referral_type.title,
                    referral.consultation.formatted_ssn,
                    referral.created_at.strftime('%F'),
                    referral.referral_status.name,
                    referral.coordinator.name,
                    ProviderDataPresenter.new(referral).name,
                    ProviderDataPresenter.new(referral).facility.try(:visn_region_prepended_name) || 'N/A'
                  ]
        end
      end # result = CSV.generate(options) do |csv|

      return result
    end # def to_csv(options = {})

  end # class << self

end # class Referral < ActiveRecord::Base
