# == Schema Information
#
# Table name: exam_requests
#
#  id                        :integer          not null, primary key
#  xml                       :text
#  claim_id                  :integer
#  error_hash                :text
#  created_at                :datetime
#  updated_at                :datetime
#  site_id                   :integer
#  identifier                :string
#  exam_request_status_id    :integer
#  exam_request_state_id     :integer
#  cancellation_reason       :text
#  cancellation_acknowledged :boolean
#  exam_requester_id         :integer
#  request_date              :datetime
#  request_id                :string
#  special_issues            :string
#  participating_system_name :string
#

class ExamRequest < ActiveRecord::Base
  belongs_to :claim
  belongs_to :site
  belongs_to :exam_request_state
  belongs_to :exam_requester
  has_many :contentions

  has_many :examinations

  serialize :error_hash, Hash

  delegate :patient_name,:patient_ssn,:file_number,
    :email,:phone_number,:benefit_type, :first_name, :last_name, :date_of_birth, to: :claim
  delegate :name, to: :exam_request_state, prefix: :request_status

  scope :request_status, -> (id) {where exam_request_state_id: id}
  scope :routing_location, -> (id) {where site_id: id }
  scope :p_name, -> (name) { ExamRequest.joins(:claim).where("concat(' ', claims.first_name, claims.last_name) ILIKE ?", "%#{name}%") }
  scope :elapsed_days, -> (age) { where("exam_requests.created_at >= ?", age) }
  scope :exam_request_date_range, -> (start_date, end_date) {ExamRequest.where(request_date: start_date.beginning_of_day .. end_date.end_of_day)}

  include ReportCsv

  def exam_state_name
  	self.exam_request_state.name
  end

  def elapsed_days
  	(Time.zone.now.to_date - self.request_date.to_date).to_i
  end

  def build_request_date(params)
    if params[:request_date].present? && params[:request_time].present?
      Time.strptime(params[:request_date] + " " + params[:request_time], "%m/%d/%Y %H:%M")
    else
      nil
    end
  end

  def exams_requested
    contentions.
      map { |cont| cont.dbq_informations }.flatten.
      map { |dbqi| dbqi.title }.join(",")
  end

  def requester_name_and_org
    exam_requester.nil? ? "Not Supplied" : "#{exam_requester.full_name}:#{exam_requester.organization}"
  end

  def total_number_of_exams_pending
    examinations.select { |exam| exam.examination_state.code != 'COMPLETED' }.length
  end

  def requester_info info
    exam_requester.nil? ? "Not Supplied" : "#{exam_requester.send(info)}"
  end
end
