# == Schema Information
#
# Table name: evaluation_specs
#
#  id                       :integer          not null, primary key
#  title                    :text
#  version                  :text
#  spec                     :text
#  created_at               :datetime
#  updated_at               :datetime
#  body_system              :string
#  evaluation_builder_title :string
#  dependent                :string
#  spec_id                  :string
#  active                   :boolean          default(TRUE)
#

class EvaluationSpec < ActiveRecord::Base
  validates_presence_of :title
  validates_presence_of :version
  validates_presence_of :spec

  # there can only be one EvaluationSpec with a given spec_id active at any time
  validates_uniqueness_of :spec_id, conditions: -> { where(active: true) }

  before_validation :clean_spec
  before_validation :update_attributes_from_form

  scope :active_exams, -> { where(active: true) }

  class << self
    def group_by_body_system
      return [] if all.empty?

      #Group sorted EvaluationSpecs by body_system
      body_system_hash = where(active: true).order(title: :asc).group_by{|evaluation_spec| evaluation_spec.body_system}
      body_system_hash["Unclassified"] = body_system_hash.delete nil if body_system_hash[nil]

      #Transform into array of option groups for selection; sort by k
      body_system_hash.map{|k,v| [k,v.map{|spec|
        [spec.title, spec.id, { 'data-keywords' => YAML.load(spec.spec)['form']['keywords'].to_s } ]
      }]}.sort
    end

    # Sync a directory of exam specs with the EvaluationSpecs in the database
    #
    # `body_systems`: an array of strings representing the body_systems you
    #                 want to sync
    # `exams_dir`: A glob expression representing the files you want to sync
    #              with the database. Starts from Rails.root.
    # `outstream`: the stream to write debugging information to. Stdout by
    #              default
    def sync_specs(body_systems=nil, exams_dir="app/exams/*.yml", outstream=STDOUT)
      outstream.write("Updating all evaluation with the what is in the app/exams directory.\n")
      outstream.write("Updating only the following body systems: #{body_systems.join(", ")}.\n") unless body_systems.nil?

      Dir[Rails.root.join(exams_dir)].each do |filename|
        yaml = File.read(filename)
        exam = FML::Form.new(yaml)

        if body_systems.nil? || body_systems.include?(exam.form["bodySystem"])
          evaluation_spec = EvaluationSpec.active_exams.find_by_spec_id(exam.id)
          if evaluation_spec
            # If we found the spec and it hasn't changed, there's nothing to do,
            # move on.
            #
            # We're going to consider a spec equal if the ruby yaml's
            # returned object is equal. This means that if you update ruby
            # versions or yaml gem versions, you might end up updating the
            # specs even though they haven't changed.
            if YAML.load(evaluation_spec.spec) == YAML.load(yaml)
              next
            end

            # Otherwise, deactivate it and recreate it
            evaluation_spec.deactivate!
          end

          outstream.write("Creating Spec from #{filename}\n")

          evaluation_spec = EvaluationSpec.create(spec: yaml)
          if evaluation_spec.errors.count > 0
            outstream.write("Error creating #{YAML.load(evaluation_spec.spec)['form']['title']} (#{File.basename(filename)}):")
            outstream.write("\n\t* #{evaluation_spec.errors.to_a.join("\n\t* ")}")
          end
        else
          outstream.write("Skipping #{filename}\n")
        end
      end
    end
  end

  def inactive
    return !self.active
  end

  def deactivate!
    self.active = false
    self.save
  end

  def ratings_calculator_fields
    form = FML::Form.new(self.spec)
    ratings_calculator_fields = form.fields.collect do |field_name, field|
      {name: field_name, calculator_value: field.attrs["ratingCalculator"]} if field.attrs["ratingCalculator"]
    end.compact
    ratings_calculator_fields += form.fields.collect do |field_name, field|
      field.options.collect do |option|
        {name: field_name, calculator_value: option["ratingCalculator"]} if option["ratingCalculator"]
      end.compact.flatten if field.options
    end.compact.flatten
  end

  def dependent?
    !self.dependent.nil?
  end

  def form
    FML::Form.new(self.spec)
  end

  private

  def clean_spec
    self.spec.gsub! /\r\n?/, "\n" if self.spec
  end

  def update_attributes_from_form
    begin
      form = FML::Form.new(self.spec)
      self.title = form.title
      self.spec_id = form.id
      self.version = form.version
      self.body_system = form.form["bodySystem"] || "Unclassified"
      self.evaluation_builder_title = form.form["evaluationBuilder"] || self.body_system
      self.dependent = form.form["dependent"]
    rescue FML::InvalidSpec => error
      self.errors.add(:base, error.to_s)
      return false
    end
  end
end
