# == Schema Information
#
# Table name: diagnoses
#
#  id                    :integer          not null, primary key
#  name                  :string
#  label                 :string
#  code                  :string
#  diagnosis_modifier_id :integer
#  created_at            :datetime         not null
#  updated_at            :datetime         not null
#  minor_system_id       :integer
#  position              :integer
#

class Diagnosis < ActiveRecord::Base
  include ContentManagerHelper
  has_many :dm_assignments, dependent: :delete_all
  has_many :diagnosis_modifiers, :through => :dm_assignments
  has_many :symp_diag_relations, dependent: :delete_all
  has_many :symptoms, through: :symp_diag_relations
  has_many :dbq_informations, through: :diagnoses_dbq_informations, dependent: :destroy
  has_many :diagnoses_dbq_informations
  belongs_to :minor_system
  validates_presence_of :name
  validates_presence_of :label
  validates_presence_of :code
  validates_uniqueness_of :name

  before_save :set_name
  before_update :set_name

  scope :order_by_position, -> () { includes(dm_assignments: :diagnosis_modifier).order("position ASC" )}
  enum format_type: {yes_no: 0, yes_no_na: 1, statement: 2, note: 3, checkbox: 4}

  FIELDS_FOR_RESULTSET = {
    0 => 'diagnoses.name',
    1 => 'diagnoses.label',
    2 => 'diagnoses.code'
  }

  class << self
    def filter(params={})

      query = Diagnosis.all

      unless params[:search][:value].blank?
        query = filter_using_diagnosis(params, query)
      end

      return query
    end

    def filter_using_diagnosis(cp, q)
      q.where('name ilike :search OR label ilike :search OR code ilike :search', :search => "%#{cp[:search][:value]}%")
    end
  end

  def html_dbq_info
    dbq_information_ids.join(", ")
  end 

  def build_html
    if yes_no?
      @bhtml = yes_no_html
    elsif yes_no_na?
      @bhtml = yes_no_na_html
    elsif statement?
      @bhtml = statement_html
    elsif note?
      @bhtml = note_html
    elsif checkbox?
      @bhtml = checkbox_html.clone
      @bhtml.gsub!('[Diagnosis Code]', self.code)
    end
    html_dbq_info
    @bhtml.gsub!('[Diagnosis Name]', self.name)
    @bhtml.gsub!('[Diagnosis Label]', self.label)
    @bhtml.gsub!('[dbq-assoc]', html_dbq_info)

    # dm_html will concat the html for dm_assignments that are to display on a Yes (diagnosis) selection by the user.
    # ndm_html will concat the html for dm_assignments that are to display on a No (diagnosis) selection by the user.
    dm_html = ""
    ndm_html = ""

    self.dm_assignments.order_by_position.each do |dma|
      # A true negative_show value means the dm_assignment should only be displayed when
      # the user selects No for the diagnosis.
      if dma.negative_show == false
        dm_html += dma.build_html(self.code)
      else
        ndm_html += dma.build_html(self.code)
      end
    end
    @bhtml.gsub!('[HTML for associated diagnosis modifiers]', dm_html.html_safe)
    @bhtml.gsub!('[HTML for associated negative diagnosis modifiers]', ndm_html.html_safe)
    return @bhtml
  end


  def base_html
    return checkbox_html
  end


  def get_dm_assignments(diagnosis_modifier)
    if diagnosis_modifier.present?
      return self.dm_assignments.where(diagnosis_modifier_id: diagnosis_modifier.id)
    else
      return false
    end
  end

  def update_dma_names
    dm_assignments.includes(:diagnosis_modifier).each { |dma| dma.set_name; dma.save }
  end

private

  def checkbox_html
    ' <div id="individual_diagnosis_container-[Diagnosis Name]" class="container diagnosis individual" data-cui-dbq-assoc="[dbq-assoc]">
        <input id="diagnosis-[Diagnosis Name]" name="diagnosis-[Diagnosis Name]" type="checkbox" value = "[Diagnosis Code]" visibility_dependents="diagnosis_modifiers-[Diagnosis Name]" onchange="Set_Dependent_Visibility_Based_On(this);">
        <label  for="diagnosis-[Diagnosis Name]">
  	      [Diagnosis Label]
        </label>
      </div>
      <div id="diagnosis_modifiers-[Diagnosis Name]" class="container lab-results inner small-12 columns hidden">
 	      [HTML for associated diagnosis modifiers]
      </div>'
  end

  def yes_no_html
    return  '<div id="individual_diagnosis_container-' + self.name + '" class="container diagnosis individual" data-cui-dbq-assoc="[dbq-assoc]">'                                +
              '<input   id="diagnosis-' + self.name + '-yes"'                                                                                   +
                      ' type="radio"'                                                                                                           +
                      ' name="diagnosis-' + self.name + '-yes_no"'                                                                              +
                      ' visibility_dependents="diagnosis_modifiers-' + self.name + ',negative_diagnosis_modifiers-' + self.name + '"'           +
                      ' onchange="Set_Dependent_Visibility_Based_On(this);"'                                                                    +
                      ' value="' + self.code + '">Yes '                                                                                         +
              '<input   id="diagnosis-' + self.name + '-no"'                                                                                    +
                      ' type="radio"'                                                                                                           +
                      ' name="diagnosis-' + self.name + '-yes_no"'                                                                              +
                      ' visibility_dependents="diagnosis_modifiers-' + self.name + ',negative_diagnosis_modifiers-' + self.name + '"'           +
                      ' onchange="Set_Dependent_Visibility_Based_On(this);"'                                                                    +
#                      ' value="' + self.code + '">No '                                                                                          +
                      ' value="">No '                                                                                                           +
               ' - ' + self.label                                                                                                               +
              '<div id="diagnosis_modifiers-' + self.name + '" class="container lab-results inner small-12 columns hidden">'                    +
 	              '[HTML for associated diagnosis modifiers]'                                                                                     +
              '</div>'                                                                                                                          +
              '<div id="negative_diagnosis_modifiers-' + self.name + '" class="container lab-results inner small-12 columns hidden">'           +
                '[HTML for associated negative diagnosis modifiers]'                                                                            +
              '</div>'                                                                                                                          +
            '</div>'
  end

  def yes_no_na_html
    '<div id="individual_diagnosis_container-' + self.name + '" class="container diagnosis individual" data-cui-dbq-assoc="[dbq-assoc]">'                                +
      '<input   id="diagnosis-' + self.name + '-yes"'                                                                                   +
              ' type="radio"'                                                                                                           +
              ' name="diagnosis-' + self.name + '-yes_no_na"'                                                                              +
              ' visibility_dependents="na-' + self.name + ',diagnosis_modifiers-' + self.name + ',negative_diagnosis_modifiers-' + self.name + '"'                                                          +
              ' onchange="Set_Dependent_Visibility_Based_On(this);"'                                                                    +
              ' value="' + self.name + '">Yes '                                                                                         +
      '<input   id="diagnosis-' + self.name + '-no"'                                                                                    +
              ' type="radio"'                                                                                                           +
              ' name="diagnosis-' + self.name + '-yes_no_na"'                                                                              +
              ' visibility_dependents="na-' + self.name + ',diagnosis_modifiers-' + self.name + ',negative_diagnosis_modifiers-' + self.name + '"'                                                          +
              ' onchange="Set_Dependent_Visibility_Based_On(this);"'                                                                    +
  #                      ' value="' + self.name + '">No '                                                                                          +
              ' value="No">No '                                                                                                           +
      '<input   id="diagnosis-' + self.name + '-na"'                                                                                    +
              ' type="radio"'                                                                                                           +
              ' name="diagnosis-' + self.name + '-yes_no_na"'                                                                              +
              ' visibility_dependents="na-' + self.name + ',diagnosis_modifiers-' + self.name + ',negative_diagnosis_modifiers-' + self.name + '"'                                                          +
              ' onchange="Set_Dependent_Visibility_Based_On(this);"'                                                                    +
              ' value="NA">NA '                                                                                                           +
       ' - ' + self.label                                                                                                               +
      '<div id="na-' + self.name + '" class="container lab-results inner small-12 columns hidden">'     +
        '<div id="diagnosis_modifier_container-please-explain"'    +
            ' class="container diagnosis_modifier" >'              +
          '<span  id="diagnosis_modifier_label-please-explain"'    +
                ' class="label diagnosis_modifier">'               +
            '<br>Please elaborate:'                                   +
          '</span>'                                               +
          '<textarea id="diagnosis_modifier-na-' + self.name + '-textarea" name="diagnosis_modifier-na-' + self.name + '-textarea" rows="4"></textarea>'  +
        '</div>'      +
      '</div>'        +
      '<div id="diagnosis_modifiers-' + self.name + '" class="container lab-results inner small-12 columns hidden">'     +
        '[HTML for associated diagnosis modifiers]'                                                                                     +
      '</div>'                                                                                                                          +
      '<div id="negative_diagnosis_modifiers-' + self.name + '" class="container lab-results inner small-12 columns hidden">'     +
        '[HTML for associated negative diagnosis modifiers]'                                                                                     +
      '</div>'                                                                                                                          +
    '</div>'
  end

  def statement_html
    '<div id="individual_diagnosis_container-[Diagnosis Name]" class="container diagnosis individual" data-cui-dbq-assoc="[dbq-assoc]">
      <label  for="diagnosis-[Diagnosis Name]">
        [Diagnosis Label]
      </label>
    </div>
    <div id="diagnosis_modifiers-[Diagnosis Name]" class="container lab-results inner small-12 columns">
      [HTML for associated diagnosis modifiers]
    </div>'
  end

  def note_html
    '<div id="individual_diagnosis_container-[Diagnosis Name]" class="container diagnosis individual" data-cui-dbq-assoc="[dbq-assoc]">
      <label  for="diagnosis-[Diagnosis Name]">
        Note: [Diagnosis Label]
      </label>
    </div>
    <div id="diagnosis_modifiers-[Diagnosis Name]" class="container lab-results inner small-12 columns">
      [HTML for associated diagnosis modifiers]
    </div>'
  end

  def set_name
    self.name = clean_name(self.name)
  end

end
