class ReferralDocumentsController < ApplicationController
  include ReferralDocumentsHelper

  before_action :set_referral, only: [:create]
  before_action :set_referral_and_document, only: [:approve, :reject, :delete, :update]
  check_authorization
  authorize_resource

  FILE_UPLOAD_DIR = 'fake_docstore'
  FILE_UPLOAD_PATH = File.join(Rails.root, "public/#{FILE_UPLOAD_DIR}")

  def create
    if (referral_document_params[:document_name].nil? or referral_document_params[:document_name].empty?)
      flash[:alert] = "No Document attached"
      redirect_to referral_path(@referral, tabpage: 'documents')
    else
      move_file_to_uploads_dir
      @referral_document = @referral.referral_documents.new(referral_document_params)
      @referral_document.location = "/#{FILE_UPLOAD_DIR}"
      @referral_document.uploader_id = current_user.id
      if current_user.is_vha_cc?
        @referral_document.approved_at = Time.now
      end
      respond_to do |format|
        if @referral.save
          format.html do
            flash[:notice] = 'Document was successfully created.'
            redirect_to referral_path(@referral, tabpage: 'documents')
          end
          format.json {render json: @referral, status: :created}
        else
          format.html do
            flash[:alert] = @referral.errors.full_messages.join(" ")
            redirect_to referral_path(@referral, tabpage: 'documents')
          end
          format.json {render json: @referral.errors, status: :unprocessable_entity}
        end
      end
    end
  end

  def ajax_deleted_referral_docs_for_user
    ret_data = []

    if params['referral_id']
      referral_id = params['referral_id']
      ret_data = Referral.find(referral_id).deleted_visible_documents_for_referral(current_user)
    elsif params['referral_doc_id']
      referral_doc_id = params['referral_doc_id']
      ret_data = [ReferralDocument.only_deleted.find(referral_doc_id)]
    end

    ret_data = format_delete_ref_doc_data(ret_data)
    render json: ret_data.as_json
  end

  def format_delete_ref_doc_data(ref_docs = [])
    ret_rows = []

    ref_docs.each do |ref_doc|
      row = {}
      row['ref_doc_id'] = ref_doc.id
      row['document_name'] = ref_doc.document_name
      row['document_link'] = link_to ref_doc.document_name, ref_doc.file_link,  target: '_blank'
      row['doc_type_name'] = ref_doc.doc_type_name
      row['uploader_name'] = ref_doc.uploader_name
      row['deleted_by'] = "#{ref_doc.deleted_by}<br>#{formatted_date_simple(ref_doc.deleted_at, current_user)}<br>#{ref_doc.deleted_reason}"
      row['formatted_deleted_at'] = formatted_datetime(ref_doc.deleted_at, current_user)
      ret_rows << row
    end

    ret_rows
  end

  def ajax_delete_referral_doc
    ret_json = {success: true}
    referral_doc_id = params['referral_doc_id']
    reason = params['reason']

    if referral_doc_id && reason
      referral_doc = ReferralDocument.find(referral_doc_id)

      if referral_doc
        referral_doc.deleted_reason = reason
        referral_doc.deleted_by = current_user.name
        referral_doc.save
        # acts as paranoid does a soft delete
        referral_doc.destroy

        # add a corresponding referral_note
        ReferralNote.delete_document_note(Referral.find(referral_doc.referral_id), current_user, referral_doc, reason)
      else
        ret_json[:success] = false
        ret_json[:msg] = 'Invalid referral doument id passed to delete referral document.'
      end
    else
      ret_json[:success] = false
      ret_json[:msg] = 'Invalid arguments passed to delete referral document.'
    end
    render json: ret_json
  end

  def approve
    @referral_document.approved_at = Time.now
    @referral_document.approver_id = current_user.id
    respond_to do |format|
      if @referral_document.save
        format.html do
          notice_text = "Document '#{@referral_document.document_name}' was successfully approved."
          redirect_to @referral, notice: notice_text
        end
        format.json do
          result_html = reviewed_by_field(@referral_document, current_user)
          render json: {reviewed_by: result_html}, status: :ok
        end
      else
        format.html do
          flash[:alert] = @referral_document.errors.full_messages.join(" ")
          redirect_to @referral
        end
        format.json {render json: @referral_document.errors, status: :unprocessable_entity}
      end
    end
  end

  def reject
    destroy(:rejected)
  end

  def delete
    if (@referral_document.uploader_id == current_user.id)
      destroy(:deleted)
    end
  end

  def update
    request_body = JSON.parse request.body.read
    respond_to do |format|
      format.json do
        if @referral_document.update_attributes(request_body)
          render json: @referral_document, status: :no_content
        else
          render json: @referral_document.errors, status: :unprocessable_entity
        end
      end
    end
  end

  private

  def destroy(action_verb)
    @referral_document.destroy
    respond_to do |format|
      format.html do
        flash[:notice] = "Document was successfully #{action_verb}."
        redirect_to referral_path(@referral, tabpage: 'documents')
      end
      format.json {head :no_content}
    end
  end

  # Use callbacks to share common setup or constraints between actions.
  def set_referral
    @referral = Referral.find(params[:referral_id])
  end

  def set_referral_and_document
    set_referral
    if params.has_key?(:id)
      @referral_document = ReferralDocument.find(params[:id])
    elsif params.has_key?(:referral_document_id)
      @referral_document = ReferralDocument.find(params[:referral_document_id])
    else
      raise Exception, "Referral Document ID not found."
      return false
    end
    # check if referral document is contained in referral
    unless @referral.referral_documents.include? @referral_document
      raise Exception, "Invalid Referral Document ##{@referral_document.id} for referral ##{@referral.id}."
      return false
    end
  end

  # Never trust parameters from the scary internet, only allow the white list through.
  def referral_document_params
    params.require(:referral_document).permit(:document_name, :referral_document_type_id)
  end

  def referral_doc_file_from_params
    doc = params[:referral_document][:document]
    params[:referral_document].extract! :document
    doc
  end

  def move_file_to_uploads_dir
    @uploaded_file = referral_doc_file_from_params
    file_name = @uploaded_file.original_filename
    tmpfile = @uploaded_file.tempfile
    tmpfile.close #Unix might let you move an open file, but not windows.
    FileUtils.mv tmpfile, File.join(FILE_UPLOAD_PATH, file_name)
  end

end
