class ExaminationProcessorsController < ApplicationController
  include ContentionHelper
  include ExaminationsHelper
  before_action :set_examination_processor, only: [:show, :edit, :update, :destroy]
  check_authorization
  authorize_resource class: false

  TIME_VALUES = ['00:00', '00:30', '01:00']

  # GET /examination_processors
  # GET /examination_processors.json
  def index
    @examination_processors = ExaminationProcessor.all
  end

  # GET /examination_processors/1
  # GET /examination_processors/1.json
  def show
  end

  # GET /examination_processors/new
  def new
    @examination_processor = ExaminationProcessor.new
  end

  # GET /examination_processors/1/edit
  def edit
  end

  def view_examination
    @examination = Examination.find_by_identifier(params[:identifier])
    @sites = Site.all
    @examination_history = Array(ExaminationHistory.where(examination_id: @examination).order(created_at: :DESC))
    @exam_request = ExamRequest.find_by_id(@examination.exam_request_id)

    if (@examination.examination_state.code == "CANCELLED" && !@examination.acknowledged)
      redirect_to acknowledge_path(:id => @examination.identifier)
    end
  end

  def cancel_examination
    @examination = Examination.find_by_id(params[:e_id])
    if (@examination == nil)
      redirect_to dashboards_path, notice: 'Unable to locate the examination.'
    end
    examinationState = ExaminationState.find_by_code "CANCELLED"
    if examinationState != nil && examinationState.code == "CANCELLED"
      #@examination.examination_state_id = examinationState.id
      @examination.acknowledged = true
      @examination.save

      examRequest = ExamRequest.find_by_id(@examination.exam_request_id)
      notes = current_user.first_name + " " + current_user.last_name + " acknowledged the cancellation of the examination" + " (" + appendContentionNames(@examination.contentions) + ")"
      @exam_request_history = ExamRequestHistory.new(exam_request_id: examRequest.id, notes: notes)
      @exam_request_history.save

      redirect_to dashboards_path, notice: 'Examination successfully cancelled.'
    end
  end

  def internal_cancel_examination
    @examination = Examination.find_by_identifier(params[:id])
    @exam_schedule = @examination.most_recent_active_appointment

     #when the vista codes are missing
     if request.post? && session[:vista].nil? && vista_codes_missing?(params)
       redirect_to get_cancel_examination_path(id: @examination.identifier), :alert => "Vista access, verify and site codes must be provided."
       return
     end

     #all vista codes are provided
     if request.post? && session[:vista].nil? && !vista_codes_missing?(params)
      begin
        session[:vista] = VIA_API.authenticate(params)
      rescue VIA_API::ViaApiAuthenticationError => e
        session.delete(:vista)
        redirect_to get_cancel_examination_path(id: @examination.identifier), :alert => e.message
        return
      end
     end
    
    #vista session token already exists
    if request.post? && session[:vista].present?
      begin
        response_appt = VIA_API::SchedulingService.cancel_appointment(*format_params_for_cancel_appt(params, @examination), session[:vista])
      rescue VIA_API::ViaApiAuthenticationError, VIA_API::ViaApiFaultError => e
        session.delete(:vista)
        redirect_to get_cancel_examination_path(id: @examination.identifier), :alert => e.message
        return
      end
      message = construct_message_from_vista_cancelappt(response_appt.first)

      @exam_schedule.active = false
      @exam_schedule.cancel_remarks = params[:remarks].to_s
      @exam_schedule.cancel_code = params[:cancel_code].to_s
      @exam_schedule.cancel_reason = params[:cancel_reason].to_s
      @exam_schedule.save

      exam_history_notes = "#{current_user.name} canceled on #{Time.now.strftime('%b %e, %Y %H:%M')}. Reason: #{ExaminationSchedule::CANCEL_REASONS[@exam_schedule.cancel_reason]}"
      ExaminationHistory.create(examination_id: @examination.id, notes: exam_history_notes)

      examinationState = ExaminationState.find_by_code "READY_TO_BE_SCHEDULED"
      @examination.examination_state = examinationState
      @examination.save

      redirect_to dashboards_path, :notice => "Canceled appointment successfully.<br><br> #{message}".html_safe
    end

  end

  def ack
    @examination = Examination.find_by_identifier(params[:id])
    if (@examination != nil && @examination.examination_state != nil && @examination.examination_state.code == "CANCELLED")
      @exam_request = ExamRequest.find_by_claim_id(@examination.claim_id)
      @examination_history = Array(ExaminationHistory.where(examination_id: @examination.id).order(created_at: :DESC))
    else
      flash[:alert] = "Unable to locate the examination"
      redirect_to dashboards_path
    end
  end

  def ack_submit
    @examination = Examination.find_by_identifier(params[:e_id])
    if (@examination != nil && @examination.examination_state != nil && @examination.examination_state.code == "CANCELLED")
      examination_schedule = @examination.examination_schedules.flatten.select{|es| es.active == true}
      if (@examination.clinician != nil && examination_schedule[0].exam_date_time != nil)
        @examination.clinician_acknowledged = true
      else
        @examination.acknowledged = true
      end
        @examination.save

      addExaminationHistory(@examination.contentions, @examination)
      flash[:notice] = "Acknowledgement successful"
    else
      flash[:alert] = "Unable to locate the examination"
    end
    redirect_to dashboards_path
  end

  def reschedule
    @examination = Examination.find_by_identifier(params[:id])

    @examination_schedule = ExaminationSchedule.new
    @examination_schedules = @examination.examination_schedules.flatten.select{|ex| ex.active==true}
    if (@examination_schedules != nil && @examination_schedules.count > 0)
      @examination_schedule = @examination_schedules[0]
    end
    @sites = Site.all
    @examination_history = Array(ExaminationHistory.where(examination_id: params[:id]).order(created_at: :DESC))
    @exam_request = ExamRequest.find_by_id(@examination.exam_request_id)
  end

  def reschedule_submit
    @examination = Examination.find_by_identifier(params[:e_id])

    if @examination.nil?
      redirect_to dashboards_path, alert: 'Unable to locate the examination.'
      return
    end
    @examination_schedule = @examination.examination_schedules[0]
    @sites = Site.all
    @examination_history = Array(ExaminationHistory.where(examination_id: params[:id]).order(created_at: :DESC))
    @exam_request = ExamRequest.find_by_id(@examination.exam_request_id)

    if params[:examination_date].present? && params[:examination_date_time].present?
      apptDateAndTime = Time.strptime(params[:examination_date] + " " + params[:examination_date_time], "%m/%d/%Y %H:%M")

      @examination.examination_state = ExaminationState.find_by_code "SCHEDULED"
      @examination.save

      examinationSchedule = ExaminationSchedule.new
      examinationSchedule.exam_date_time = apptDateAndTime
      examinationSchedule.active = true
      examinationSchedule.examination_id = @examination.id

      examinationScheduleExists = false
      @examination_schedule = @examination.examination_schedules[0]
      if (@examination_schedule != nil)
        @examination_schedule = @examination.examination_schedules.flatten.select{|ex| ex.active==true}
        if (@examination_schedule != nil && @examination_schedule.count > 0)
          @examination_schedule[0].active = false
          @examination_schedule[0].save

          examinationScheduleExists = true
        end
      end

      if (examinationScheduleExists)
        examinationSchedule.save
        flash[:notice] = "Examination successfully rescheduled"

        if @examination.clinician
          clinician = User.find(@examination.clinician)
          ExamManagementMailer.examination_scheduled(clinician).deliver_now
        end
        #Send Appointment Scheduled Notification
        appointmentScheduledEventNotification = AppointmentScheduledEvent.new
        eventId = appointmentScheduledEventNotification.build_notification("RESCHEDULE", @exam_request.claim_id, @examination, @examination_schedule[0], @exam_request.site_id)
        VlerDASJob.sendNotification(appointmentScheduledEventNotification, eventId, @exam_request.claim_id, "Appointment Scheduled Event Notification")
      else
        flash[:alert] = "Examination Record not found."
      end
    else
      flash.now[:alert] = "Please provide reschedule day and time."
      render :reschedule
      return
    end

    redirect_to dashboards_path
  end

  def cancelled_examination
    @examination = Examination.find_by_identifier(params[:id])
    if (@examination != nil && @examination.examination_state != nil && @examination.examination_state.code == "CANCELLED")
      @exam_request = ExamRequest.find_by_claim_id(@examination.claim_id)
      @examination_history = Array(ExaminationHistory.where(examination_id: @examination.id).order(created_at: :DESC))
    else
      flash[:alert] = "Unable to locate the examination"
      redirect_to dashboards_path
    end
  end

  def verify_veteran_registration
    @examination = Examination.find_by_identifier(params[:id])
    claim = @examination.claim
    @pid = claim.participant_id
    @patient_name = claim.first_name + " " + claim.last_name

    # MVI retrieve person for registration info
    if params[:commit] == "Retrieve veteran registration"
      unique_request_id = "CUI" + SecureRandom.base64(10)
      begin
        person_data = MVI.retrieve_person @pid, unique_request_id
        session[:person_data] = person_data

        redirect_to veteran_retrieval_path(id: @examination.identifier, pid: @pid, mvi_request_id: unique_request_id) 
      rescue MVI::MviApiFaultError, MVI::MviPersonNotFoundError, MVI::MviApplicationReject, RuntimeError => e
        redirect_to verify_veteran_registration_path(id: @examination.identifier), alert: e.message.titleize
      end
    end
  end

  def veteran_retrieval
    @examination = Examination.find_by_identifier(params[:id])
    @site = @examination.site
    @station_number = @site.site_station_number
    person_data = session[:person_data]
    @pid = @examination.claim.participant_id

    # Get VHA site data
    @reg_list = get_veteran_sites_list(person_data)

    @patient_name = person_data.dig(:patient, :patient_person, :name, :given) + " " + person_data.dig(:patient, :patient_person, :name, :family)

    # Is veteran already registered at VBA-specified site
    @veteran_registered = @reg_list.map {|site| site[2] }.include?(@station_number)

    if params[:commit] =~ /Add Station Number/
      request_id = params[:mvi_request_id]
      icn = @reg_list.select {|site| site[1] =~ /NI/ && site[3] =~ /USVHA/ }.first[0]

      veteran_hash = create_veteran_hash(person_data, request_id, @station_number, icn)
      begin
        ack_message = MVI.proxy_add_to_vista veteran_hash
        redirect_to '/ep/schedule_examination/'+@examination.identifier, notice: ack_message + ". Station number, #{@station_number}, added for #{@patient_name}."
        session.delete(:person_data)
      rescue MVI::MviApplicationError, MVI::MviApplicationReject, RuntimeError => e
        redirect_to veteran_retrieval_path(id: @examination.identifier, person_data: person_data), alert: e.message.titleize
      end
    end
  end

  def schedule_examination
    # redirect to Vista login if vista credentials are not present
    if session[:vista].nil?
      redirect_to vista_path, :alert => "Please login to Vista before scheduling an examination."
      return
    end

   ActiveRecord::Base.transaction do
    @examination = Examination.find_by_identifier(params[:id])

    if @examination.nil?
      redirect_to dashboards_path, notice: 'Unable to locate the examination.'
    end

    @examination_history = Array(ExaminationHistory.where(examination_id: @examination.id).order(created_at: :DESC))
    @exam_request = ExamRequest.find_by_id(@examination.exam_request_id)

    examination_state = ExaminationState.find_by_id(@examination.examination_state_id)
    if examination_state != nil && examination_state.code == ExaminationStateHelper::CANCELLED
      render :cancel_examination
      return
    end

    if params[:examination_date].present? && params[:examination_date_time].present? && valid_date_string(params[:examination_date])
      begin
        appointment_list = VIA_API::SchedulingService.make_appointment(*format_params_for_make_appt(params, @examination), session[:vista])
      rescue Savon::HTTPError, VIA_API::ViaApiFaultError => e
        redirect_to "/ep/schedule_examination/" + @examination.identifier, alert: "Alert: #{e.message}"
        return
      end
      message = construct_message_from_vista_makeappt(appointment_list.first)     

      @examination.examination_state = ExaminationState.find_by_code "SCHEDULED"

      examination_schedule = @examination.examination_schedules.flatten.select{|es| es.active == true}.first
      #if exam was cancelled there won't be any active examination_schedules
      if (examination_schedule.nil? )
        examination_schedule = ExaminationSchedule.create(examination_id: @examination.id, active: true)
        @examination.examination_schedules << examination_schedule
      end
      
      update_exam_schedule(examination_schedule)
      @examination.save

      exam_history_notes = "#{current_user.name} scheduled exam for #{@appt_date_time.strftime('%b %e, %Y %H:%M')} at #{examination_schedule.appt_address_1}/#{examination_schedule.site_station_number}"
      ExaminationHistory.create(examination_id: @examination.id, notes: exam_history_notes)

      if @examination.clinician
        clinician = User.find(@examination.clinician)
        ExamManagementMailer.examination_scheduled(clinician).deliver_now
      end

      appointmentScheduledEventNotification = AppointmentScheduledEvent.new
      eventId = appointmentScheduledEventNotification.build_notification("SCHEDULE", @exam_request.claim_id, @examination, examination_schedule, @exam_request.site_id)
      VlerDASJob.sendNotification(appointmentScheduledEventNotification, eventId, @exam_request.claim_id, "Appointment Scheduled Event Notification")

      redirect_to dashboards_path, notice: "Examination successfully scheduled.<br><br> #{message}".html_safe
    
    elsif request.post?
      flash.now[:alert] = "Please provide a day/date and time."
      render :schedule_examination
    end
   end
  end


  def review_examination
    @examination = Examination.find_by_id(params[:id])
    @contentions = @examination.contentions
    if params[:examination_reviewed]
      @examination.examination_state = ExaminationState.find_by_code "REVIEWED"
      @examination.reviewed_by = current_user.id
      @examination.reviewed_at = DateTime.now
      @examination.save
      redirect_to dashboards_path, notice: 'Examination successfully marked as Reviewed.'
    elsif params[:examination_rejected]
      @examination.examination_state = ExaminationState.find_by_code "REJECTED"
      @examination.save
      redirect_to dashboards_path, notice: 'Examination has been rejected and will be sent back to the clinician to update.'
    elsif params[:add_note]
      message = params[:Note]
      from_supervisor = current_user.id
      to_clinician = @examination.clinician
      note = ExaminationNote.create(from_id: from_supervisor, to_id: to_clinician, note: message)
      @examination.examination_notes << note
      @examination.save
    end
  end


  def modify
  end

  def create
    flash.clear

    @examination = Examination.find_by_id(params[:e_id])
    @sites = Site.all.order(:name)
    #retrieve the examination history based on the examination_id
    @examination_history = Array(ExaminationHistory.where(examination_id: params[:e_id]).order(created_at: :DESC))

    if params[:examination_confirm]
      @examination_from_db = Examination.find_by_id(params[:e_id])

      #check to see if clinician has been assigned to the examination
      if @examination_from_db.clinician != nil
        flash[:notice] = "Examination state has been changed to 'Ready To Be Scheduled'."

        #update examination status to READY_TO_BE_SCHEDULED
        @examinationState = ExaminationState.find_by_code "READY_TO_BE_SCHEDULED"
        @examination.examination_state_id = @examinationState.id
        @examination.examination_state_start_date = DateTime.current
        @examination.save
        site_scheduling_users = User.all.includes(:sites).where(:sites => {name: @examination.site.name}).includes(:site_role_sets).where(:site_role_sets => {scheduling: "true"})

        # TODO:Refactor
        if site_scheduling_users.present?
          site_scheduling_users.each do |ss|
            ExamManagementMailer.examination_ready_to_schedule(ss).deliver_now
          end
        end

        redirect_to dashboards_path
      else
        flash[:alert] = "Examination has to be assigned to a clinician."
        render :view_examination
      end
    elsif params[:modify_submit]
      @examination_from_db = Examination.find_by_id(params[:e_id])
      # Check to see if a new site was picked.
      if @examination_from_db.site_id.to_s != params[:site_id]
        new_site = Site.find_by_id(params[:site_id])
        prev_site = Site.find_by_id(@examination_from_db.site_id)

        notes = "#{current_user.name} changed the site from #{prev_site.name} to #{new_site.name}."
        examination_history = ExaminationHistory.new(examination_id: @examination_from_db.id, notes: notes)
        examination_history.save

        #update the examination and save
        @examination_from_db.site_id = params[:site_id]
        @examination_from_db.assign_a_clinician(nil) #cs 2016-09-13
        @examination_from_db.save

        flash[:notice] = "Examination has been successfully assigned to #{new_site.name}."
      end
      @examination_history = Array(ExaminationHistory.where(examination_id: params[:e_id]).order(created_at: :DESC))
      render :view_examination

    elsif params[:modify_cancel]
      render :view_examination
    elsif params[:change_site]
      render :change_site
    else
      redirect_to dashboards_path
    end
  end

  def change_site
  end

  # POST /examination_processors
  # POST /examination_processors.json
  def create_old
    @examination_processor = ExaminationProcessor.new(examination_processor_params)

    respond_to do |format|
      if @examination_processor.save
        format.html { redirect_to @examination_processor, notice: 'Examination processor was successfully created.' }
        format.json { render action: 'show', status: :created, location: @examination_processor }
      else
        format.html { render action: 'new' }
        format.json { render json: @examination_processor.errors, status: :unprocessable_entity }
      end
    end
  end

  # PATCH/PUT /examination_processors/1
  # PATCH/PUT /examination_processors/1.json
  def update
    respond_to do |format|
      if @examination_processor.update(examination_processor_params)
        format.html { redirect_to @examination_processor, notice: 'Examination processor was successfully updated.' }
        format.json { head :no_content }
      else
        format.html { render action: 'edit' }
        format.json { render json: @examination_processor.errors, status: :unprocessable_entity }
      end
    end
  end

  # DELETE /examination_processors/1
  # DELETE /examination_processors/1.json
  def destroy
    @examination_processor.destroy
    respond_to do |format|
      format.html { redirect_to examination_processors_url }
      format.json { head :no_content }
    end
  end

  private
    def create_veteran_hash person_data, request_id, station_number, icn
      veteran_hash = {}
      veteran_hash[:retrieve_person_request_id] = request_id
      veteran_hash[:station_number] = station_number
      veteran_hash[:patient_icn] = icn
      veteran_hash[:fname] = person_data.dig(:patient, :patient_person, :name, :given)
      veteran_hash[:lname] = person_data.dig(:patient, :patient_person, :name, :family)
      # patient's phone not provided in 1306 sample responses
      veteran_hash[:telecom_value] = ""
      veteran_hash[:telecom_usage] = ""
      veteran_hash[:gender] = person_data.dig(:patient, :patient_person, :admininstrative_gender_code, :code)
      veteran_hash[:birth_date] = person_data.dig(:patient, :patient_person, :birth_time)
      veteran_hash[:addr_usage] = person_data.dig(:patient, :patient_person, :addr).delete(:use)
      veteran_hash[:addr] = person_data.dig(:patient, :patient_person, :addr)
      veteran_hash[:ssn] = get_veteran_ssn(person_data.dig(:patient, :patient_person, :as_other_i_ds))

      veteran_hash[:clinic_name] = @site.name
      # don't have site contact number
      veteran_hash[:cinic_phone] = ""

      veteran_hash
    end

    def get_veteran_ssn as_other_ids
      if as_other_ids.is_a? Array
        ssn_id_hash = as_other_ids.select {|ids| ids[:classCode] == "SSN"}
        ssn = ssn_id_hash.dig(:id, :extension)
      elsif as_other_ids.is_a? Hash
        ssn = ( as_other_ids.dig(:classCode) == "SSN" ? as_other_ids.dig(:id, :extension) : "" )
      end

      ssn
    end
    # Use callbacks to share common setup or constraints between actions.
    def set_examination_processor
      @examination_processor = ExaminationProcessor.find(params[:id])
    end

    # Never trust parameters from the scary internet, only allow the white list through.
    def examination_processor_params
      params[:examination_processor]
    end

    def retrieve_examination_history(examination_id)
      result = Array(ExaminationHistory.where(examination_id: params[:e_id]).order(created_at: :DESC))
      return result
    end

    def addExaminationHistory(contentions, examination)
      notes = current_user.first_name + " " + current_user.last_name + " acknowledged the cancellation of examination for "
      contentions.each_with_index do |contention, index|
        separator = ""
        if index > 0 && index == contentions.length-1
          separator = " and "
        elsif index > 0
          separator = ", "
        else
          separator = " "
        end
        notes = notes + separator + contention.contention_details[0].name
      end
      @examination_history = ExaminationHistory.new(examination_id: examination.id, notes: notes)
      @examination_history.save
    end

    def format_params_for_make_appt params, exam
      @site = exam.site
      @notes = params[:book_notes].to_s
      patient = Struct.new(:mpi_pid, :local_pid).new(nil, @examination.claim.participant_id)
      site_station_number = @site.site_station_number
      appointment_type = "COMPENSATION &amp; PENSION"
      time_string = params[:examination_date] + " " + params[:examination_date_time]
      time_format = "%m/%d/%Y %H:%M"
      @appt_date_time = datetime_string_to_time_with_zone(time_string, time_format, current_user)
      return [patient, site_station_number, appointment_type, @appt_date_time, @notes]
    end

    def format_params_for_cancel_appt params, exam
      pid = exam.claim.participant_id
      patient = Struct.new("Patient", :local_pid).new(pid)
      timestamp = @exam_schedule.exam_date_time.strftime("%Y%m%d.%H%M")
      cancel_code = params[:cancel_code].to_s
      cancel_reason = params[:cancel_reason].to_s
      remarks = params[:remarks].to_s
      site_station_number = @exam_schedule.site_station_number
      return [timestamp, patient, site_station_number, cancel_code, cancel_reason, remarks]
    end

    def construct_message_from_vista_cancelappt(response_appt)
      time = Time.strptime(response_appt.timestamp, "%Y%m%d.%H").strftime("%a, %b %d, %Y %I:%M %p")
          
      str = "<ul>"
      str += "<li>Purpose: #{ExaminationSchedule::CANCEL_CODES[response_appt.purpose]}</li>"
      str += "<li>Reason: #{ExaminationSchedule::CANCEL_REASONS[response_appt.reason]}</li>"
      str += "<li>Timestamp: #{time}</li>"
      str += "<li>Site Station Number: #{response_appt.provider.id}</li>"
      str += "</ul>"
    end

    def construct_message_from_vista_makeappt(response_appt)
      time = Time.strptime(response_appt.timestamp, "%Y%m%d.%H").strftime("%a, %b %d, %Y %I:%M %p")
          
      str = "<ul>"
      str += "<li>Timestamp: #{time}</li>"
      str += "<li>Appointment Type: #{response_appt.appointment_type}</li>"
      str += "<li>Appointment Length: #{response_appt.appointment_length}</li>"
      str += "<li>Purpose: #{response_appt.purpose}</li>"
      str += "<li>Book Notes: #{response_appt.book_notes}</li>"
      str += "<li>Site Station Number: #{response_appt.provider.id}</li>"
      str += "</ul>"
    end

    def update_exam_schedule exam_sched
      exam_sched.appt_address_1 = @site.name
      exam_sched.appt_address_2 = @site.address
      exam_sched.appt_address_city = @site.city
      exam_sched.appt_address_state = @site.state
      exam_sched.appt_address_zipcode = @site.zip_code
      exam_sched.exam_date_time = @appt_date_time
      exam_sched.book_notes = @notes
      exam_sched.site_station_number = @site.site_station_number
      exam_sched.active = true
      exam_sched.save
    end

    def vista_codes_missing? params
      !params[:access_code].present? || !params[:verify_code].present? || !params[:site_code].present?
    end

    def get_veteran_sites_list person_data
      site_registration_list = person_data.dig(:patient, :id).select { |site| site[:@extension] =~ /USVHA/ }.map {|i| i[:@extension] }
      reg_list = []
      site_registration_list.each do |site_str|
        reg_list.push(site_str.split("^"))
      end
      reg_list
    end
end
