class EvaluationsController < ApplicationController
  before_filter :assign_claim
  before_filter :assign_evaluation, except: [:new, :create]
  before_filter :saved_redirect, only: [:edit, :update]
  before_filter :assign_claim_evaluations, only: [:edit, :update, :autosave]
  skip_before_action :clear_current_site

  check_authorization
  authorize_resource

  def new
    @contention = @claim.contentions.find(params[:contention_id])
    @evaluation = @claim.evaluations.new
    @existing_evaluations = @claim.evaluations - @contention.evaluations
    @users = User.examiners.map{ |u| [u.last_name_first_name, u.id] }
    authorize! :show, @contention
  end

  def create
    contention_id = params[:evaluation][:contention_id]
    @evaluation = @claim.evaluations.build(evaluation_params)
    @evaluation.assigner = current_user

    if (@evaluation.evaluation_spec && @evaluation.evaluation_spec.title != 'Full-Body Evaluation' && @evaluation.user == nil)
      #check for Owner selection
      flash[:alert] = "Error adding evaluation to contention: User Assignee examiner can't be blank"
      redirect_to new_claim_evaluation_path(@claim, contention_id: contention_id)
    elsif @evaluation.save
      @claim.contentions.find(contention_id).evaluations << @evaluation
      flash[:notice] = "Added evaluation to contention"
      redirect_to claim_contention_path(@claim, contention_id)
    else
      flash[:alert] = "Error adding evaluation to contention: #{@evaluation.errors.full_messages.join(", ")}"
      redirect_to new_claim_evaluation_path(@claim, contention_id: contention_id)
    end
  end

  def show
    @form = @evaluation.doc.nil? ? FML::Form.new(@evaluation.evaluation_spec.spec) : FML::Form.from_json(@evaluation.doc.to_json)
    respond_to do |format|
      format.html
      format.xml do
        render :xml => @evaluation.to_xml
      end
      format.pdf do
        params[:show_all].present? ? @show_all = true : @show_all = false
        render :pdf => "cui-#{@evaluation.guid}.pdf",
               :layout => "pdf.html",
               :locals => {claim: @claim, form: @form, evaluation: @evaluation}
      end
    end
  end

  def edit
    @form = @evaluation.doc.nil? ? FML::Form.new(@evaluation.evaluation_spec.spec) : FML::Form.from_json(@evaluation.doc.to_json)
    @suggested_exams = get_suggested_exams(@form)
    flash[:evaluation] = "Evaluation"
  end

  def update
    if params[:commit] == "Review"
      begin
        authorize! :review, @evaluation

        # Add suggested exams
        ids = params.select { |k,v| k.start_with? "_suggested_" }
                    .map { |k,v| k.split("_")[2] }
                    .each do |spec_id|
          id = EvaluationSpec.find_by_spec_id(spec_id).id
          evaluation = @claim.evaluations.create(
            evaluation_spec_id: id,
            assigner: current_user,
            user: current_user)
          contention = Evaluation.find(params["id"]).contentions.first
          contention.evaluations << evaluation
        end

        @form = update_evaluation_and_create_form(@evaluation)
        @form.validate
        @evaluation.check_dependencies
        @evaluation.review!

        flash[:notice] = "Please review your application, check for errors, and hit complete when you are satisfied."
        redirect_to review_claim_evaluation_path(@claim, @evaluation)
      rescue FML::ValidationErrors => errors
        @errors = errors
        @suggested_exams = get_suggested_exams(@form)
        flash.now[:alert] = "This form contains the following errors"
        render :edit
      rescue DependencyError => errors
        @errors = errors
        @suggested_exams = get_suggested_exams(@form)
        flash.now[:alert] = errors.errors[0].message
        render :edit
      rescue CredentialError
        # we only care about this exception on submissions
        flash[:notice] = "Please review your application, check for errors, and hit complete when you are satisfied."
        redirect_to review_claim_evaluation_path(@claim, @evaluation)
      end
    elsif params[:commit] == "Complete"
      authorize! :complete, @evaluation
      @evaluation.update_attributes(evaluation_params)
      begin
        @evaluation.complete!(current_user)
      rescue DependencyError => errors
        @form = @evaluation.form
        @suggested_exams = get_suggested_exams(@form)
        @errors = errors
        flash.now[:alert] = errors.errors[0].message
        return render :edit
      rescue CredentialError => error
        @form = @evaluation.form
        @suggested_exams = get_suggested_exams(@form)
        flash.now[:alert] = error.message
        @errors = errors
        return render :review
      end
      flash[:notice] = "Evaluation completed"
      if @evaluation.evaluation_spec.title == "Full-Body Evaluation"
       #start
       @exam = @evaluation.examination
       @exam.examination_state = ExaminationState.find_by_code("COMPLETED")
       @exam.examination_state_start_date = DateTime.current
       @exam.save
       # Send Appointment completed event
       @exam_request = ExamRequest.find_by_id(@exam.exam_request_id)
       appointmentCompletedEventNotification = AppointmentCompletedEvent.new
       eventId = appointmentCompletedEventNotification.build_notification(@exam_request.claim_id, @exam.contention_id, @exam_request.site_id)
       VlerDASJob.sendNotification(appointmentCompletedEventNotification, eventId, @exam_request.claim_id, "Appointment Completed Event Notification")

       flash[:notice] = "Examination is now ready for review by supervising clinician."
       #end
       redirect_to edit_examination_path(@evaluation.examination), notice: 'Evaluation is complete.' #cs 2016-09-21
      else
        redirect_to @claim
      end
    else
      @form = update_evaluation_and_create_form(@evaluation)
      @suggested_exams = get_suggested_exams(@form)
      flash.now[:notice] = "Saved as draft"
      render :edit
    end
  end

  def destroy
    if @evaluation.destroy
      flash[:notice] = "Successfully deleted evaluation"
      redirect_to claim_path(@claim)
    else
      flash[:error] = "Error deleting evaluation"
      redirect_to :back
    end
  end

  def autosave
    @form = update_evaluation_and_create_form(@evaluation)
  end

  def review
#cs     @form = update_evaluation_and_create_form(@evaluation)  #cs 2016-09-12
    @form = FML::Form.from_json(@evaluation.doc.to_json)
  end

  def activity
    @logs = @evaluation.evaluation_logs
  end

  private

  def get_suggested_exams(form)
    return [] if !form.attrs.key?("suggestedExams") || form.attrs["suggestedExams"].length == 0

    form.attrs["suggestedExams"].map do |spec_id|
      EvaluationSpec.where(spec_id: spec_id.strip, active: true).first
    end.reject(&:nil?).map { |e| [e.title, e.spec_id] }
  end

  def update_evaluation_and_create_form(evaluation)
    form = FML::Form.new(evaluation.evaluation_spec.spec)
    gather_params(form, params)
    form.fill(params)
    evaluation.update_attributes(doc: form.to_json,  primary_evaluation_id:  params[:primary_evaluation_id])
    form
    return form
  end

  def assign_evaluation
    @evaluation = @claim.evaluations.find(params[:id])
  end

  def assign_claim_evaluations
    @claim_evaluations = (@evaluation.claim.evaluations - [@evaluation]).sort_by { |e| e.evaluation_spec.title}
  end

  def saved_redirect
    return unless @evaluation.completed?
    flash[:notice] = "Evaluation is already completed"
    redirect_to claim_evaluation_path(@claim, @evaluation)
    return false
  end

  def evaluation_params
    params.require(:evaluation).permit(:evaluation_spec_id, :vha_user_vista_access_code, :vha_user_vista_verify_code,
                                       :vha_user_electronic_signature, :user_id)
  end

  def gather_params(form, params)
    form.fields.each do |name,field|
      if field.type == "partialdate"
        params[field.name] = YAML.dump({
          "day"      => params["#{name}_day"],
          "month"    => params["#{name}_month"],
          "year"     => params["#{name}_year"],
          "unknown"  => params["#{name}_unknown"],
          "estimate" => params["#{name}_estimate"],
        })
      end
    end
  end
end
