# == Schema Information
#
# Table name: referral_appointments
#
#  id               :integer          not null, primary key
#  content          :json
#  created_at       :datetime         not null
#  updated_at       :datetime         not null
#  referral_id      :integer
#  appointment_time :datetime
#  added_to_cprs_at :datetime
#  added_to_cprs_id :integer
#

require 'rails_helper'

RSpec.describe ReferralAppointment, :type => :model do

  context "Associations" do
    it "should have one referral", cpp: true do
	    referral_appointment = ReferralAppointment.reflect_on_association(:referral)
	    expect(referral_appointment.macro).to eq :belongs_to
    end

    it "should have one added_to_cprs_user", cpp: true do
	    referral_appointment = ReferralAppointment.reflect_on_association(:added_to_cprs_user)
	    expect(referral_appointment.macro).to eq :belongs_to
    end
  end

  context "Validations" do

    let(:user) {create(:vha_user, first_name: "John", last_name: "Doe")}
    let(:content) { {scheduled_by: "John Doe", appointment_type: "test appointment type"}}
    let(:referral) {create(:referral, referral_type: ReferralType.where(title: 'MD Office Visit').first)}
    let(:referral_appointment_added_to_cprs) {create(:referral_appointment_added_to_cprs, referral: referral,
                                                     content: content, added_to_cprs_id: user.id)}

    it "appointment_type field should present", cpp: true do
      is_expected.to validate_presence_of(:appointment_type)
    end

    it "appointment_time field should present", cpp: true do
      is_expected.to validate_presence_of(:appointment_time)
    end

    it "should_not_have_veteran_appointment_conflicts", cpp: true do
      appointment = referral_appointment_added_to_cprs
      appointment.veteran_id = 1
      appointment.save
      same_time = appointment.dup
      saved = same_time.save
      expect(saved).to eq false
      ten_before = appointment.dup
      ten_before.appointment_time = ten_before.appointment_time - 10.minutes
      saved = ten_before.save
      expect(saved).to eq false
      ten_after = appointment.dup
      ten_after.appointment_time = ten_after.appointment_time + 10.minutes
      saved = ten_after.save
      expect(saved).to eq false
      [same_time, ten_before, ten_after].each do |ref|
        expect(ref.errors.messages.map { |e| e.last }.flatten.include?(ReferralAppointment::VET_TIME_CONFLICT_MESSAGE)).to eq true
      end
    end
  end

  context "Methods" do

    let(:user) {create(:vha_user, first_name: "John", last_name: "Doe")}
    let(:content) { {scheduled_by: "John Doe", appointment_type: "test appointment type"}}
    let(:referral) {create(:referral, referral_type: ReferralType.where(title: 'MD Office Visit').first)}
    let(:referral_appointment_added_to_cprs) {create(:referral_appointment_added_to_cprs, referral: referral,
                                                     content: content, added_to_cprs_id: user.id)}
    let(:referral_appointment_not_added_to_cprs) {create(:referral_appointment, added_to_cprs_at: nil)}
    let!(:referral_appointment_needs_reminder) {create(:referral_appointment,
                                                      appointment_time: 2.days.ago.beginning_of_day)}


    it "responds to added_to_cprs?", cpp: true do
      expect(referral_appointment_added_to_cprs.added_to_cprs?).to eq true
      expect(referral_appointment_not_added_to_cprs.added_to_cprs?).to eq false
    end

    it "should return the added_to_cprs_username", cpp: true do
      expect(referral_appointment_added_to_cprs.added_to_cprs_username).to eq "John Doe"
      expect(referral_appointment_not_added_to_cprs.added_to_cprs_username).to eq nil
    end

    it "should not be a vista appointment", cpp: true do
      expect(referral_appointment_added_to_cprs.vista_appointment?).to eq false
      expect(referral_appointment_not_added_to_cprs.vista_appointment?).to eq false
    end

    it "should return appointment details", cpp: true do
      expect(referral_appointment_added_to_cprs.details).to match_array ["John Doe", "MD Office Visit"]
    end

    it "should be able to set reminders", cpp: true do
      needs_reminder = ReferralAppointment.needs_reminder.first
      expect(needs_reminder.appointment_time.utc.to_i).to eq (2.days.ago.beginning_of_day).utc.to_i
    end

  end
end
