# == Schema Information
#
# Table name: consultations
#
#  id                     :integer          not null, primary key
#  consultation_number    :string           not null
#  ordering_provider_id   :integer
#  valid_from             :datetime
#  valid_to               :datetime
#  content                :json
#  created_at             :datetime         not null
#  updated_at             :datetime         not null
#  care_category_id       :integer
#  consultation_order_id  :integer
#  consultation_status_id :integer
#  consultation_type_id   :integer
#  veteran_id             :integer
#

require 'rails_helper'

RSpec.describe Consultation, :type => :model do

  context "when creating associations" do

    before :context do
      @veteran = create(:veteran)
    end

    it "should have an associated veteran", cpp: true do
      consultation = create(:consultation, veteran: @veteran)
      expect(consultation.veteran(nil).full_name).to eq @veteran.full_name
    end

    it "should display consultation status name", cpp: true do
      consultation = create(:consultation, :open_consultation, veteran: @veteran)
      expect(consultation.status_name).to eq 'Open'

      consultation = create(:consultation, :complete_consultation, veteran: @veteran)
      expect(consultation.status_name).to eq 'Complete'
    end

  end # context "when creating associations"


  context "when validating consultations" do

    it "should not create multiple consultations with the same consultation number", cpp: true do
      create(:consultation, consultation_number: 'TEST-12345')
      expect {
        create(:consultation, consultation_number: 'TEST-12345')
      }.to raise_error(ActiveRecord::RecordInvalid)
    end

    it "should not create consultations with blank consultation numbers", cpp: true do
      expect {
        create(:consultation, consultation_number: nil)
      }.to raise_error(ActiveRecord::RecordInvalid)
    end

    it "should not create consultations without a veteran", cpp: true do
      expect {
        create(:consultation, veteran: nil)
      }.to raise_error(ActiveRecord::RecordInvalid)
    end

    it "should not create consultations without a care category", cpp: true do
      expect {
        create(:consultation, care_category: nil)
      }.to raise_error(ActiveRecord::RecordInvalid)
    end

    it "should not create consultations without valid to/from dates", cpp: true do
      expect {
        create(:consultation, valid_from: nil)
      }.to raise_error(ActiveRecord::RecordInvalid)

      expect {
        create(:consultation, valid_to: nil)
      }.to raise_error(ActiveRecord::RecordInvalid)
    end

  end # context "when validating consultations"


  context "when performing filtering" do

    # creating test data used for filtering tests
    before :context do
      @test_visn = Visn.where(name: 'VA Southeast Network').first
      @test_care_category = CareCategory.where(title: 'Echocardiogram').first
      @test_vha_user = create(:vha_user)
      @default_veteran = create(:veteran)
      @veteran_with_testname = create(:veteran, first_name: 'Homer', last_name: 'Simpson')
      @veteran_with_ssn = create(:veteran, first_name: 'TestSSN', last_name: 'UserSSN', ssn: '333669999')
      @consultations = {
        with_testname:         create(:consultation, veteran: @veteran_with_testname),
        with_test_ssn:         create(:consultation, veteran: @veteran_with_ssn),
        with_care_category:    create(:consultation, :with_care_category,
                                      veteran: @default_veteran, care_category_title: @test_care_category.title),
        with_visn:             create(:consultation, :with_referral_having_visn,
                                      veteran: @default_veteran, visn: @test_visn),
        with_consult_number:   create(:consultation, veteran: @default_veteran, consultation_number: 'ABC-123'),
        with_auth_number:      create(:consultation, :with_referral_having_authorization_number,
                                      veteran: @default_veteran, authorization_number: 'ABC-123-456'),
        invalid_date_range:    create(:consultation, veteran: @default_veteran, valid_from: 1.year.ago, valid_to: 9.months.ago),
        with_coordinator:      create(:consultation, :with_referral_having_coordinator,
                                      veteran: @default_veteran, coordinator: @test_vha_user),
        open_consultation:     create(:consultation, :open_consultation, veteran: @default_veteran),
        complete_consultation: create(:consultation, :complete_consultation, veteran: @default_veteran)
      }
    end

    it "should return all consultations in collection when no filter params are passed in", cpp: true do
      results = Consultation.filter
      expect(results.count).to be >= @consultations.values.count
      @consultations.values.each do |created_consultation|
        expect(results).to include(created_consultation)
      end
    end

    it "should filter based on veteran name", cpp: true do
      results = Consultation.filter('first_name' => @veteran_with_testname.first_name)
      expect(results.count).to eq 1
      expect(results.first.veteran(nil)).to eq @veteran_with_testname
    end

    it "should return no records when filtering on name not in database", cpp: true do
      results = Consultation.filter('first_name' => 'asdfrgasgasghadsfghafsg')
      expect(results).to be_empty
    end

    it "should filter based on veteran SSN", cpp: true do
      results = Consultation.filter('ssn' => '333669999')
      expect(results.count).to eq 1
      expect(results.first.veteran(nil).first_name).to eq 'TestSSN'
      expect(results.first.veteran(nil).ssn).to eq '333669999'
    end

    it "should return no records when filtering on SSN not in database", cpp: true do
      results = Consultation.filter('ssn' => '333668888')
      expect(results).to be_empty
    end

    it "should filter based on authorization number", cpp: true do
      results = Consultation.filter('authorization_number' => 'ABC-123-456')
      expect(results.count).to eq 1
      expect(results.first).to eq @consultations[:with_auth_number]
    end

    it "should return no records when filtering on authorization number not in database", cpp: true do
      results = Consultation.filter('authorization_number' => 'NOT-VALID-AUTHNUM')
      expect(results).to be_empty
    end

    it "should filter based on consultation number", cpp: true do
      results = Consultation.filter('consultation_number' => 'ABC-123')
      expect(results.count).to eq 1
      expect(results.first).to eq @consultations[:with_consult_number]
    end

    it "should return no records when filtering on consultation number not in database", cpp: true do
      results = Consultation.filter('consultation_number' => 'NOT-VALID-CONSULT_NUM')
      expect(results).to be_empty
    end

    it "should filter based on VISN id", cpp: true do
      results = Consultation.filter('visn_id' => @test_visn.id.to_s)
      expect(results.count).to eq 1
      expect(results.first).to eq @consultations[:with_visn]
    end

    it "should return no records when filtering on VISN without consultations", cpp: true do
      visn_not_found = Visn.where(name: 'Sierra Pacific Network').first
      expect(visn_not_found).not_to be_nil

      results = Consultation.filter('visn_id' => visn_not_found.id.to_s)
      expect(results).to be_empty
    end

    it "should filter based on care category id", cpp: true do
      results = Consultation.filter('care_category_id' => @test_care_category.id.to_s)
      expect(results.count).to eq 1
      expect(results.first).to eq @consultations[:with_care_category]
    end

    it "should return no records when filtering on care category id not found", cpp: true do
      care_category_not_found = CareCategory.where(title: 'Diabetes').first
      expect(care_category_not_found).not_to be_nil

      results = Consultation.filter('care_category_id' => care_category_not_found.id.to_s)
      expect(results).to be_empty
    end

    it "should filter based on coordinator id", cpp: true do
      results = Consultation.filter('coordinator_id' => @test_vha_user.id.to_s)
      expect(results.count).to eq 1
      expect(results.first).to eq @consultations[:with_coordinator]
    end

    it "should return no records when filtering on coordinator not in database", cpp: true do
      not_a_coordinator = create(:examiner)
      results = Consultation.filter('coordinator_id' => not_a_coordinator.id.to_s)
      expect(results).to be_empty
    end

    it "should filter based on a single selected status", cpp: true do
      open_status = ConsultationStatus.where(title: 'Open').first
      open_results = Consultation.filter('status' => [open_status.id])
      expect(open_results.count).to eq 1
      expect(open_results.first).to eq @consultations[:open_consultation]
      expect(open_results).not_to include(@consultations[:complete_consultation])

      complete_status = ConsultationStatus.where(title: 'Complete').first
      complete_results = Consultation.filter('status' => [complete_status.id])
      expect(complete_results.count).to eq 1
      expect(complete_results.first).to eq @consultations[:complete_consultation]
      expect(complete_results).not_to include(@consultations[:open_consultation])
    end

    it "should filter based on multiple statuses by returning consultations with any of the included statuses", cpp: true do
      open_status = ConsultationStatus.where(title: 'Open').first
      complete_status = ConsultationStatus.where(title: 'Complete').first
      results = Consultation.filter('status' => [open_status.id, complete_status.id])
      expect(results.count).to eq 2
      expect(results).to include(@consultations[:open_consultation])
      expect(results).to include(@consultations[:complete_consultation])
    end


    context "including a record with all filter fields" do

      # creating test data used for filtering tests
      before :example do
        @consultation_with_all_fields =
          create(:consultation, :with_care_category, :open_consultation,
                 :with_single_referral_having_multiple_fields,
                 veteran: @veteran_with_ssn,
                 consultation_number: 'ABC-999',
                 authorization_number: 'ABC-123-999',
                 care_category_title: @test_care_category.title,
                 visn: @test_visn,
                 coordinator: @test_vha_user)
      end

      it "should filter based on veteran name", cpp: true do
        results = Consultation.filter('first_name' => @veteran_with_ssn.first_name)
        expect(results.count).to eq 2
        expect(results).to include(@consultation_with_all_fields)
        expect(results).to include(@consultations[:with_test_ssn])
      end

      it "should filter based on veteran SSN", cpp: true do
        results = Consultation.filter('ssn' => '333669999')
        expect(results.count).to eq 2
        expect(results).to include(@consultation_with_all_fields)
        expect(results).to include(@consultations[:with_test_ssn])
      end

      it "should filter based on VISN id", cpp: true do
        results = Consultation.filter('visn_id' => @test_visn.id.to_s)
        expect(results.count).to eq 2
        expect(results).to include(@consultation_with_all_fields)
        expect(results).to include(@consultations[:with_visn])
      end

      it "should filter based on care category id", cpp: true do
        results = Consultation.filter('care_category_id' => @test_care_category.id.to_s)
        expect(results.count).to eq 2
        expect(results).to include(@consultation_with_all_fields)
        expect(results).to include(@consultations[:with_care_category])
      end

      it "should filter based on coordinator id", cpp: true do
        results = Consultation.filter('coordinator_id' => @test_vha_user.id.to_s)
        expect(results.count).to eq 2
        expect(results).to include(@consultation_with_all_fields)
        expect(results).to include(@consultations[:with_coordinator])
      end

      it "should filter based on all filterable consultation/veteran record fields plus coordinator", cpp: true do
        results = Consultation.filter(
          'first_name' => @veteran_with_ssn.first_name,
          'last_name' => @veteran_with_ssn.last_name,
          'ssn' => @veteran_with_ssn.ssn,
          'visn_id' => @test_visn.id.to_s,
          'care_category_id' => @test_care_category.id.to_s,
          'coordinator_id' => @test_vha_user.id.to_s,
          'consultation_number' => 'ABC-999'
        )
        expect(results.count).to eq 1
        expect(results).to include(@consultation_with_all_fields)
      end

      it "should filter based on all filterable consultation/veteran record fields plus authorization number", cpp: true do
        results = Consultation.filter(
          'first_name' => @veteran_with_ssn.first_name,
          'last_name' => @veteran_with_ssn.last_name,
          'ssn' => @veteran_with_ssn.ssn,
          'visn_id' => @test_visn.id.to_s,
          'care_category_id' => @test_care_category.id.to_s,
          'authorization_number' => 'ABC-123-999',
          'consultation_number' => 'ABC-999'
        )
        expect(results.count).to eq 1
        expect(results).to include(@consultation_with_all_fields)
      end

    end # context "including a record with all filter fields"


    context "and generating CSV documents based on filtered results" do

      before :context do
        @veteran_names = Cpp::Local::Veteran.all.map &:first_name
        @find_any_name = -> (data_row) do
          @veteran_names.any? {|name| data_row.include? name }
        end
      end

      it "should return all records in CSV file", cpp: true do
        csv_rows = Consultation.to_csv({}).split("\n")
        # count includes header line
        expect(csv_rows.count).to eq (@consultations.count + 1)

        csv_header = csv_rows.first
        csv_rows_without_header = csv_rows.drop 1

        expect(@find_any_name.call(csv_header)).to eq false
        csv_rows_without_header.each do |csv_row|
          expect(@find_any_name.call(csv_row)).to eq true
        end
      end

      it "should return one record in CSV file for one-record filter", cpp: true do
        csv_rows = Consultation.to_csv('first_name' => @veteran_with_testname.first_name).split("\n")
        # count includes header line
        expect(csv_rows.count).to eq 2

        # header line should not contain veteran names, and first line should
        # contain veteran whose name was passed into to_csv filter parameter.
        expect(@find_any_name.call(csv_rows[0])).to eq false
        expect(@find_any_name.call(csv_rows[1])).to eq true
        expect(csv_rows[1]).to include(@veteran_with_testname.first_name)
      end

    end # context "and generating CSV documents based on filtered results"

  end # context "when performing filtering"

end
