require 'rails_helper'
require "cancan/matchers"

describe "Permissions", type: :model do
  let(:supervisor) { create(:supervisor) }
  let(:msa) { create(:medical_assistant) }
  let(:examiner) { create(:examiner) }
  let(:examiner_msa) { create(:examiner, roles: ["examiner", "medical_assistant"]) }

  context "for supervisors" do
    subject(:ability) { Ability.new(supervisor) }

    context "working with claims" do
      it { should be_able_to(:index, Claim) }
      it { should be_able_to(:new, Claim) }
      it { should be_able_to(:create, build(:claim)) }
      it { should be_able_to(:show, create(:claim)) }
    end

    context "working with contentions" do
      let(:claim) { create(:claim, contentions: contentions) }
      let(:contention) { create(:contention, claim_folder_reviewed: true, history: "someone entered text") }

      it { should be_able_to(:new, Contention) }
      it { should be_able_to(:create, Contention) }
      it { should be_able_to(:show, Contention) }
      it { should be_able_to(:edit, Contention) }
      it { should be_able_to(:update, Contention) }
      it { should be_able_to(:resolve, Contention) }

      it "should be able to accept resolved contentions" do
        contention.resolve!(User.create)
        expect(subject).to be_able_to(:accept, contention)
      end

      it "should not be able to accept unresolved contentions" do
        expect(subject).not_to be_able_to(:accept, contention)
      end

      it { should be_able_to(:create, Rejection) }
    end

    context "working with evaluations" do
      let(:my_evaluation) { build(:evaluation, user: supervisor) }
      let(:not_my_evaluation) { build(:evaluation, user: examiner) }

      it { should be_able_to(:new, Evaluation) }
      it { should be_able_to(:create, build(:evaluation)) }
      it { should be_able_to(:show, build(:evaluation)) }
      it { should be_able_to(:review, build(:evaluation)) }
      it { should be_able_to(:complete, build(:evaluation)) }

      [:edit, :update, :autosave].each do |action|
        it { should     be_able_to(action,     my_evaluation) }
        it { should_not be_able_to(action, not_my_evaluation) }
      end
    end
  end

  context "for MAs" do
    subject(:ability) { Ability.new(msa) }

    # can do anything with claims
    context "working with claims" do
      it { should be_able_to(:index, Claim) }
      it { should be_able_to(:new, Claim) }
      it { should be_able_to(:create, build(:claim)) }
      it { should be_able_to(:show, create(:claim)) }
    end

    # can create and view contentions, but not edit them
    context "working with contentions" do
      it { should be_able_to(:new, Contention) }
      it { should be_able_to(:create, Contention) }
      it { should be_able_to(:show, create(:contention)) }

      it { should_not be_able_to(:edit, create(:contention)) }
      it { should_not be_able_to(:update, create(:contention)) }
      it { should_not be_able_to(:resolve, create(:contention)) }
      it { should_not be_able_to(:accept, create(:contention)) }
      it { should_not be_able_to(:create, Rejection) }
    end

    context "working with with evaluations" do
      it { should be_able_to(:show, build(:evaluation)) }

      it { should be_able_to(:show, build(:evaluation)) }
      it { should be_able_to(:new, Evaluation) }
      it { should be_able_to(:create, Evaluation) }
      it { should be_able_to(:destroy, Evaluation) }
      it { should_not be_able_to(:edit, build(:evaluation)) }
      it { should_not be_able_to(:update, build(:evaluation)) }
      it { should_not be_able_to(:review, build(:evaluation)) }
      it { should_not be_able_to(:complete, build(:evaluation)) }
      it { should_not be_able_to(:autosave, build(:evaluation)) }
    end
  end
   
  context "for Examiners" do
    subject(:ability) { Ability.new(examiner) }

    let(:claim) { create(:claim, contentions: [contention]) }
    let(:contention) { create(:contention, claim_folder_reviewed: true, history: "someone entered text") }
    let(:my_evaluation) { build(:evaluation, user: examiner) }
    let(:not_my_evaluation) { build(:evaluation, user: supervisor) }

    context "working with claims" do
      it { should_not be_able_to(:new, Claim) }
      it { should_not be_able_to(:create, Claim) }

      [:show, :edit, :update, :index].each do |action|
        it { should be_able_to(action, claim) }
      end
    end

    context "working with contentions" do
      it { should_not be_able_to(:new, Contention) }
      it { should_not be_able_to(:create, Contention) }
      it { should_not be_able_to(:accept, contention) }
      it { should_not be_able_to(:create, Rejection) }

      [:show, :edit, :update, :resolve].each do |action|
        it { should be_able_to(action, contention) }
      end
    end

    context "working with with evaluations" do
      it { should be_able_to(:review,   not_my_evaluation) }
      it { should be_able_to(:complete, not_my_evaluation) }

      [:new, :create, :show, :edit, :update, :autosave, :destroy].each {|action|
        it { should     be_able_to(action,     my_evaluation) }
        it { should_not be_able_to(action, not_my_evaluation) }
      }
    end
  end

  context "For Examiners who are also MSAs" do
    subject(:ability) { Ability.new(examiner_msa) }

    let(:claim) { create(:claim, contentions: [contention]) }
    let(:contention) { create(:contention, claim_folder_reviewed: true, history: "someone entered text") }
    let(:my_evaluation) { build(:evaluation, user: examiner_msa) }
    let(:not_my_evaluation) { build(:evaluation, user: supervisor) }

    # can see a list of all claims, but only show/edit/update their own
    context "working with claims" do
      # as MSAs they can do anything with claims
      [:show, :edit, :update, :index, :new, :create].each do |action|
        it { should be_able_to(action, claim) }
      end
    end

    # can do anything with contentions but accept or reject
    context "working with contentions" do
      [:show, :edit, :update, :new, :create, :resolve].each do |action|
        it { should be_able_to(action, contention) }
      end

      # they should not be able to accept or reject a contention
      it { should_not be_able_to(:accept, contention) }
      it { should_not be_able_to(:create, Rejection) }
    end

    context "working with with evaluations" do
      # as MSAs they can show any evaluation
      it { should be_able_to(:show, not_my_evaluation) }

      # as examiners they can review or complete any evaluation
      it { should be_able_to(:review, not_my_evaluation) }
      it { should be_able_to(:complete, not_my_evaluation) }
      it { should be_able_to(:new, Evaluation) }
      it { should be_able_to(:create, Evaluation) }

      # as examiners they can new, create, edit, update their own evals
      [:edit, :update].each {|action|
        it { should     be_able_to(action,     my_evaluation) }
        it { should_not be_able_to(action, not_my_evaluation) }
      }
    end
  end
end
