# lib/tasks/db.rake
# Uses tar/gzip for directory compression
# Uses pg_dump and psql for database access
#
# Implements db:backup, db:restore and db:klone as rake tasks.
#
# Backups are stored as a compressed directory of *.sql files - one
# file per table.  The time-stamped directory is compressed into a *.tgz file
# via `tar czf`
#
# Cloning a database produces no temporary files.  The content and structure of
# one database is "piped" into the target database.
#
#
############
# db:backup also invokes db:tables  db:schema:dump  and  db:structure:dump
#
# The db:tables tasks compares the table names defined in this rake file with
# the table names that are defined in the database.  If there is a difference
# an error message will be displayed on STDERR.  An error in db:tables
# will prevent the db:backup tasks from running.  The purpose of the error
# is to alert DevOps maintainers that a change in the database structure must
# invole a change to the content of the TABLES array in this rake file.
#
############
# db:restore will delete the database and recreate it using
# db:drop   db:create  and db:migrate
#
# Restoring tables requires that table dependency be taken into account.
# Tables who are referenced by a foreign key from another table MUST be
# restored first before the "using" table is restored.
#
# This task automatically uses the latest backup *.tgz file
# for a specific database.
#
############
# db:klone is a special purpose task that copies the development database
# to the test database on the same server.  If you want a more general
# purpose database cloning capability install the 'db-clone' gem.
#
# If the test database does not currently exist, it will be created.
#
############
# db:reset is defined in the Rakefile in the Rails.root directory
#

namespace :db do

  REQUIRED_UTILITIES = %w[ tar gzip pg_dump psql ]

  I_AM_LOCAL        = ENV['DBHOST'].include?('localhost')

  DB_BACKUP_TIMESTAMP_FORMAT  = '%Y%m%d_%H%M%S'

  BACKUP_DIR        = Rails.root + 'db' + 'backup'
  DATABASE_NAME     = ENV['DBNAME'] + "_" + ENV['RAILS_ENV']

  # NOTE: The order of the table names is important.  A table must be restored before
  #       any other table can reference it.
  TABLES = %w(
    alerts
    appointment_objects
    claims
    clarification_details
    clarification_types
    consultation_types
    consultation_statuses
    consultation_orders
    contention_details
    contention_objects
    contentions_evaluations
    contentions_examinations
    dashboards
    delayed_jobs
    diagnoses
    diagnosis_codes
    diagnosis_modifiers
    dm_assignments
    evaluation_logs
    evaluation_specs
    evaluation_templates
    exam_management_notifications
    exam_request_histories
    exam_request_processors
    exam_request_states
    examination_histories
    examination_notes
    examination_schedules
    examination_states
    html_repositories
    major_systems
    medical_specialties
    medical_specialties_providers
    minor_systems
    notification_logs
    notifications
    pointless_feedback_messages
    preferred_geo_locations
    referral_document_types
    referral_statuses
    referral_types
    referral_reasons
    regional_offices
    rejections
    request_objects
    service_periods
    site_role_sets
    sites
    supervised_clinicians
    supervising_clinicians
    symp_diag_relations
    symptoms
    vbms_r_fact_groups
    exam_prioritization_special_issues
    exam_requesters
    contentions
    exam_requests
    examinations
    evaluations
    rework_reason_free_texts
    rework_reasons

    users
    user_preferences

    other_health_insurances
    veterans

    care_categories
    consultations

    visns
    facilities
    providers
    providers_users

    referrals
    referral_approvals
    referral_documents
    referral_notes
    referral_appointments

    general_questions
    qm_assignments
    question_modifiers

    support_request_categories
    support_request_organizations
    support_requests

    clinics
    examination_review_questionnaires

    dbq_informations
    boilerplate_messages
    boilerplate_messages_dbq_informations
    contentions_dbq_informations
    symptoms_dbq_informations
    diagnoses_dbq_informations
    general_questions_dbq_informations
)


  ################################################################
  ## rake db:check_utilities
  #
  desc "Make sure all required utilities are present"
  task check_utilities: :environment do
    exit() unless all_required_utilities_exist?
  end


  ################################################################
  ## rake db:tables
  #
  desc "check to see if any tables have been changed"
  task tables: :environment do
    current_tables = ActiveRecord::Base.connection.tables
    # NOTE: schema_migrations is left out of the list because db:restore does a db:migrate task
    #       which rebuilds the table.... so no need to restore from backup.
    changed_tables = (TABLES - current_tables) + (current_tables - TABLES) - ['schema_migrations']

    # SMELL: whatif the reason for doing the restore is that a table was deleted by mistake?
    #        This check would preclude the restore.... So maybe we don't want to do
    #        this tasks on a restore.
    unless changed_tables.empty?
      STDERR.puts <<~EOS

        ERROR: There has been a change in the tables within #{DATABASE_NAME}.

               The db:backup and db:restore tasks need to be modified to accomidate this change.
               The following table names are impacted:

               #{changed_tables.join(',  ')}

      EOS
      exit
    end
  end # task tables: :environment do


  ################################################################
  ## rake db:pgpass
  #
  desc "Check for and create as necessary a $HOME/.pgpass file"
  task :pgpass do
    pgpass_pathname = Pathname.new(ENV['HOME']) + '.pgpass'
    generic_entry   = "*:*:*:#{ENV['DBUSER']}:#{ENV['DBPASS']}"

    if pgpass_pathname.exist?
      contents = pgpass_pathname.read
      unless contents.include?(generic_entry)
        pgpass_pathname.chmod(0666)
        fileout = File.new(pgpass_pathname, 'a')
        fileout.puts generic_entry
        fileout.close
      end
    else
      fileout = File.new(pgpass_pathname, 'w')
      fileout.puts generic_entry
      fileout.close
    end

    pgpass_pathname.chmod(0600)
  end # task :pgpass do


  ################################################################
  ## rake db:backup
  #
  desc "Backup the database #{DATABASE_NAME}"
  task backup: %w( db:check_utilities db:tables db:pgpass ) do

    timestamp       = Time.current.strftime(DB_BACKUP_TIMESTAMP_FORMAT)
    backup_dir      = BACKUP_DIR + ( DATABASE_NAME + '_' + timestamp )
    backup_dir.mkdir

    puts "\nBacking up #{DATABASE_NAME} to #{backup_dir.basename}.tgz ..."

    TABLES.each do |table_name|
      print "  #{table_name} ... "

      table_pathname = backup_dir + (table_name + '.sql')

      command = "pg_dump --data-only --host #{ENV['DBHOST']} --format=plain --username=#{ENV['DBUSER']} --blobs --table=#{table_name} --file=#{table_pathname} #{DATABASE_NAME}"
      system command

      puts "done"
    end

    compress_dir(backup_dir)
  end # task backup: %w( db:tables ) do


  ################################################################
  ## rake db:restore
  #
  desc "Restore the database #{DATABASE_NAME}"
  task restore: %w( db:check_utilities db:drop db:create db:migrate db:schema:dump  )  do

    # TODO: give user a choice of which backup to use.
    backups     = BACKUP_DIR.children.select {|path| path.basename.to_s.start_with?(DATABASE_NAME) }

    if backups.empty?
      STDERR.puts "\nERROR: There are no backups available for #{DATABASE_NAME}\n\n"
      exit(-1)
    end

    last_backup = backups.sort.last
    backup_dir  = uncompress_file(last_backup)

    puts "\nRestoring #{DATABASE_NAME} from #{last_backup.basename} ..."

    TABLES.each do |table_name|
      print "  #{table_name} ... "

      table_pathname = backup_dir + (table_name + '.sql')

      command = "psql --single-transaction --host=#{ENV['DBHOST']} --username=#{ENV['DBUSER']} --dbname=#{DATABASE_NAME} --file=#{table_pathname} > /dev/null"
      system command

      puts "done"
    end

  end # task restore: %w( db:drop db:create db:migrate db:schema:dump db:structure:dump ) do


  ################################################################
  ## rake db:klone
  #
  # NOTE: There is a gem 'db-clone' that adds a clone task.
  #       Its default is to copy production to development; but,
  #       it also has a manual menu option that allows the choice
  #       for the source and destination.
  #
  # This task, however, is special purpose only allowing the
  # cloning of the development database on one server to the
  # test database on the same server.  It is intended for
  # local workstation use by developers and maybe automated
  # test jobs.

  desc "Kopy the development db to the test db"
  task klone: %w[ db:check_utilities ] do

    source_destination = {
      source: {
        host:     ENV['DBHOST'],
        port:     ENV['DBPORT'],
        username: ENV['DBUSER'],
        database: ENV['DBNAME'] + "_development"
      },
      destination: {
        host:     ENV['DBHOST'],
        port:     ENV['DBPORT'],
        username: ENV['DBUSER'],
        database: ENV['DBNAME'] + "_test"
      }
    }

    # Start by clearing out whatever currently exists in the test database
    command = "RAILS_ENV=test rake"
    command += " db:drop" if test_db_exist?
    command += " db:create"
    system command

    command = build_postgresql_clone_cmd(source_destination)

    puts "Executing: #{command}"
    system command

  end # task :klone do


  ################################################################
  ## rake db:klone_dev2prod

  desc "Kopy the development db to the production db"
  task klone_dev2prod: %w[ db:check_utilities ] do

    unless DBRESET  &&  I_AM_LOCAL
      puts <<~EOS

        ERROR: This is a very __dangerious__ command.  It is only allowed when DBHOST
               is localhost and DBRESET is true.
                  Your DBHOST:  #{ENV['DBHOST']}
                  Your DBRESET: #{DBRESET}

      EOS
      exit
    end

    source_destination = {
      source: {
        host:     ENV['DBHOST'],
        port:     ENV['DBPORT'],
        username: ENV['DBUSER'],
        database: ENV['DBNAME'] + "_development"
      },
      destination: {
        host:     ENV['DBHOST'],
        port:     ENV['DBPORT'],
        username: ENV['DBUSER'],
        database: ENV['DBNAME'] + "_production"
      }
    }

    # Start by clearing out whatever currently exists in the test database
    command = "RAILS_ENV=production rake"
    command += " db:drop" if production_db_exist?
    command += " db:create"
    system command

    command = build_postgresql_clone_cmd(source_destination)

    puts "Executing: #{command}"
    system command

  end # task klone_dev2prod: %w[ db:check_utilities ] do







  ################################################################
  ## Utility methods used by the backup/restore process

  # Does the target database of the db:klone task exist?
  def test_db_exist?
    exit() unless all_required_utilities_exist?
    command = "psql -lqt -U #{ENV['DBUSER']} | cut -d \\| -f 1 | grep -qw #{ENV['DBNAME']}_test"
    system command
  end


  # Does the target database of the db:klone task exist?
  def production_db_exist?
    exit() unless all_required_utilities_exist?
    command = "psql -lqt -U #{ENV['DBUSER']} | cut -d \\| -f 1 | grep -qw #{ENV['DBNAME']}_production"
    system command
  end


  # Create a pg_dump command that clones databases
  def build_postgresql_clone_cmd( src_dest )
    [
      "pg_dump --no-password --clean",
      "--host=#{src_dest[:source][:host]}",
      "--port=#{src_dest[:source][:port]}",
      "--username=#{src_dest[:source][:username]}",
      "#{src_dest[:source][:database]}",
      "| psql",
      "--host=#{src_dest[:destination][:host]}",
      "--port=#{src_dest[:destination][:port]}",
      "--username=#{src_dest[:destination][:username]}",
      "#{src_dest[:destination][:database]}"
    ].join(' ')
  end


  # Use tar/gzip to uncompress a file into a sub-directory
  def uncompress_file(in_path)
    parent_dir  = in_path.parent
    backup_dir  = parent_dir + (in_path.basename.to_s.gsub('.tgz',''))

    command = "cd #{parent_dir}  &&  tar xzf #{in_path.basename}"
    system command

    backup_dir
  end


  # Use tar/gzip to compress a directory
  def compress_dir(in_path)
    out_path  = Pathname.new(in_path.to_s + '.tgz')

    command   = "cd #{in_path}/..  &&  tar czf #{out_path.basename} #{in_path.basename}  &&  rm -rf #{in_path}"
    system command

    out_path
  end


  # Check for existence of required utilities
  def all_required_utilities_exist?
    result = true
    errors = []

    REQUIRED_UTILITIES.each do |utility|
      if `which #{utility}`.empty?
        result = false
        errors << utility
      end
    end

    unless result
      puts "\nERROR: This task can not be executed because the following utilit#{errors.size > 1 ? 'ies are' : 'y is'} missing:"
      puts "       #{errors.join(', ')}"
      puts
    end

    return result
  end

end # namespace :db do
