module VIA_API
  module Models

  # SMELL: If this is appointments coming from VISTA  HOW does that system
  #        know about the CPP Providers and their facilities ???  It is more
  #        likely that these VISTA-based appointments are for VA sites and
  #        clinicians and NOT for CPP facilities and providers.

    # Defined using spec/support/via_api/get_appointments_response.xml
    class Appointment < ::EnhancedHash
      require 'facility'
      require 'provider'

    # access by:                         API uses:
      property :id
      property :timestamp
      property :status
      property :facility,                  coerce: VIA_API::Models::Facility
      property :provider,                from: :clinic,
                                           coerce: VIA_API::Models::Provider
      property :lab_time,                from: :labDateTime
      property :xray_time,               from: :xrayDateTime
      property :ekg_time,                from: :ekgDateTime
      property :purpose
      property :appointment_type,        from: :type
      property :current_status,          from: :currentStatus
      property :content,                 from: :__content__
      property :appointment_length,      from: :appointmentLength
      property :book_notes,              from: :bookNotes
      property :reason

      # list of all appointment statuses considered 'cancelled' -- all upper case
      CANCELLED_STATUSES = ['N', 'C', 'PC', 'NO-SHOW', 'CANCELLED BY CLINIC', 'CANCELLED BY PATIENT']

      # list of all purposes considered 'active' for appointments -- all upper case
      ACTIVE_PURPOSES = ['C&P', '10-10', 'SCHEDULED VISIT']

      def vista_appointment?
        true
      end

      # return true if purpose and status indicate that this appointment is active
      def active?
        ACTIVE_PURPOSES.include?(purpose.upcase) && CANCELLED_STATUSES.exclude?(current_status.upcase)
      end

      def inactive?
        not active?
      end

      # return appointment timestamp as Ruby date/time object
      def appointment_time
        return VIA_API::parse_vista_date(timestamp)
      end

      def past_appointment?
        appointment_time < Time.now.beginning_of_day
      end

      # Handles options from get_appointments and returns true if the
      # appointment passed in is to be ignored.
      def ignored_appointment?(options)
        return true if options[:future_only] && past_appointment?
        return true if options[:active_status_only] && inactive?
        false
      end

      # returns an array of fields that appear in 'Details' section of appointment datatable
      def details
        [facility_info, provider_info, purpose, current_status]
      end

      def scheduled_by
        provider.name
      end

      def facility_name
        provider.name
      end

      # VISTA appointments are read-only and never flagged as approved in our system.
      def added_to_cprs?
        false
      end

      def added_to_cprs_username
        nil
      end

      # "id: name" formatted facility information
      def facility_info
        "#{facility.tag}: #{facility.text}"
      end

      # "id: name" formatted provider (clinic) information
      def provider_info
        "#{provider.id}: #{provider.name}"
      end

      def facility_name
        provider.try!(:facility).try!(:name)
      end

    end # class Appointment < EnhancedHash
  end
end # module VIA_API
