# == Schema Information
#
# Table name: sites
#
#  id              :integer          not null, primary key
#  name            :string
#  address         :string
#  city            :string
#  state           :string
#  country         :string
#  created_at      :datetime         not null
#  updated_at      :datetime         not null
#  access_requests :string           default([]), is an Array
#  role_requests   :string           default([]), is an Array
#  site_station_number     :string
#  zip_code        :string
#  deleted_at      :datetime
#

class Site < ActiveRecord::Base

    acts_as_paranoid

    has_many :site_role_sets
    has_many :examinations
    has_many :exam_requests
    validates_presence_of :name
    validates_presence_of :address
    validates_presence_of :city
    validates_presence_of :state
    validates_presence_of :country
    scope :site_role_admin, -> { where(:site_role_sets => {admin: "true"}) }
    scope :site_role_triage, -> { where(:site_role_sets => {triage: "true"}) }
    scope :site_role_scheduling, -> { where(:site_role_sets => {scheduling: "true"}) }
    scope :site_role_clinician, -> { where(:site_role_sets => {clinician: "true"}) }
    scope :site_role_super_clinician, -> { where(:site_role_sets => {super_clinician: "true"}) }
    scope :site_role_qa, -> { where(:site_role_sets => {qa: "true"}) }
    scope :order_asc, -> { all.order("name ASC")}

    has_many :users, through: :site_role_sets do
      def << (u)
        if !self.include? u
          self.push u
        end
      end
    end

  def add_user(u)
    self.users << u
    self.save
  end

  def remove_user(u)
    self.users.delete(u)
  end

  # Get the roles that a user has on this site
  def get_site_roles(user)
    return self.site_role_sets.find_by_user_id(user.id)
  end


  def add_requesting_user(user, roles)
    if !self.users.include?(user)
      if !self.access_requests.include?(user.id.to_s)
        if user.id && roles
          self.access_requests << user.id.to_s
          self.role_requests << roles
          self.save
        end
      else
        #If user is requesting access again, just update the roles being requested.
        position = self.access_requests.find_index(user.id.to_s)
        self.role_requests[position] = roles
      end
    end
  end

  def remove_requesting_user(user)
    if self.access_requests.include?(user.id.to_s)
      position = self.access_requests.find_index(user.id.to_s)
      self.access_requests.delete(user.id.to_s)
      self.role_requests.delete_at(position)
      self.save
    end
  end

  def get_requesting_users
    users = []
    self.access_requests.each do |r|
      users << User.find(r)
    end
    return users
  end

  def get_roles_requested(user)
    if self.access_requests.include?(user.id.to_s)
      position = self.access_requests.find_index(user.id.to_s)
      roles = self.role_requests[position]
      return roles
    end
    return false
  end


end
