require 'awesome_print'
class NpiRegistry
  include ActiveModel::Model

  attr_accessor :number

############################################################################
##                                                                        ##
## The web-service API query will return a max of 200 results/request.    ##
## The Skip field in the API will let you skip up to 1000 records.        ##
## By using these two fields with your search criteria,                   ##
## you can get up to a maximum of 1,200 records over six requests.        ##
##                                                                        ##
##     https://npiregistry.cms.hhs.gov/api/?number=&                      ##
##                                          enumeration_type=&            ##
##                                          taxonomy_description=&        ##
##                                          first_name=&                  ##
##                                          last_name=&                   ##
##                                          organization_name=&           ##
##                                          address_purpose=&             ##
##                                          city=Baltimore&               ##
##                                          state=&                       ##
##                                          postal_code=&                 ##
##                                          country_code=&                ##
##                                          limit=200&                    ##
##                                          skip=1000                     ##
##                                                                        ##
## The query keys are set by the web-service; however to be consistent    ##
## with how the provider model is defined within this web application we  ##
## are standardizing the app-size API to use "npi" as the key in place of ##
## of "number" so that from the applications point of view the access is  ##
## is consistent.                                                         ##
##                                                                        ##
############################################################################

  class << self

    # standardize API for application making web-service access
    # consistent with local database

    def standardize_api(params)

      if params.keys.include?(:npi)
        params[:number] = params[:npi]
        params.delete(:npi)
      end

      return params
    end


    def extract_physician_name(a_hash)

      name  = [
                a_hash["authorized_official_name_prefix"] || a_hash['name_prefix'],
                a_hash["authorized_official_first_name"]  || a_hash['first_name'],
                a_hash["authorized_official_middle_name"] || a_hash['middle_name'],
                a_hash["authorized_official_last_name"]   || a_hash['last_name'],
                ',',
                a_hash["authorized_official_credential"]  || a_hash['credential']
              ].join(' ').titlecase.squeeze(' ').strip.gsub(' ,', ',').gsub(/,$/,'')

      name = a_hash['name'].titlecase if name.blank?

      return name
    end


    # find a facility that matches the address of this provider.  If none,
    # create one.
    # SMELL: Just exactly what is a facility and is it really needed?
    #        We have been saying that a facility is just a physical address; for example,
    #        an office park.  The addresses of providers contain suite numbers which
    #        are specific to the office building.  Since the providers have their
    #        array of addresses - its starting to look like the concept of a
    #        facility is no longer needed.

    def find_or_create_facility(a_hash)

      basic       = a_hash["basic"]
      full_name   = "#{basic['name_prefix']}  #{basic['first_name']} #{basic['middle_name']}  #{basic['last_name']}, #{basic['credential']}".strip.squeeze(' ').gsub(/,$/,'')

      organization_name = basic["organization_name"] || full_name || basic["name"]

      facility          = Facility.find_by_name(organization_name)

      if facility.blank?
        # TODO: assign VISN based upon postal_code of the facility
        #       I dislike random test data; but its necessary to fill out the dropdown menus
        facility = Facility.create(name: organization_name, visn_id: 1+rand(18), content: a_hash["addresses"])
      end

      return facility
    end


    # Access the NPI Registry web-service
    def search(params)
      @params           = standardize_api(params)
      @params[:limit]   = 200       # 200 is the largest number allowed by the API
      @content          = []
      @npi_registry_id  = []

      conn            = Faraday.new(:url => ENV['URL_FOR_NPI_LOOKUP'])
      response        = conn.get '/api', @params

      @response_data  = JSON.parse response.body

      # Keys of @response_data - "result_count" & "results"
      @result_count   = @response_data["result_count"]
      if @response_data["results"].nil?
        return nil
      else
        @response_data["results"].each do |result|
          @content << result
        end
      end

      # Increment skipvalue by 200 for queries that return 1000+ results
      while @result_count == 200 && @params[:skip].to_i < 1000
        @new_params           = @params
        @new_params[:skip]    = @new_params[:skip].to_i + 200
        response              = conn.get '/api', @new_params
        @response_data        = JSON.parse response.body
        @result_count         = @response_data["result_count"]
        @response_data["results"].each do |result|
          @content << result
        end
      end

      @content.each do |result|
        @npi_registry   = Provider.find_by_npi(result["number"])
        if @npi_registry.blank?
          @npi_registry = Provider.create(
            npi:            result["number"],
            name:           result["basic"]["name"],
            physician_name: extract_physician_name(result["basic"]) || 'unknown',
            content:        result,
            facility_id:    find_or_create_facility(result).id
          )

          @npi_registry_id << @npi_registry.id
        end
      end

      @provider = Provider.find(@npi_registry_id)
      return @provider
    end


    # Access the the local database table first before accessing the
    # web-service
    def filter(params)
      @filter_params = standardize_api(params)

      if @filter_params[:number].nil?
        # For searches not based on npi
        @provider = search(@filter_params)
      else
        @provider = Provider.find_by_npi(@filter_params[:number])
        if @provider.blank?
          @provider = search(@filter_params)
          @provider = @provider.first unless @provider.nil?
        end
      end
      return @provider
    end

  end # class << self

end # class NpiRegistry
