# == Schema Information
#
# Table name: claims
#
#  id                         :integer          not null, primary key
#  completed_at               :datetime
#  created_at                 :datetime
#  updated_at                 :datetime
#  patient_ssn                :string
#  file_number                :string
#  first_name                 :string
#  middle_name                :string
#  last_name                  :string
#  date_of_birth              :date
#  facility_name              :string
#  facility_number            :string
#  exam_request_data          :text
#  vbms_claim_id              :string
#  claim_date                 :date
#  email                      :string
#  phone_number               :string
#  benefit_type               :string
#  label                      :string
#  end_product_code           :string
#  edipi                      :string
#  participant_id             :string
#  gender                     :string
#  vista_ien                  :string
#  alternate_phone            :string
#  integration_control_number :string
#  poa_vso_name               :string
#  poa_vso_number             :string
#  regional_office_id         :integer
#

class Claim < ActiveRecord::Base
  has_many :contentions, dependent: :destroy
  has_many :evaluations, dependent: :destroy

  has_many :examinations
  has_many :exam_requests
  has_many :examination_schedules, through: :examinations

  validates_presence_of :first_name
  validates_presence_of :last_name
  #validates_presence_of :patient_ssn
  validates_presence_of :file_number
  validates_presence_of :date_of_birth

  auto_strip_attributes :first_name, :middle_name, :last_name, :patient_ssn, :file_number, :date_of_birth

  accepts_nested_attributes_for :contentions, allow_destroy: true, reject_if: proc {|attributes| attributes.collect{|k,v| v.blank?}.all? }

  serialize :exam_request_data, Hash

  scope :with_exam_schedules, -> () { joins(:examinations => :examination_schedules).distinct }

  scope :active_exam_schedule, -> () { where("examination_schedules.active = true").references(:claims) }

  scope :with_start_and_end_date, proc { |start_date, end_date|
    where("examination_schedules.exam_date_time BETWEEN ? AND ?", start_date, end_date).active_exam_schedule if start_date.present? && end_date.present?
  }

  scope :with_patient_name, proc { |name|
    where("CONCAT_WS(' ', first_name, middle_name, last_name) ILIKE :name OR CONCAT_WS(' ', first_name, last_name) ILIKE :name", name: "%#{name}%") if name.present?
  }

  scope :with_patient_id, proc { |patient_id|
    where("vbms_claim_id ILIKE ?" , "%#{patient_id}%") if patient_id.present?
  }

  TIMESTAMP_FORMAT = '%m/%d/%Y'.freeze

  def self.appointment_search(params)
    search_params = clean_params(params)
    with_exam_schedules
    .with_start_and_end_date(search_params[:start_date], search_params[:end_date])
    .with_patient_name(search_params[:name])
    .with_patient_id(search_params[:patient_id_search])
  end

  def summary
    self.contentions.collect{|contention| contention.name }.join(", ")
  end

  def masked_ssn
    self.patient_ssn.present? ? "XXX-XX-#{self.patient_ssn[-4..-1]}" : ""
  end

  def other_contentions(contention)
   self.contentions.all - [contention]
  end

  def patient_name
    [self.first_name, self.middle_name, self.last_name].join(" ").squish
  end

  def build_date_of_birth(params)
    if params[:date_of_birth_day].present? && params[:date_of_birth_month].present? && params[:date_of_birth_day].present?
      DateTime.new(params[:date_of_birth_year].to_i,
                   params[:date_of_birth_month].to_i,
                   params[:date_of_birth_day].to_i)
    else
      nil
    end
  end

  def self.create_from_exam_request_xml(exam_request_xml)
    new_claim = Claim.new

    exam_request_hash = Hash.from_xml(exam_request_xml)

    if exam_request_hash && exam_request_hash['feed'] && exam_request_hash['feed']['entry'] && exam_request_hash['feed']['entry']['content'] &&
      exam_request = exam_request_hash['feed']['entry']['content']['ExamRequest']
      examiner_email_address = exam_request['VhaUserContactInfo'].try(:[], 'EmailAddress')
      examiner_username = exam_request['VhaUserContactInfo'].try(:[], 'UserName')
      examiner = User.find_or_create_by(email: examiner_email_address.downcase) do |examiner|
        examiner.first_name, examiner.last_name = examiner_username, examiner_username
        examiner.password = SecureRandom.hex
        examiner.roles << "examiner"
      end

      ActiveRecord::Base.transaction do
        new_claim = create_new_claim_from_exam_data(exam_request)

        contentions_data = exam_request['ContentionsInformation']['Contention']
        contentions_data = [contentions_data] if contentions_data.is_a? Hash
        contentions_data.each do |contention|
          contention_name = contention['ContentionClassificationName']
          new_contention = Contention.create(name: contention_name, history: contention['ContentionDescription'], claim_id: new_claim.id)

          evaluations_data = exam_request['ExamsInformation']['Exam']
          evaluations_data = [evaluations_data] if evaluations_data.is_a? Hash
          evaluations_data.each do |evaluation|
            dbq_evaluation_type = evaluation['ExamTemplateDocumentType']
            if dbq_evaluation_type.present?
              dbq_evaluation_key = get_dbq_evaluation_key(dbq_evaluation_type)
              if dbq_evaluation_key.present?
                spec = EvaluationSpec.find_by_title(EVALUATION_MAPPINGS[dbq_evaluation_key])
                if spec
                  new_evaluation = Evaluation.create(claim: new_claim, evaluation_spec: spec, user: examiner)

                  # This is a little weird, because it will be logged after the "evaluation created" message
                  # that gets created as a post-commit hook on the evaluation model but there's no way to
                  # log before it gets created in the database
                  EvaluationLog.create(evaluation_id: new_evaluation.id,
                                             message: "Evaluation was submitted as an ExamRequest")

                  new_contention.evaluations << new_evaluation
                end
              end
            end
          end
        end
      end
    end
    new_claim
  end


  def self.clean_params(params)

    return params if params.blank?

    clean_params = params.permit(:start_date, :end_date, :name, :patient_id_search)

    clean_params[:start_date] = Date.strptime(params[:start_date], TIMESTAMP_FORMAT).beginning_of_day if clean_params[:start_date].present?
    clean_params[:end_date]   = Date.strptime(params[:end_date],  TIMESTAMP_FORMAT).end_of_day if clean_params[:end_date].present?

    clean_params
  end

  private

  def self.create_new_claim_from_exam_data(exam_request)
    claim_data = exam_request['ClaimInformation']
    patient_data = claim_data['VeteranServiceMemberInformation']
    Claim.create(first_name: patient_data['FirstName'],
                 middle_name: patient_data['MiddleName'],
                 last_name: patient_data['LastName'],
                 patient_ssn: patient_data['SocialSecurityNumber'].try(:[], 'Value'),
                 file_number: patient_data['FileNumber'],
                 date_of_birth: patient_data['DateOfBirth'].try(:to_date),
                 facility_name: exam_request['FacilityInformation'].try(:[], 'Name'),
                 facility_number: exam_request['FacilityInformation'].try(:[], 'Id'),
                 exam_request_data: exam_request.delete_if{ |key,value| key.start_with?('xmlns:')}
                )
  end

  def self.get_dbq_evaluation_key(dbq_evaluation_type)
    EVALUATION_MAPPINGS.keys.select { |key| dbq_evaluation_type.downcase.include?(key.downcase) }.first
  end
end
