class ReferralAppointmentsController < ApplicationController
  include ReferralAppointmentsHelper

  before_action :set_referral, only: [:create]
  before_action :set_referral_and_appointment, only: [:add_to_cprs, :update]
  check_authorization
  authorize_resource

  def create
    ra_params_hash = convert_appointment_time(referral_appointment_params.to_h)
    @referral_appointment = @referral.referral_appointments.new(ra_params_hash)
    respond_to do |format|
      if @referral.save
        format.html do
          flash[:notice] = 'Appointment was successfully created.'
          redirect_to referral_path(@referral, tabpage: 'appointments')
        end
        format.json { render json: @referral, status: :created }
      else
        if @referral_appointment.appointment_time.nil?
          @referral.errors.clear
          @referral.errors.add(:base, "Please select a date and time for the appointment.")
        end
        format.html do
          flash[:alert] = @referral.errors.full_messages.join(", ")
          if @referral_appointment.errors.any?
            flash[:alert] << '. ' << @referral_appointment.errors.messages.map do |e|
              e.last.map do |e|
                e.split('.').map(&:strip).map(&:capitalize).join('. ')
              end
            end.join(", ")
          end
          redirect_to referral_path(@referral, tabpage: 'appointments')
        end
        format.json { render json: @referral.errors, status: :unprocessable_entity }
      end
    end
  end


  # Method called via Ajax call that edits appointment data in database.
  # If VIA_ACTIVE_FOR_CPP and record was previously saved (Approved), then appointment
  # also needs to be cancelled and re-scheduled.
  def update
    # first make appointment in VISTA and set added flags only if that operation succeeds.
    if VIA_ACTIVE_FOR_CPP && @referral_appointment.added_to_cprs?
      response_hash = cancel_appointment_in_vista
      notice_text = (response_hash[:error_message] if response_hash[:error])
    else
      response_hash = {}
    end

    # if there was no error cancelling appointment, then make appointment with
    # updated data.
    unless response_hash[:error]
      request_body = JSON.parse(request.body.read)

      attributes_hash = convert_appointment_time({
        'appointment_type' => request_body['appointment_type'],
        'scheduled_by' => request_body['scheduled_by'],
        'appointment_time' => request_body['appointment_time']
      })

      @referral_appointment.assign_attributes(attributes_hash)

      if VIA_ACTIVE_FOR_CPP && @referral_appointment.added_to_cprs?
        response_hash = make_appointment_in_vista
        notice_text = (response_hash[:error_message] if response_hash[:error])
      end
    end

    respond_to do |format|
      format.json do
        if (not response_hash[:error]) && @referral_appointment.save
          render json: @referral_appointment, status: :no_content
        else
          render json: (notice_text || @referral_appointment.errors), status: :unprocessable_entity
        end
      end
    end
  end


  def add_to_cprs
    # first make appointment in VISTA and set added flags only if that operation succeeds.
    if VIA_ACTIVE_FOR_CPP
      response_hash = make_appointment_in_vista
      notice_text = (response_hash[:error_message] if response_hash[:error])
    else
      response_hash = {}
    end

    respond_to do |format|
      if (not response_hash[:error]) && @referral_appointment.save
        format.html do
          notice_text = "Appointment ##{@referral_appointment.appointment_type} was successfully approved."
          redirect_to @referral, notice: notice_text
        end
        format.json do
          result_html = created_in_cprs_datacell_html(@referral_appointment, current_user)
          render json: {created_by: result_html}, status: :ok
        end
      else
        format.html do
          flash[:alert] = notice_text || @referral_appointment.errors.full_messages.join(" ")
          redirect_to @referral
        end
        format.json { render json: (notice_text || @referral_appointment.errors), status: :unprocessable_entity }
      end
    end

  end


  def destroy
    @appointment = ReferralAppointment.find(params[:id])
    @appointment.destroy
    respond_to do |format|
      format.js {}
    end
  end

private
  # Use callbacks to share common setup or constraints between actions.
  def set_referral
    @referral = Referral.find(params[:referral_id])
  end

  def set_referral_and_appointment
    set_referral
    if params.has_key?(:id)
      @referral_appointment = ReferralAppointment.find(params[:id])
    elsif params.has_key?(:referral_appointment_id)
      @referral_appointment = ReferralAppointment.find(params[:referral_appointment_id])
    else
      raise Exception, "Referral appointment ID not found."
    end
    # check if referral appointment is contained in referral
    unless @referral.referral_appointments.include? @referral_appointment
      raise Exception, "Invalid Referral appointment ##{@referral_appointment.id} for referral ##{@referral.id}."
    end
  end


  # handle VIA_API calls to cancel appointment.  Returns deleted appointment if successful.
  def cancel_appointment_in_vista
    # get clinic id from site id saved in user preferences
    clinic = Clinic.get_clinic_from_user(@referral.coordinator)
    patient = Struct.new(:mpi_pid, :local_pid).new(@referral.consultation.veteran_mpi_pid, @referral.consultation.veteran_local_pid)

    begin
      # cancel appointment through VISTA.  The cancel_code, cancel_reason, and
      # remarks parameter are always set to default values.
      appointment_list = VIA_API::SchedulingService.cancel_appointment(
        @referral_appointment.appointment_time,
        # NOTE: The patient struct is added to avoid the mpi_lookup call in the below line as it does not work in pre-prod.
        # @referral.veteran(session[:vista]),
        patient,
        clinic.clinic_id,
        11,
        'OTHER',
        'REBOOKED',
        session[:vista]
      )
      cancelled_appointment = (appointment_list.first if appointment_list.present?)

      if cancelled_appointment.present?
        return {error: false}
      else
        return {error: true, error_message: "Error cancelling appointment in VISTA."}
      end

    rescue VIA_API::ViaApiError => e
      Rails.logger.error(notice_text)
      return {error: true, error_message: "Error cancelling appointment in VISTA: #{e.message}"}
    end

  end


  # handle VIA_API calls to make appointment, and save response id successful.
  # Returns {success: <boolean>} hash, where success is true if VIA_API call and
  # internal save are both successful.
  def make_appointment_in_vista
    # first make appointment in VISTA and set added flags only if that operation succeeds.
    saved_appointment, notice_text = nil, nil
    begin
      # get clinic id from site id saved in user preferences
      clinic = Clinic.get_clinic_from_user(@referral.coordinator)

      # make appointment through VISTA.  The nil parameter is the appointment type, which
      # should get set to the default.

      patient = Struct.new(:mpi_pid, :local_pid).new(@referral.consultation.veteran_mpi_pid, @referral.consultation.veteran_local_pid)
      appointment_list = VIA_API::SchedulingService.make_appointment(
        # NOTE: The patient struct is added to avoid the mpi_lookup call in the below line as it does not work in pre-prod.
        # @referral.veteran(session[:vista]),
        patient,
        clinic.clinic_id,
        nil,
        @referral_appointment.appointment_time,
        @referral_appointment.book_notes,
        session[:vista]
      )
      saved_appointment = (appointment_list.first if appointment_list.present?)

      if saved_appointment.present?
        end_time = @referral_appointment.appointment_time + (saved_appointment.appointment_length.to_i).minutes
        @referral_appointment.purpose = saved_appointment.purpose
        @referral_appointment.appointment_end_time = end_time
        @referral_appointment.added_to_cprs_at = Time.now
        @referral_appointment.added_to_cprs_id = current_user.id
        return {error: false}
      else
        return {error: true, error_message: "Error saving appointment to VISTA."}
      end

    rescue VIA_API::ViaApiError => e
      Rails.logger.error(notice_text)
      return {error: true, error_message: e.message}
    end
  end


  # convert 'appointment_time' to TimeWithZone if found in params hash.
  # Return params hash with converted appointment time.
  def convert_appointment_time(params_hash)
    if (params_hash['appointment_time'])
      params_hash['appointment_time'] = datetime_string_to_time_with_zone(
        params_hash['appointment_time'],
        ReferralAppointment::APPOINTMENT_TIME_FORMAT,
        current_user
      )
    end
    params_hash
  end


  # Never trust parameters from the scary internet, only allow the white list through.
  def referral_appointment_params
    params.require(:referral_appointment).permit( :appointment_type,
                                                  :scheduled_by,
                                                  :appointment_time)
  end

end
