# This is a helper controller for the user model. It allows more
# routes available to do some of things that need to be done.

class PersonController < ApplicationController
  include DataTablePaginator
  before_action :get_users, only: [:index, :paged_list]
  check_authorization
  authorize_resource class: false

  def index
    @sites = Site.all.order(name: :asc).map{ |s| [s.name, s.id] }
    @sites.unshift(["No Site - show all CUI users", 0])
    site_select_id = params[:site_select_id]
    if site_select_id
      if !site_select_id.present?
        flash.now[:alert] = "Please select a site."
        render :index
        return
      end
      if params[:site_select_id] != "0"
        @site = Site.find params[:site_select_id]
      end
    end
  end


  def show
    if params[:alert_id].present?
      id = params[:alert_id].to_i
      Alert.find(id).deactivate
    end
    @user = User.find params[:id]
    @users = User.all.order(last_name: :asc).map{ |u| [u.last_name_first_name, u.id] }
    @sites = Site.all.order(name: :asc).map{ |s| [s.name, s.id] }
    if params[:site_selection]
      if !params[:site_selection].present?
        flash.now[:alert] = "Please select a site."
        render :show
        return
      end
      @site = Site.find params[:site_selection]
    end
  end


  def new
    redirect_to :root, flash: {alert: "Your privileges are insufficient"} unless current_user.is_app_admin?
    @user = User.new
    @user.user_preference = UserPreference.new
  end


  def create
    user_params = params[:user].permit(:first_name, :last_name, :email, :roles, :password, :npi).to_h
    user_params["roles"] = [user_params["roles"]]  if current_user.is_cpp_user?
    @user      = User.new(user_params)
    @user.user_preference = UserPreference.new
    @user.authorization_state = "authorized" if @user.is_cpp_user?
    @npi       = params[:user][:npi]
    @cpp_roles = params[:user][:roles] # To retain form input

    if @user.valid?  &&  @user.save
      flash[:notice] = @user.name + " has been successfully added"
      redirect_to action: "new"
    else
      errors_string = String.new
      @user.errors.full_messages.each do |msg|
        errors_string += "- " + msg + "<br/><br/>"
      end
      errors_string     = errors_string.html_safe
      flash.now[:alert] = errors_string
      render "new"
    end
  end # def create


  def edit
  end


  def update
    # This conditional is to catch a site selection from dropdown made on index (ie. /user/) and redirect back to index.
    if params[:site_select]
      redirect_to action: "index", site_select_id: params[:site_select][:id]
    elsif params[:user]
      redirect_to action: "show", id: params[:user][:id], user_selection: true
    elsif params[:site_selection]
      if params[:commit] == "Update roles"
        uroles = params[:user_roles]
        user = User.find params[:id]
        site = Site.find params[:site_id]
        site_roles = user.get_site_roles(site)
        if !site_roles.present?
          site.add_user(user)
          site_roles = user.get_site_roles(site)
        end
        uroles && uroles[:admin] ? site_roles.admin = true : site_roles.admin = false
        uroles && uroles[:triage] ? site_roles.triage = true : site_roles.triage = false
        uroles && uroles[:scheduling] ? site_roles.scheduling = true : site_roles.scheduling = false
        uroles && uroles[:clinician] ? site_roles.clinician = true : site_roles.clinician = false
        uroles && uroles[:super_clinician] ? site_roles.super_clinician = true : site_roles.super_clinician = false
        uroles && uroles[:qa] ? site_roles.qa = true : site_roles.qa = false
        site_roles.save
        flash[:notice] = "Roles for " + user.name + " have been successfully updated for " + site.name
        redirect_to action: "show", site_selection: params[:site_id]
        return true
      end
      redirect_to action: "show", site_selection: params[:site_selection][:id]
    end
  end


  def destroy
  end


  def access_requests
    @users = User.all.where(authorization_state: "pending").order(last_name: :asc)

    if params[:decision] && params[:user_id]
      @user = User.find params[:user_id]

      if !@user.action_token.blank? && params[:action_token]==@user.action_token
        if params[:decision] == "denied"
          @user.authorization_state = "rejected"
          flash.now[:alert] = @user.name + " has been denied access to CUI. They will be removed from the pending list ..."
        elsif params[:decision] == "approved"
          @user.authorization_state = "authorized"
          flash.now[:notice] = @user.name + " has been granted access to CUI. "
        end
        @user.action_token= ""
        @user.save
        UserMailer.access_response(@user).deliver_now

        # remove any pending alerts regarding this access request
        Alert.complete_access_request(@user.id, current_user.id)
      end
    end
  end


  def access_request
    @user = User.find params[:id]
    @user.action_token = SecureRandom.uuid
    @user.save!
  end

  # Returns a paginated list of User records, based on DataTables
  # parameters passed in.  Params keys include:
  # :start - the offset when querying users in the database
  # :length - the number of records being queried
  # :draw - DataTables internal counter userd to sort async responses
  # :siteid - CUI page, single-site details only: indicates the selected site
  def paged_list
    Rails.logger.debug "PARAMS >>> #{params}"
    if current_user.is_cpp_user?
      render json: datatables_json_for_query(@users,
        methods: [:last_name_first_name, :human_cpp_rolename, :npi_list],
        only: [:id, :last_name_first_name, :human_cpp_rolename, :npi_list]
      )
    else
      if params[:siteid].blank?
        render json: datatables_json_for_query(@users,
          methods: [:last_name_first_name, :list_sites],
          only: [:id, :last_name_first_name, :list_sites]
        )
      else
        site = Site.find params[:siteid]
        user_to_fields_func = -> (u) do
          {
            id: u.id,
            last_name_first_name: u.last_name_first_name,
            roles: u.get_roles_string(site)
          }
        end
        render json: datatables_json_for_query_using_func(@users, user_to_fields_func)
      end
    end
  end

private

  def get_users
    if current_user.is_cpp_user? && current_user.is_app_admin?
      @users = User.cpp_users.by_last_name
    elsif current_user.is_app_admin?
      @users = User.cui_users.by_last_name
    else
      redirect_to :root, flash: {alert: "Your privileges are insufficient"}
    end
  end

end # class PersonController < ApplicationController
