class ConsultationsController < ApplicationController
  include DataTablePaginator

  before_action :set_consultation, only: [:show, :edit, :update, :destroy, :add_referral]

  # The before_action methods below are nested; methods should only call one action below.
  before_action :set_consultation_prefs, only: :index
  before_action :set_filtered_consultations, only: :paged_list
  before_action :set_filtered_consultations_and_user_prefs, only: :search

  # hook up authorization and user permissions
  check_authorization
  authorize_resource

  # GET /consultations
  # GET /consultations.json
  def index
    # setting pre-loaded dropdown options before index page is rendered
    if @consultation_filter['facility_id'].blank?
      @facilities = Facility.none
    else
      @facilities = Visn.find(@consultation_filter["visn_id"]).facilities
    end
  end


  # GET /consultations/1
  # GET /consultations/1.json
  def show
  end


  # GET /consultations/new
  def new
    @consultation = Consultation.new(
        veteran_first_name: params[:first_name],
        veteran_last_name: params[:last_name],
        veteran_middle_name: params[:middle_name],
        veteran_ssn: params[:ssn],
        veteran_mpi_pid: params[:mpi_pid],
        veteran_local_pid: params[:local_pid],
        consultation_number: params[:consult_id],
        consultation_status_id: ConsultationStatus.find_by_title('New').id,
        consultation_title: params[:consult_name],
        veteran_dob: params[:dob],
        via_created_date: params[:consult_created_at],
        consultation_text: params[:consult_text],
        veteran_other_health_insurance_name: params[:other_health_insurance_name])
    @veteran_ssn = nil
  end


  # GET /consultations/1/edit
  def edit
  end


  # POST /consultations
  # POST /consultations.json
  def create
    @consultation = Consultation.new(consultation_params)
    @consultation = urgent_priority(@consultation)
    if !consultation_params[:valid_from].blank?
      begin
        @consultation[:valid_from]  = DateTime.strptime(consultation_params[:valid_from],'%m/%d/%Y')
        @consultation[:valid_to]    = DateTime.strptime(consultation_params[:valid_to],'%m/%d/%Y')
      rescue
      end
    end
    respond_to do |format|
      if @consultation.save
        format.html { redirect_to @consultation, notice: 'A new consultation was successfully created.' }
        format.json { render action: 'show', status: :created, location: @consultation }
      else
        @consultation.valid_from = @consultation.valid_from.to_date.strftime('%m/%d/%Y') if @consultation.valid_from
        @consultation.valid_to = @consultation.valid_to.to_date.strftime('%m/%d/%Y') if @consultation.valid_to
        @veteran_ssn = params[:veteran_ssn]
        format.html { render action: 'new', veteran_ssn: @veteran_ssn }
        format.json { render json: @consultation.errors, status: :unprocessable_entity }
      end
    end
  end


  # add new referral to consultation with basic params (referral typem auth #, provider)
  # and preset params: arandomized referral number, status set to Pending, and default reason ID
  def add_referral
    clean_params = referral_params.merge!(coordinator_id: current_user.id, content: {},
      referral_number: (rand 999), referral_status_id: 1, referral_reason_id: 1)
    clean_params.merge!(provider_id: @consultation.ordering_provider_id) if referral_params[:provider_id].empty?
    referral = Referral.where(authorization_number: clean_params[:authorization_number])
    if referral.empty? && !(clean_params[:referral_type_id].empty?)
      new_referral = @consultation.referrals.create(clean_params) if referral.empty?

      respond_to do |format|
        if new_referral
          auth_num = new_referral.authorization_number
          success_notice = "Referral with authorization number #{auth_num} was successfully created."
          format.html do
            auth_num = new_referral.authorization_number
            redirect_to @consultation, notice: success_notice
          end
          format.json do
            provider_data = ProviderDataPresenter.new(new_referral)
            new_referral_data = {
              referral_type: new_referral.referral_type.title,
              authorization_number: new_referral.authorization_number,
              provider_name: provider_data.name,
              NPI: provider_data.npi,
              date: new_referral.created_at.strftime('%F'),
              referral_status: new_referral.referral_status.name
            }
            render json: {data: new_referral_data, message: success_notice, id: new_referral.id}, status: :created
          end
        else
          format.html { redirect_to @consultation, error: @consultation.errors }
          format.json { render json: {error: @consultation.errors}, status: :unprocessable_entity }
        end
      end
    elsif clean_params[:referral_type_id].empty?
      respond_to do |format|
        format.html { redirect_to @consultation, error: "Please select Referral type" }
        format.json { render json: {error: "Please select Referral type"}, status: :unprocessable_entity }
      end
    elsif !(referral.empty?)
      respond_to do |format|
        format.html { redirect_to @consultation, error: "The Authorization number entered is already in use. Please enter a unique Authorization Number." }
        format.json { render json: {error: "The Authorization number entered is already in use. Please enter a unique Authorization Number."}, status: :unprocessable_entity }
      end
    end
  end


  # PATCH/PUT /consultations/1
  # PATCH/PUT /consultations/1.json
  def update
    formatted_params = consultation_params
    formatted_params = format_update_parameters(formatted_params)
    respond_to do |format|
      if @consultation.update(formatted_params)
        format.html { redirect_to @consultation, notice: 'Consultation was successfully updated.' }
        format.json { head :no_content }
      else
        format.html { render action: 'show' }
        format.json { render json: @consultation.errors, status: :unprocessable_entity }
      end
    end
  end


  # DELETE /consultations/1
  # DELETE /consultations/1.json
  def destroy
    @consultation.destroy
    respond_to do |format|
      format.html { redirect_to consultations_url }
      format.json { head :no_content }
    end
  end

  def reset_filter
    save_to_user_preference({})
    render json: {success: @saved}
  end

  def search
    render json: {success: @saved}
  end


  def export_consultations
    respond_to do |format|
      format.csv { send_data Consultation.to_csv(current_user.user_preference.consultation_filter) }
    end
  end

  # Returns a paginated list of Consultation records, based on DataTables
  # parameters passed in.  Params keys include:
  # :start - the offset when querying users in the database
  # :length - the number of records being queried
  # :draw - DataTables internal counter userd to sort async responses
  def paged_list
    Rails.logger.debug "PARAMS >>> #{params}"

    consultation_to_fields_func = -> (consultation) do
      {
        id:                 consultation.id,
        patient_name:       consultation.veteran_full_name,
        ssn:                consultation.hidden_formatted_ssn,
        episode_of_care:    consultation.care_category.title,
        status:             consultation.status_name,
        ordering_physician: ProviderDataPresenter.new(consultation).name,
        date:               consultation.created_at.strftime(DATE_FORMAT)
      }
    end
    ordered_consultations = apply_ordering_to_query(
      @filtered_consultations, Consultation::FIELDS_FOR_RESULTSET, params['order']
    )
    render json: datatables_json_for_query_using_func(
      ordered_consultations, consultation_to_fields_func
    )
  end

  # Ajax method to retrieve Ordering Physician autocomplete options.
  # Search term is passed in as 'term' param
  def physician_select
    providers = Provider.none
    if param_present?(:term)
      if param_present?(:facility_id)
        providers = Facility.find(params['facility_id'])
                            .providers
                            .where("providers.physician_name ilike ?", "#{params['term']}%")
      elsif param_present?(:visn_id)
        providers = Provider.filter visn_id: params['visn_id'], physician_name: params['term']
      end
    end

    # returning values as JSON-ified (id, name) key-value pairs.
    # NOTE: performing pluck/uniq because providers table in Postgres doesn't
    # like distinct() and pluck is faster for small numbers of columns.
    render json: providers.order('providers.name asc')
                          .limit(50)
                          .pluck('providers.id, providers.physician_name')
                          .uniq()
                          .map {|arr| {id: arr[0], name: arr[1]} }
                          .to_json
  end

  # If VIA_ACTIVE_FOR_CPP: Search VISTA for consultations to be created locally.
  # Otherwise, create a temporary consult with "New User" as name, and user can
  #   fill out the other fields.
  # Returns JSON with format: {"data": [<list of Consultation json objects>] }
  def get_patient_consult_details
    merged_consultations = []
    consult_details = []
    temp_consult = Consultation.new(veteran_ssn: params[:ssn])
    veteran = temp_consult.veteran(session[:vista])

    # If Veteran lookup with Vista fails then the ViaApiError is added to the consultation errors and nil is returned.
    vet_error = temp_consult.errors.messages[:via_api_error] if veteran.nil?

    begin
      if veteran.present? && vet_error.blank?
        if VIA_ACTIVE_FOR_CPP
          consult_details = VIA_API::EmrService.get_consults_for_patient(veteran, session[:vista])
        else
          consult_details = [{
            id: veteran.generate_consultation_number,
            date_of_birth: veteran.date_of_birth,
            city: '',
            zip: '',
            timestamp: Time.now.strftime(TIMESTAMP_FORMAT),
            title: 'New Consultation created in CPP',
            text: "Consultation created in CPP for patient: #{veteran.last_name}, #{veteran.first_name}"
          }]
        end

        consult_details.each do |consult_detail|
          # hashifying veteran and consult_detail, and merging the two hashes
          merged_consultations << JSON.parse(veteran.to_json).merge(JSON.parse(consult_detail.to_json))
        end

        render json: {data: merged_consultations, status: :ok}, status: :ok, message: :success
      elsif vet_error.present?
        render json: {error: vet_error, status: :error}, status: :unprocessable_entity
      else
        render json: {error: "Veteran with SSN #{params[:ssn]} not found.", status: :error}, status: :unprocessable_entity
      end

    rescue VIA_API::ViaApiError => e
      render json: {error: e.to_s, status: :error}, status: :unprocessable_entity

    end

  end


  #########################################################################
  ## Don't Peek at my Privates!!

private

  # Basic before_action: only set consultation filter preferences.
  def set_consultation_prefs
    @consultation_filter = current_user.user_preference.consultation_filter
  end

  # Set consultation filter preferences and filtered list.
  def set_filtered_consultations
    set_consultation_prefs
    @filtered_consultations = Consultation.filter(@consultation_filter)
  end

  # Save search params to user preferences first, then set consultation filter
  # preferences and filtered list.
  def set_filtered_consultations_and_user_prefs
    save_to_user_preference
    set_filtered_consultations
  end

  # Save parameters (passed in or from HTTP request hash) to user preferences.
  # The 'params' hash passed in from the request is used as the default.
  # Pass in an empty hash to reset prefs.
  def save_to_user_preference(consultation_params = nil)
    consultation_params ||= consultation_filter_params.to_h
    consultation_params['status'] ||= []
    @saved = current_user.user_preference.save_filter(:consultation, consultation_params)
    current_user.reload if @saved
  end

  # Set up existing consultation, and create temporary new referral for
  # rendering "Add New Referral" form
  def set_consultation
    if params.key? :consultation_id
      @consultation = Consultation.find(params[:consultation_id])
    else
      @consultation = Consultation.find(params[:id])
    end
    @referral = Referral.new
    @ordering_provider = ProviderDataPresenter.new(@consultation)
  end

  # Never trust parameters from the scary internet, only allow the white list through.
  def consultation_params
    params.require(:consultation).permit(:consultation_number,
      :care_category_id, :urgent_priority, :valid_from, :valid_to,
      :consultation_order_id, :veteran_id, :ordering_physician_name,
      :requesting_provider_telephone, :requesting_provider_fax,
      :requesting_provider_contact_name, :consultation_status_id,
      :ordering_provider_id, :veteran_ssn, :veteran_mpi_pid, :veteran_local_pid, :veteran_first_name, :veteran_last_name,
      :veteran_middle_name, :consultation_title,:veteran_dob, :via_created_date, :consultation_text,
      :veteran_other_health_insurance_name)
  end

  # Never trust parameters from the scary internet, only allow the white list through.
  def consultation_filter_params
    params.permit(:from_date, :to_date, :first_name, :last_name, :ssn,
                  :coordinator_id, :care_category_id, :ordering_provider_text,
                  :ordering_provider_id, :consultation_number, :visn_id,
                  :facility_id, :authorization_number, {status: []})
  end

  def referral_params
    params.require(:referral).permit(:referral_type_id,
      :authorization_number, :provider_id)
  end

  #Changing urgent priority to boolean
  def urgent_priority(consultation)
    consultation.urgent_priority = (consultation.urgent_priority.to_s.downcase == "true")
    return consultation
  end
  #End of changing urgent priority to boolean

  def format_update_parameters(parameters)
    parameters[:urgent_priority] = (parameters[:urgent_priority].to_s.downcase == "true")
    unless parameters[:valid_from].blank?
      begin
        parameters[:valid_from] = DateTime.parse(Time.strptime(parameters[:valid_from],'%m/%d/%Y').to_s)
        parameters[:valid_to]   = DateTime.parse(Time.strptime(parameters[:valid_to],'%m/%d/%Y').end_of_day.to_s)
      rescue
        Rails.logger.error "Date parameters (from #{parameters[:valid_from]} " +
                           "to #{parameters[:valid_to]}) are invalid."
      end
    end

    return parameters
  end

  def param_present?(key)
    params[key.to_s].present?
  end

end # class ConsultationsController < ApplicationController
