require 'rails_helper'

require 'via_api.rb'

RSpec.describe 'VIA_API::FindPatientService', cpp: true do

  before do
    WebMock.enable!
    login_via_response = Rails.root.join("spec/support/via_api/login_via_response.xml").read
    mpi_lookup_response = Rails.root.join("spec/support/via_api/mpi_lookup_response.xml").read
    match_by_name_city_state_response = Rails.root.join("spec/support/via_api/match_by_name_city_state_response.xml").read
    url  = VIA_API::VIA_FINDPATIENT_ENDPOINT_URL
    login_url = VIA_API::VIA_EMR_ENDPOINT_URL
    stub_request(:post, login_url).with { |request| request.body.include?("loginVIA") }.to_return(status: 200, body: login_via_response)
    stub_request(:post, url).with { |request| request.body.include?("mpiLookup") }.to_return(status: 200, body: mpi_lookup_response)
    stub_request(:post, url).with { |request| request.body.include?("matchByNameCityStateMs") }.to_return(status: 200, body: match_by_name_city_state_response)
    params = { access_code: "access", verify_code: "verify", site_code: "site" }
    @vista_session = { vista: VIA_API.authenticate(params)}
  end

  after do
    WebMock.disable!
  end

  describe 'methods', cpp: true do

    before :context do
      @veteran = VIA_API::Models::Veteran.new( {:first_name=>"FAITH",
                                        :middle_name=>"K",
                                        :last_name=>"AARSVOLD",
                                        :date_of_birth=>"07/16/1942",
                                        :address=>["712 S OAK", nil, nil],
                                        :city=>"KIMBALL",
                                        :county=>"65",
                                        :state=>"NEBRASKA",
                                        :zip=>"69145",
                                        :phone=>"(444) 444-4444",
                                        :ssn=>"101012294",
                                        :gender=>"F",
                                        :age=>"74",
                                        :patient_name=>"AARSVOLD,FAITH K",
                                        :mpi_pid=>"1009609054",
                                        :mpi_checksum=>"897943",
                                        :local_pid=>"7177262",
                                        :confidentiality=>{:tag=>"0"},
                                        :patient_flags=>{:count=>"0"}
                                       } )
    end

    let(:ssn) {"123456789"}

    it 'Performs the mpi_lookup operation' do
      veterans = VIA_API::FindPatientService.mpi_lookup(ssn,@vista_session[:vista])
      expect(veterans.first).to eq @veteran
    end

    it 'Performs the match_by_name_city_state operation' do
      veterans = VIA_API::FindPatientService.match_by_name_city_state(@vista_session[:vista])
      expect(veterans.first).to eq @veteran
    end


    it "should return all portions of the address", cpp: true do
      expect(@veteran.address_attrs[:address][0]).to eq "712 S OAK"
      expect(@veteran.address_attrs[:city]).to eq "KIMBALL"
      expect(@veteran.address_attrs[:state]).to eq "NEBRASKA"
      expect(@veteran.address_attrs[:zip]).to eq "69145"
    end

    it "should return the full name", cpp: true do
      expect(@veteran.full_name).to eq "FAITH K AARSVOLD"
    end

    it "should return a formatted ssn", cpp: true do
      expect(@veteran.formatted_ssn).to eq "101-01-2294"
    end

    # FIXME:
    # it "should return the status of other health insurance", cpp: true do

    # end

    it "should return the first name,middle name & last name in a array", cpp:true do
      expect(VIA_API::Models::Veteran.split_name(@veteran[:patient_name])).to eq [@veteran.first_name, @veteran.middle_name, @veteran.last_name]
    end

    # FIXME: Update the test cases for filtered_veterans method
    # it "should return all veterans with blank filter", cpp: true do

    # end

    # it "should filter veterans by ssn", cpp: true do

    # end

    # it "should filter veterans by first name", cpp: true do

    # end

    # it "should filter veterans by last name", cpp: true do

    # end

    # it "should filter veterans by date", cpp: true do

    # end

  end
end
