require 'rails_helper'

RSpec.describe EarDiseaseAndOtherSenseOrganEvaluationPresenter do
  context "IEPD compliance" do
    let(:iepd) {
      {'Chronic cholesteatoma' => {
        symptoms: ['During suppuration', 'With aural polyps']},
      'Chronic mastoiditis' => {
        symptoms: ['During suppuration', 'With aural polyps']},
      'Chronic suppurative otitis media' => {
        symptoms: ['During suppuration', 'With aural polyps']},
      'Loss of auricle' => {
        symptoms: ['Complete loss of auricle for left ear',
                  'Complete loss of auricle for right ear',
                  'Deformity of one, with loss of less of one-third or more of the substance for right ear',
                  'Deformity of one, with loss of less of one-third or more of the substance for left ear',
                  'Deformity of one, with loss of one-third or more of the substance for right ear',
                  'Deformity of one, with loss of one-third or more of the substance for left ear']},
      'Meniere\'s Syndrome (endolymphatic hydrops)' => {
        symptoms: ['Hearing impairment and attacks of vertigo and cerebellar gait '\
                  'occurring from one to four times a month, with or without tinnitus',
                  'Hearing impairment and attacks of vertigo and cerebellar gait '\
                  'occurring more than once weekly, with or without tinnitus',
                  'Hearing impairment and attacks of vertigo occurring less than '\
                  'once per month, with or without tinnitus']},
      'Peripheral vestibular disorders' => { symptoms: ['Occasional dizziness', 'Occasional staggering']},
      'Sense of smell' => { symptoms: ['Complete loss', 'Partial loss']},
      'Sense of taste' => { symptoms: ['Complete loss', 'Partial loss']},
      'Tympanic membrane, perforation' => { symptoms: ['Perforated tympanic membrane']},
      'Tinnitus' => { symptoms: ['Recurrent', 'Single occurrence']},
    }}

    it "should have an exam with each title in the iepd" do
      iepd.each do |title, val|
        expect(EXAMS).to have_key(title), title
      end
    end

    it "should have each 'symptom' in a rating calculator" do
      iepd.each do |title, values|
        exam = EXAMS[title]
        values[:symptoms].each do |symptom|
          expect(exam).to have_a_symptom(symptom)
        end
      end
    end

    it "should have each 'ratingCalculator' in a symptom listed above" do
      iepd.each do |title, iepd_values|
        expect(EXAMS[title]).to only_contain_symptoms_from(iepd_values[:symptoms])
      end
    end
  end

  before do
    schema_xml = XML::Document.file(Rails.root.join("app/xsd/ExamResponse/exchange/ExamResponse-2.0.xsd").to_s)
    @schema = XML::Schema.document(schema_xml)
    @claim = create(:claim, date_of_birth: Date.new(1900))
    @contention = create(:contention)
    @claim.contentions << @contention
    allow_any_instance_of(Evaluation).to receive(:to_pdf) { "" }

    @evaluation_spec = EvaluationSpec.create(spec: File.read("app/exams/tinnitus.yml"))
    @evaluation = Evaluation.create(claim: @claim, evaluation_spec: @evaluation_spec, user: create(:user))
    @contention.evaluations << @evaluation

    @spec_json = File.read("spec/evaluations/tinnitus.json")
    @evaluation.doc = @spec_json
  end

  # it "should generate valid XML for an ear exam" do
  #   doc = XML::Document.string(@evaluation.to_xml)
  #   doc.validate_schema(@schema)
  # end

  context "when the eval is secondary, we output its parent as well" do
    before do
      user = create(:user)
      claim = create(:claim)
      claim.contentions << create(:contention, history: "This or That")

      dependent_eval = Evaluation.new(claim: claim, evaluation_spec: @evaluation_spec, user: user)
      dependent_eval.doc = @spec_json
      claim.contentions.first.evaluations << dependent_eval

      parent_eval = Evaluation.new(claim: claim, evaluation_spec: @evaluation_spec, completed_at: Time.now, user: user)
      parent_eval.doc = @spec_json
      claim.contentions.first.evaluations << parent_eval

      dependent_eval.primary_evaluation_id = parent_eval.id
      dependent_eval.save

      @evaluation_xml = dependent_eval.to_xml
    end

    # it "should include the parent eval" do
    #   xml = XML::Document.string(@evaluation_xml)
    #   facts = xml.find("//aaf:auditoryAcuityFacts")
    #
    #   expect(facts.length).to eq 2
    #
    #   schema_xml = XML::Document.file(Rails.root.join("app/xsd/ExamResponse/exchange/ExamResponse-2.0.xsd").to_s)
    #   schema = XML::Schema.document(schema_xml)
    #   xml.validate_schema(schema)
    # end
  end
end
