require 'rails_helper'

RSpec.describe CardiovascularEvaluationPresenter do
  before do
    schema_xml = XML::Document.file(Rails.root.join("app/xsd/ExamResponse/exchange/ExamResponse-2.0.xsd").to_s)
    @schema = XML::Schema.document(schema_xml)

    @claim = create(:claim, date_of_birth: Date.new(1900))

    @user = create(:user)
    @contention = create(:contention)
    @claim.contentions << @contention
    @spec_json = File.read("spec/evaluations/cardio.json")

    @evaluation_spec = EvaluationSpec.create(spec: File.read("app/exams/aneurysm-any-larger-artery-evaluation.yml"))

    allow_any_instance_of(Evaluation).to receive(:to_pdf) { "" }
  end

  context "when the eval is secondary, we output its parent as well" do
    before do
      @claim.contentions << create(:contention, history: "This or That")

      dependent_eval = Evaluation.new(claim: @claim, evaluation_spec: @evaluation_spec, user: @user)
      dependent_eval.doc = @spec_json
      @claim.contentions.first.evaluations << dependent_eval

      parent_eval = Evaluation.new(claim: @claim, evaluation_spec: @evaluation_spec, completed_at: Time.now, user: @user)
      parent_eval.doc = @spec_json
      @claim.contentions.first.evaluations << parent_eval

      dependent_eval.primary_evaluation_id = parent_eval.id
      dependent_eval.save

      @evaluation_xml = dependent_eval.to_xml(false)
      @doc = XML::Document.string(@evaluation_xml)
    end

    it "should include the parent eval" do
      expect(@doc.find("//cf:cardiovascularFacts").length).to eq 2
    end

    # it "should be valid" do
    #   @doc.validate_schema(@schema)
    # end
  end

  context "when there is no contraindication field" do
    before do
      cardio_spec = EvaluationSpec.create(spec: File.read("app/exams/cardiomyopathy.yml"))
      cardio_json = File.read("spec/evaluations/cardiomyopathy_no_contra.json")
      evaluation = Evaluation.new(claim: @claim, evaluation_spec: cardio_spec, user: @user)
      evaluation.doc = cardio_json
      @evaluation_xml = Nokogiri::XML(evaluation.to_xml(false))
    end

    it "should not have a contraindication field" do
      expect(@evaluation_xml.xpath("//cf:contraindication").empty?).to eq true
    end

    # it "should be valid" do
    #   xml = XML::Document.string(@evaluation_xml.to_s)
    #   xml.validate_schema(@schema)
    # end
  end

  context "when there is a false contraindication field" do
    before do
      cardio_spec = EvaluationSpec.create(spec: File.read("app/exams/cardiomyopathy.yml"))
      cardio_json = File.read("spec/evaluations/cardiomyopathy_contra.json")
      @evaluation = Evaluation.new(claim: @claim, evaluation_spec: cardio_spec, user: @user)
      @evaluation.doc = cardio_json
      @evaluation_xml = Nokogiri::XML(@evaluation.to_xml(false))
    end

    it "should have a contraindication field with 'No'" do
      expect(@evaluation_xml.xpath("//cf:contraindication")[0].text).to eq "No"
    end

    # it "should be valid" do
    #   xml = XML::Document.string(@evaluation_xml.to_s)
    #   xml.validate_schema(@schema)
    # end
  end

  context "when there is a true contraindication field" do
    before do
      cardio_spec = EvaluationSpec.create(spec: File.read("app/exams/cardiomyopathy.yml"))
      cardio_json = JSON.load(File.read("spec/evaluations/cardiomyopathy_contra.json"))

      # set contraindication field to true
      cardio_json["form"]["fieldsets"][0]["fieldset"][31]["field"]["value"] = true
      @evaluation = Evaluation.new(claim: @claim, evaluation_spec: cardio_spec, user: @user, doc: cardio_json.to_json)
      @evaluation_xml = Nokogiri::XML(@evaluation.to_xml(false))
    end

    it "should have a contraindication field with 'Yes'" do
      expect(@evaluation_xml.xpath("//cf:contraindication")[0].text).to eq "Yes"
    end

    # it "should be valid" do
    #   xml = XML::Document.string(@evaluation_xml.to_s)
    #   xml.validate_schema(@schema)
    # end
  end

  context "when there is an symptomatic field in a small artery exam" do
    context "when it is false" do
      before do
        aneurysm_spec = EvaluationSpec.create(spec: File.read("app/exams/aneurysm-small-artery.yml"))
        aneurysm_json = File.read("spec/evaluations/aneurysm_small_artery.json")

        @evaluation = Evaluation.new(claim: @claim, evaluation_spec: aneurysm_spec, user: @user, doc: aneurysm_json)
        @evaluation_xml = Nokogiri::XML(@evaluation.to_xml(false))
      end

      it "should say 'asymptomatic'" do
        node = @evaluation_xml.xpath('//cf:cardiovascularSymptoms/cf:value[text()="Asymptomatic"]')
        expect(node.length).to eq 1
      end

      # it "should be valid" do
      #   xml = XML::Document.string(@evaluation_xml.to_s)
      #   xml.validate_schema(@schema)
      # end
    end

    context "when it is true" do
      before do
        aneurysm_spec = EvaluationSpec.create(spec: File.read("app/exams/aneurysm-small-artery.yml"))
        aneurysm_json = JSON.load(File.read("spec/evaluations/aneurysm_small_artery.json"))

        aneurysm_json["form"]["fieldsets"][0]["fieldset"][5]["field"]["value"] = true

        @evaluation = Evaluation.new(claim: @claim, evaluation_spec: aneurysm_spec, user: @user, doc: aneurysm_json)
        @evaluation_xml = Nokogiri::XML(@evaluation.to_xml(false))
      end

      it "should not say 'asymptomatic'" do
        node = @evaluation_xml.xpath('//cf:cardiovascularSymptoms/cf:value[text()="Asymptomatic"]')
        expect(node.length).to eq 0
      end

      # it "should be valid" do
      #   xml = XML::Document.string(@evaluation_xml.to_s)
      #   xml.validate_schema(@schema)
      # end
    end
  end

  context "when there is a selectExtremity field" do
    context "when it is nil" do
      before do
        aneurysm_spec = EvaluationSpec.create(spec: File.read("app/exams/aneurysm-any-larger-artery-evaluation.yml"))
        aneurysm_json = JSON.load(File.read("spec/evaluations/aneurysm_large_artery.json"))
        aneurysm_json["form"]["fieldsets"][0]["fieldset"][6]["field"]["value"] = nil

        @evaluation = Evaluation.new(claim: @claim, evaluation_spec: aneurysm_spec, user: @user, doc: aneurysm_json)
        @evaluation_xml = Nokogiri::XML(@evaluation.to_xml(false))
      end

      it "should not have a bsf:bilateralType field" do
        expect(@evaluation_xml.xpath("//bsf:bilateralType").length).to eq 0
      end

      # it "should be valid" do
      #   xml = XML::Document.string(@evaluation_xml.to_s)
      #   xml.validate_schema(@schema)
      # end
    end

    context "when it is some value" do
      before do
        aneurysm_spec = EvaluationSpec.create(spec: File.read("app/exams/aneurysm-any-larger-artery-evaluation.yml"))
        aneurysm_json = File.read("spec/evaluations/aneurysm_large_artery.json")

        @evaluation = Evaluation.new(claim: @claim, evaluation_spec: aneurysm_spec, user: @user, doc: aneurysm_json)
        @evaluation_xml = Nokogiri::XML(@evaluation.to_xml(false))
      end

      it "should have a bsf:bilateralType field" do
        expect(@evaluation_xml.xpath("//bsf:bilateralType").length).to eq 1
      end

      # it "should be valid" do
      #   xml = XML::Document.string(@evaluation_xml.to_s)
      #   xml.validate_schema(@schema)
      # end
    end
  end
end
