# == Schema Information
#
# Table name: referrals
#
#  id                   :integer          not null, primary key
#  coordinator_id       :integer          not null
#  referral_number      :string           not null
#  authorization_number :string           not null
#  content              :json
#  created_at           :datetime         not null
#  updated_at           :datetime         not null
#  consultation_id      :integer
#  referral_status_id   :integer
#  referral_type_id     :integer
#  referral_reason_id   :integer
#  provider_id          :integer
#

require 'rails_helper'

RSpec.describe Referral, :type => :model do

  context "when creating associations" do

    before(:context) do
      @veteran = create(:veteran)
      @consultation = create(:consultation, veteran: @veteran)
    end

    it "should have an associated veteran and consultation", cpp: true do
      referral = create(:referral, consultation: @consultation)
      expect(referral.consultation.consultation_number).to eq @consultation.consultation_number
      expect(referral.consultation.veteran(@vista_session).full_name).to eq @veteran.full_name
    end

  end # context "when creating associations"


  context "when validating referrals" do

    it "should not create referrals with blank referral type", cpp: true do
      expect {
        create(:referral, referral_type: nil)
      }.to raise_error(ActiveRecord::RecordInvalid)
    end

    it "should not create referrals with blank coordinator id", cpp: true do
      expect {
        create(:referral, coordinator_id: nil)
      }.to raise_error(ActiveRecord::RecordInvalid)
    end

    it "should not create referrals with blank authorization numbers", cpp: true do
      expect {
        create(:referral, authorization_number: nil)
      }.to raise_error(ActiveRecord::RecordInvalid)
    end

    it "should not create multiple referrals with the same authorization number", cpp: true do
      create(:referral, authorization_number: 'TEST-67890')
      expect {
        create(:referral, authorization_number: 'TEST-67890')
      }.to raise_error(ActiveRecord::RecordInvalid)
    end

  end # context "when validating referrals"


  context "when performing filtering" do

    # creating test data used for referral filtering tests.
    # Test data is used for VHA CC unit tests, and appears when
    # performing non-VHA tests but generally functions as 'white noise'.
    before(:context) do
      @test_visn = Visn.where(name: 'Rocky Mountain Network').first
      @test_care_category = CareCategory.where(title: 'Neurology').first
      @test_referral_type = ReferralType.where(title: 'Radiology').first
      @test_provider_with_visn = create(:provider, :with_visn_by_name, visn_name: 'Rocky Mountain Network')

      @default_veteran = create(:veteran)
      @veteran_with_testname = create(:veteran, first_name: 'Marge', last_name: 'Simpson')
      @veteran_with_ssn = create(:veteran, first_name: 'Test2SSN', last_name: 'User2SSN', ssn: '777444111')

      @vha_cc_user = create(:vha_user)
      @non_vha_user = create(:community_provider)

      @consultations = {
        default:                  create(:consultation, veteran: @default_veteran),
        urgent:                   create(:consultation, :urgent, veteran: @default_veteran),
        recently_no_longer_valid: create(:consultation, :recently_no_longer_valid, veteran: @default_veteran),
        no_longer_valid:          create(:consultation, :no_longer_valid, veteran: @default_veteran),
        with_testname:            create(:consultation, veteran: @veteran_with_testname),
        with_test_ssn:            create(:consultation, veteran: @veteran_with_ssn),
        with_consult_number:      create(:consultation, veteran: @default_veteran,
                                    consultation_number: 'GHI-123'),
        with_care_category:       create(:consultation, veteran: @default_veteran,
                                    care_category: @test_care_category)
      }

      @referrals = {
        recently_no_longer_valid: create(:referral, :complete,
                                    consultation: @consultations[:recently_no_longer_valid]),
        no_longer_valid:          create(:referral, :complete,
                                    consultation: @consultations[:no_longer_valid]),
        with_testname:            create(:referral, consultation: @consultations[:with_testname]),
        with_test_ssn:            create(:referral, consultation: @consultations[:with_test_ssn]),
        with_consult_number:      create(:referral, consultation: @consultations[:with_consult_number]),
        with_care_category:       create(:referral, consultation: @consultations[:with_care_category]),
        with_test_referral_type:  create(:referral, consultation: @consultations[:default],
                                    referral_type: @test_referral_type),
        with_coordinator:         create(:referral, consultation: @consultations[:default],
                                    coordinator: @vha_cc_user),
        with_auth_number:         create(:referral, consultation: @consultations[:default],
                                    authorization_number: 'GHI-654-321'),
        with_test_provider:       create(:referral, consultation: @consultations[:default],
                                    provider: @test_provider_with_visn),
        review_pending:           create(:referral, :review_pending,
                                    consultation: @consultations[:default]),
        new:                  create(:referral, :new,
                                    consultation: @consultations[:default]),
        with_facility_name:   create(:referral, provider: create(:provider, :with_facility_and_visn_number,
                                                               facility_name: 'TestFacility', visn_number: 15 ))

      }

      @visible_referrals = @referrals.reject do |key, referral|
        referral == @referrals[:no_longer_valid]
      end
    end


    context "for VHA CC users" do

      it "should return all referrals in collection when no filter params are passed in", cpp: true do
        results = Referral.filter({}, @vha_cc_user)
        expect(results).to include_only_values_with_listed_keys(@visible_referrals.keys, @referrals)
      end

      it "should filter based on veteran name", cpp: true do
        results = Referral.filter({'first_name' => @veteran_with_testname.first_name}, @vha_cc_user)
        expect(results).to include_only_values_with_listed_keys([:with_testname], @referrals)
      end

      it "should filter based on veteran last name", cpp: true do
        results = Referral.filter({'last_name' => @veteran_with_testname.last_name}, @vha_cc_user)
        expect(results).to include_only_values_with_listed_keys([:with_testname], @referrals)
      end

      it "should return no records when filtering on first name not in database", cpp: true do
        results = Referral.filter({'first_name' => 'asdfrgasgasghadsfghafsg'}, @vha_cc_user)
        expect(results).to be_empty
      end

      it "should return no records when filtering on last name not in database", cpp: true do
        results = Referral.filter({'last_name' => 'asdfrgasgasghadsfghafsg'}, @vha_cc_user)
        expect(results).to be_empty
      end

      it "should filter based on veteran SSN", cpp: true do
        results = Referral.filter({'ssn' => '777444111'}, @vha_cc_user)
        expect(results).to include_only_values_with_listed_keys([:with_test_ssn], @referrals)
      end

      it "should return no records when filtering on SSN not in database", cpp: true do
        results = Referral.filter({'ssn' => '777000000'}, @vha_cc_user)
        expect(results).to be_empty
      end

      it "should filter based on authorization number", cpp: true do
        results = Referral.filter({'authorization_number' => 'GHI-654-321'}, @vha_cc_user)
        expect(results).to include_only_values_with_listed_keys([:with_auth_number], @referrals)
      end

      it "should return no records when filtering on authorization number not in database", cpp: true do
        results = Referral.filter({'authorization_number' => 'NOT-VALID-AUTHNUM'}, @vha_cc_user)
        expect(results).to be_empty
      end

      it "should filter based on consultation number", cpp: true do
        results = Referral.filter({'consultation_number' => 'GHI-123'}, @vha_cc_user)
        expect(results).to include_only_values_with_listed_keys([:with_consult_number], @referrals)
      end

      it "should return no records when filtering on consultation number not in database", cpp: true do
        results = Referral.filter({'consultation_number' => 'NOT-VALID-CONSULT_NUM'}, @vha_cc_user)
        expect(results).to be_empty
      end

      it "should filter based on VISN id", cpp: true do
        results = Referral.filter({'visn_id' => @test_visn.id.to_s}, @vha_cc_user)
        expect(results).to include_only_values_with_listed_keys([:with_test_provider], @referrals)
      end

      it "should return no records when filtering on VISN without referrals", cpp: true do
        visn_not_found = Visn.where(name: 'Sierra Pacific Network').first
        expect(visn_not_found).not_to be_nil

        results = Referral.filter({'visn_id' => visn_not_found.id.to_s}, @vha_cc_user)
        expect(results).to be_empty
      end

      it "should filter based on facility id", cpp: true do
        results = Referral.filter({'facility_id' => Facility.where(name: 'TestFacility').first.id }, @vha_cc_user)
        expect(results).to include_only_values_with_listed_keys([:with_facility_name], @referrals)
      end

      it "should return no records when filtering on facility id not in database", cpp: true do
        results = Referral.filter({'facility_id' => 99999}, @vha_cc_user)
        expect(results).to be_empty
      end

      it "should filter based on referral_type_id", cpp: true do
        results = Referral.filter({'referral_type_id' => @referrals[:with_test_referral_type].referral_type.id}, @vha_cc_user)
        expect(results).to include_only_values_with_listed_keys([:with_test_referral_type], @referrals)
      end


      it "should filter based on provider id", cpp: true do
        results = Referral.filter({'provider_id' => @test_provider_with_visn.id.to_s}, @vha_cc_user)
        expect(results).to include_only_values_with_listed_keys([:with_test_provider], @referrals)
      end

      it "should return no records when filtering on provider containing no referrals", cpp: true do
        dummy_provider_id = create(:provider).id
        results = Referral.filter({'provider_id' => dummy_provider_id}, @vha_cc_user)
        expect(results).to be_empty
      end

      it "should filter based on care category id", cpp: true do
        results = Referral.filter({'care_category_id' => @test_care_category.id.to_s}, @vha_cc_user)
        expect(results).to include_only_values_with_listed_keys([:with_care_category], @referrals)
      end

      it "should return no records when filtering on care category id not found", cpp: true do
        care_category_not_found = CareCategory.where(title: 'Diabetes').first
        expect(care_category_not_found).not_to be_nil

        results = Referral.filter({'care_category_id' => care_category_not_found.id.to_s}, @vha_cc_user)
        expect(results).to be_empty
      end

      it "should filter based on coordinator id", cpp: true do
        results = Referral.filter({'coordinator_id' => @vha_cc_user.id.to_s}, @vha_cc_user)
        expect(results).to include_only_values_with_listed_keys([:with_coordinator], @referrals)
      end

      it "should return no records when filtering on coordinator not in database", cpp: true do
        not_a_coordinator = create(:examiner)
        results = Referral.filter({'coordinator_id' => not_a_coordinator.id.to_s}, @vha_cc_user)
        expect(results).to be_empty
      end

      it "should filter closed referrals and only return referrals closed in the last 30 days", cpp: true do
        complete_status = ReferralStatus.where(name: 'Complete').first
        results = Referral.filter({'status' => [complete_status.id]}, @vha_cc_user)
        expect(results).to include_only_values_with_listed_keys([:recently_no_longer_valid], @referrals)
      end

      it "should filter based on multiple statuses by returning active referrals with any of the included statuses", cpp: true do
        review_status = ReferralStatus.where(name: 'Review Pending').first
        complete_status = ReferralStatus.where(name: 'Complete').first
        results = Referral.filter({'status' => [review_status.id, complete_status.id]}, @vha_cc_user)
        expect(results).to include_only_values_with_listed_keys([:review_pending, :recently_no_longer_valid], @referrals)
      end


      context "performing custom sidebar queries" do

        before(:context) do
          @referrals_for_sidebar = {
            expired: create(:referral, :review_pending,
                        consultation: @consultations[:recently_no_longer_valid]),
            stat:    create(:referral, :review_pending,
                        consultation: @consultations[:urgent]),
            closed_not_stat: create(:referral, :complete,
                        consultation: @consultations[:urgent]),
            with_cprs_appointment: create(:referral, :with_appointments_added_to_cprs,
                        consultation: @consultations[:default]),
            with_non_cprs_appointments: create(:referral, :with_new_appointments,
                        consultation: @consultations[:default]),
          }
          @referrals_including_sidebar = @referrals_for_sidebar.merge(@referrals)
        end

        it "should filter for Expired referrals", cpp: true do
          results = Referral.filter({'expired_data' => true}, @vha_cc_user)
          expect(results).to include_only_values_with_listed_keys([:expired], @referrals_including_sidebar)
        end

        it "should filter for STAT referrals", cpp: true do
          results = Referral.filter({'stat_data' => true}, @vha_cc_user)
          expect(results).to include_only_values_with_listed_keys([:stat], @referrals_including_sidebar)
        end

        it "should filter for referrals with new appointments", cpp: true do
          results = Referral.filter({'new_appointments' => true}, @vha_cc_user)
          expect(results).to include_only_values_with_listed_keys([:with_non_cprs_appointments], @referrals_including_sidebar)
        end

        it "should filter for referrals with appointments requiring medical record letters", cpp: true do
          referrals_including_old_appointments = @referrals.merge({
            one_week_ago_and_incomplete: create(:referral, :with_new_appointments,
                        referral_appointment_time: 1.week.ago,
                        consultation: @consultations[:default]),
            one_week_ago_and_complete: create(:referral, :complete, :with_new_appointments,
                        referral_appointment_time: 1.week.ago,
                        consultation: @consultations[:default])
          })

          results = Referral.filter({'medical_record_letters' => true}, @vha_cc_user)
          expect(results).to include_only_values_with_listed_keys([:one_week_ago_and_incomplete], referrals_including_old_appointments)
        end

        it "should filter for referrals with appointments that need reminder", cpp: true do
          referrals_including_old_appointments = @referrals.merge({
            two_days_ago_and_incomplete: create(:referral, :with_new_appointments,
                        referral_appointment_time: 2.days.ago,
                        consultation: @consultations[:default]),
            two_days_ago_and_complete: create(:referral, :complete, :with_new_appointments,
                                              referral_appointment_time: 2.days.ago,
                                              consultation: @consultations[:default]),
            one_week_ago_and_complete: create(:referral, :complete, :with_new_appointments,
                        referral_appointment_time: 1.week.ago,
                        consultation: @consultations[:default])
          })
          results = Referral.needs_reminder
          expect(results).to include_only_values_with_listed_keys([:two_days_ago_and_incomplete], referrals_including_old_appointments)
        end

        it "should increment the STAT counter when adding STAT referrals, and decrement when destroying STAT referrals", cpp: true do
          old_count = Referral.stat_count(@vha_cc_user)

          stat_referral_added = create(:referral, :review_pending, consultation: @consultations[:urgent])
          expect(Referral.stat_count(@vha_cc_user)).to eq(old_count + 1)

          stat_referral_added.destroy!
          expect(Referral.stat_count(@vha_cc_user)).to eq(old_count)
        end

        it "should increment the expired counter when adding expired referrals, and decrement likewise", cpp: true do
          old_count = Referral.expired_count(@vha_cc_user)

          expired_referral_added = create(:referral, :review_pending, consultation: @consultations[:recently_no_longer_valid])
          expect(Referral.expired_count(@vha_cc_user)).to eq(old_count + 1)

          expired_referral_added.destroy!
          expect(Referral.expired_count(@vha_cc_user)).to eq(old_count)
        end

        it "should increment the new appointments counter when adding referrals with new appointments, and decrement likewise", cpp: true do
          old_count = Referral.with_new_appointments_count

          new_appointments_referral_added = create(:referral, :with_new_appointments,
            consultation: @consultations[:default])
          expect(Referral.with_new_appointments_count).to eq(old_count + 1)

          new_appointments_referral_added.referral_appointments.destroy_all
          expect(Referral.with_new_appointments_count).to eq(old_count)
        end

        it "should not increment the new appointments counter when adding referrals with appointments added to CPRS", cpp: true do
          old_count = Referral.with_new_appointments_count

          referral_with_cprs_appointments_added = create(:referral, :with_appointments_added_to_cprs,
            consultation: @consultations[:default])
          expect(Referral.with_new_appointments_count).to eq(old_count)
        end

        it "should increment the medical letter counter when adding referrals requiring medical letters, and decrement likewise", cpp: true do
          old_count = Referral.med_letter_count(@vha_cc_user)

          med_letter_referral_added = create(:referral, :with_new_appointments,
            referral_appointment_time: 1.week.ago,
            consultation: @consultations[:default])
          expect(Referral.med_letter_count(@vha_cc_user)).to eq(old_count + 1)

          med_letter_referral_added.referral_appointments.destroy_all
          expect(Referral.med_letter_count(@vha_cc_user)).to eq(old_count)
        end

      end # context "performing custom sidebar queries"

    end # context "for VHA CC users"

    context "#methods" do

      describe "#use_ordering_provider" do
        it "should use ordering provider when no provider is given", cpp: true do
          referral = create(:referral)
          expect(referral.use_ordering_provider).to eq true
        end

        it "should not use ordering provider when provider is given", cpp: true do
          referral = create(:referral, provider: create(:provider, :with_visn_by_name,
                                                        visn_name: 'VA Mid-Atlantic Health Care Network'))
          expect(referral.use_ordering_provider).to eq false
        end
      end


      describe "#referral_details_drop_down" do
        it "should return referral details", cpp: true do
          referral = create(:referral, authorization_number: '#ABC-50-5650-4150')
          expect(referral.referral_details_dropdown).to include '#ABC-50-5650-4150'
        end

      end

      describe "#veteran" do
        before(:context) do
          @veteran = create(:veteran, ssn: "123456789",
                           first_name: 'John',
                           middle_name: 'Doe',
                           last_name: "Testuser",
                           other_health_insurance_name: "Insurance Provider")
          @vista_session = {}
          @referral = create(:referral, consultation: create(:consultation, veteran: @veteran))
        end
        it "should return the veteran", cpp: true do
          expect(@referral.veteran(@vista_session).last_name).to eq 'Testuser'
        end
      end

      describe "#visible_documents_for_referral(user)" do
        before(:context) do
          @vha_user = create(:vha_user)
          @non_vha_user = create(:community_provider)
          @other_non_vha_user = create(:community_provider)
          @referral = create(:referral, coordinator: @non_vha_user)
          referral_document1 = create(:referral_document, referral: @referral, uploader_id: @non_vha_user.id,
                                      content: {document_name: 'testdoc1.png'})
        end

        it "should display all self-submitted documents for non-vha user", cpp: true do
          expect(@referral.visible_documents_for_referral(@non_vha_user).count).to eq 1
          expect(@referral.visible_documents_for_referral(@non_vha_user).
              first.content['document_name']).to eq 'testdoc1.png'
        end

      end

    end


    context "for non-VHA community provider users" do

      before(:context) do
        @east_coast_visn = Visn.where(name: 'VA Mid-Atlantic Health Care Network').first
        @midsouth_visn   = Visn.where(name: 'VA MidSouth Healthcare Network').first

        @east_coast_provider = create(:provider, :with_visn_by_name, visn_name: 'VA Mid-Atlantic Health Care Network')
        @midsouth_provider = create(:provider, :with_visn_by_name, visn_name: 'VA MidSouth Healthcare Network')

        @east_coast_non_vha_user = create(:community_provider, providers: [@east_coast_provider])
        @midsouth_non_vha_user = create(:community_provider, providers: [@midsouth_provider])

        # Referrals visible by default for @east_coast_non_vha_user
        @east_coast_referrals = {
          recently_no_longer_valid: create(:referral, :complete,
                                      consultation: @consultations[:recently_no_longer_valid],
                                      provider: @east_coast_provider),
          with_testname:            create(:referral, consultation: @consultations[:with_testname],
                                      provider: @east_coast_provider),
          with_test_ssn:            create(:referral, consultation: @consultations[:with_test_ssn],
                                      provider: @east_coast_provider),
          with_consult_number:      create(:referral, consultation: @consultations[:with_consult_number],
                                      provider: @east_coast_provider),
          with_care_category:       create(:referral, consultation: @consultations[:with_care_category],
                                      provider: @east_coast_provider)
        }

        # Referrals visible by default for @midsouth_non_vha_user
        @midsouth_referrals = {
          with_test_referral_type:  create(:referral, consultation: @consultations[:default],
                                      referral_type: @test_referral_type,
                                      provider: @midsouth_provider),
          with_auth_number:         create(:referral, consultation: @consultations[:default],
                                      authorization_number: 'GHI-987-654',
                                      provider: @midsouth_provider),
          review_pending:           create(:referral, :review_pending,
                                      consultation: @consultations[:default],
                                      provider: @midsouth_provider)
        }
        @referrals_for_both_users = @east_coast_referrals.merge(@midsouth_referrals)
      end

      it "should not display any referrals not from the user's designated provider", cpp: true do
        east_coast_user_results = Referral.filter({}, @east_coast_non_vha_user)
        midsouth_user_results   = Referral.filter({}, @midsouth_non_vha_user)

        # @referrals are the referrals from the original context used by VHA CC tests
        expect(east_coast_user_results).to include_no_values_from_list(@referrals.values)
        expect(midsouth_user_results).to   include_no_values_from_list(@referrals.values)
      end

      it "should return all referrals visible to the user when no filter params are passed in", cpp: true do
        east_coast_user_results = Referral.filter({}, @east_coast_non_vha_user)
        midsouth_user_results   = Referral.filter({}, @midsouth_non_vha_user)

        expect(east_coast_user_results).to include_only_values_with_listed_keys(
          @east_coast_referrals.keys, @referrals_for_both_users
        )
        expect(midsouth_user_results).to   include_only_values_with_listed_keys(
          @midsouth_referrals.keys, @referrals_for_both_users
        )
      end

      it "should not display referrals for consultations that are no longer valid", cpp: true do
        invalid_referral = create(:referral, :complete,
                                  consultation: @consultations[:no_longer_valid],
                                  provider: @east_coast_provider)

        results = Referral.filter({}, @east_coast_non_vha_user)
        expect(results).to_not include(invalid_referral)
      end

      it "should not display referrals that are not yet active", cpp: true do
        new_referral = create(:referral, :new,
                                  consultation: @consultations[:default],
                                  provider: @east_coast_provider)

        results = Referral.filter({}, @east_coast_non_vha_user)
        expect(results).to_not include(new_referral)
      end

      it "should filter based on veteran name, only on referrals visible to the user", cpp: true do
        east_coast_user_results = Referral.filter({'first_name' => @veteran_with_testname.first_name}, @east_coast_non_vha_user)
        midsouth_user_results   = Referral.filter({'first_name' => @veteran_with_testname.first_name}, @midsouth_non_vha_user)

        expect(east_coast_user_results).to include_only_values_with_listed_keys([:with_testname], @referrals_for_both_users)
        expect(midsouth_user_results).to   include_no_values_from_list(@referrals_for_both_users.values)
      end

      it "should filter based on veteran SSN, only on referrals visible to the user", cpp: true do
        east_coast_user_results = Referral.filter({'ssn' => '777444111'}, @east_coast_non_vha_user)
        midsouth_user_results   = Referral.filter({'ssn' => '777444111'}, @midsouth_non_vha_user)

        expect(east_coast_user_results).to include_only_values_with_listed_keys([:with_test_ssn], @referrals_for_both_users)
        expect(midsouth_user_results).to   include_no_values_from_list(@referrals_for_both_users.values)
      end

      it "should filter based on authorization number, only on referrals visible to the user", cpp: true do
        east_coast_user_results = Referral.filter({'authorization_number' => 'GHI-987-654'}, @east_coast_non_vha_user)
        midsouth_user_results   = Referral.filter({'authorization_number' => 'GHI-987-654'}, @midsouth_non_vha_user)

        expect(east_coast_user_results).to include_no_values_from_list(@referrals_for_both_users.values)
        expect(midsouth_user_results).to   include_only_values_with_listed_keys([:with_auth_number], @referrals_for_both_users)
      end

      it "should filter based on consultation number, only on referrals visible to the user", cpp: true do
        east_coast_user_results = Referral.filter({'consultation_number' => 'GHI-123'}, @east_coast_non_vha_user)
        midsouth_user_results   = Referral.filter({'consultation_number' => 'GHI-123'}, @midsouth_non_vha_user)

        expect(east_coast_user_results).to include_only_values_with_listed_keys([:with_consult_number], @referrals_for_both_users)
        expect(midsouth_user_results).to   include_no_values_from_list(@referrals_for_both_users.values)
      end

      it "should filter based on VISN id from corresponding provider, displaying all visible records", cpp: true do
        east_coast_user_results = Referral.filter({'visn_id' => @east_coast_visn.id.to_s}, @east_coast_non_vha_user)
        midsouth_user_results   = Referral.filter({'visn_id' => @midsouth_visn.id.to_s}, @midsouth_non_vha_user)

        expect(east_coast_user_results).to include_only_values_with_listed_keys(
          @east_coast_referrals.keys, @referrals_for_both_users
        )
        expect(midsouth_user_results).to   include_only_values_with_listed_keys(
          @midsouth_referrals.keys, @referrals_for_both_users
        )
      end

      it "should filter based on VISN id not from corresponding provider, displaying no visible records", cpp: true do
        east_coast_user_results = Referral.filter({'visn_id' => @midsouth_visn.id.to_s}, @east_coast_non_vha_user)
        midsouth_user_results   = Referral.filter({'visn_id' => @east_coast_visn.id.to_s}, @midsouth_non_vha_user)

        expect(east_coast_user_results).to be_empty
        expect(midsouth_user_results).to   be_empty
      end

      it "should filter based on provider id from corresponding provider, displaying all visible records", cpp: true do
        east_coast_user_results = Referral.filter({'provider_id' => @east_coast_provider.id.to_s}, @east_coast_non_vha_user)
        midsouth_user_results   = Referral.filter({'provider_id' => @midsouth_provider.id.to_s}, @midsouth_non_vha_user)

        expect(east_coast_user_results).to include_only_values_with_listed_keys(
          @east_coast_referrals.keys, @referrals_for_both_users
        )
        expect(midsouth_user_results).to   include_only_values_with_listed_keys(
          @midsouth_referrals.keys, @referrals_for_both_users
        )
      end

      it "should filter based on provider id not from corresponding provider, displaying no visible records", cpp: true do
        east_coast_user_results = Referral.filter({'provider_id' => @midsouth_provider.id.to_s}, @east_coast_non_vha_user)
        midsouth_user_results   = Referral.filter({'provider_id' => @east_coast_provider.id.to_s}, @midsouth_non_vha_user)

        expect(east_coast_user_results).to be_empty
        expect(midsouth_user_results).to   be_empty
      end

      it "should filter based on care category id, only on referrals visible to the user", cpp: true do
        east_coast_user_results = Referral.filter({'care_category_id' => @test_care_category.id.to_s}, @east_coast_non_vha_user)
        midsouth_user_results   = Referral.filter({'care_category_id' => @test_care_category.id.to_s}, @midsouth_non_vha_user)

        expect(east_coast_user_results).to include_only_values_with_listed_keys([:with_care_category], @referrals_for_both_users)
        expect(midsouth_user_results).to   include_no_values_from_list(@referrals_for_both_users.values)
      end

      it "should filter based on referral status, only on referrals visible to the user", cpp: true do
        review_status = ReferralStatus.where(name: 'Review Pending').first

        east_coast_user_results = Referral.filter({'status' => [review_status.id]}, @east_coast_non_vha_user)
        midsouth_user_results   = Referral.filter({'status' => [review_status.id]}, @midsouth_non_vha_user)

        expect(east_coast_user_results).to include_no_values_from_list(@referrals_for_both_users.values)
        expect(midsouth_user_results).to   include_only_values_with_listed_keys([:review_pending], @referrals_for_both_users)
      end

    end # context "for non-VHA community provider users"


    context "and generating CSV documents based on filtered results" do

      before(:context) do
        @veteran_names = Cpp::Local::Veteran.all.map &:first_name
        @find_any_name = -> (data_row) do
          @veteran_names.any? {|name| data_row.include? name }
        end
      end

      it "should return all records in CSV file" do
        csv_rows = Referral.to_csv({}, @vha_cc_user).split("\n")
        # count includes header line
        expect(csv_rows.count).to be > 2

        csv_header = csv_rows.first
        csv_rows_without_header = csv_rows.drop 1

        expect(@find_any_name.call(csv_header)).to eq false
        csv_rows_without_header.each do |csv_row|
          expect(@find_any_name.call(csv_row)).to eq true
        end
      end

      it "should return one record in CSV file for one-record filter" do
        referral_for_csv = create(:referral, authorization_number: 'CSV-TEST-098')
        csv_rows = Referral.to_csv({'authorization_number' => 'CSV-TEST-098'}, @vha_cc_user).split("\n")
        # count includes header line
        expect(csv_rows.count).to eq 2

        # header line should not contain veteran names, and first line should
        # contain veteran whose name was passed into to_csv filter parameter.
        expect(@find_any_name.call(csv_rows[0])).to eq false
        expect(@find_any_name.call(csv_rows[1])).to eq true
        expect(csv_rows[1]).to include(referral_for_csv.consultation.veteran(@vista_session).first_name)
      end

    end # context "and generating CSV documents based on filtered results"

  end # context "when performing filtering"


  context "when sorting filtered referrals" do

    before(:context) do
      @another_vha_cc_user = create(:vha_user)

      # Lambda function to abstract out common ordering and filtering operations:
      # 1. Perform filter operation to get all results
      # 2. Get paginated list with ordering that must be of the form
      #    {'column' => <column_number_as_string>, 'dir' => <'asc' | 'desc'>}
      # 3. Results are filtered based on teh ids passed in, so that referrals
      #    created in other contexts are disregarded.
      @perform_filtering_func = -> (ordering, ids_to_filter) do
        results = Referral.filter({}, @another_vha_cc_user)

        ordered_results = Referral.get_paginated_list(results, @another_vha_cc_user,
          {order: {'0' => ordering}}
        )
        # filter out results to only include data values equal to first, second, and third referrals above
        return ordered_results[:data].select do |json_record|
          ids_to_filter.include?(json_record[:id])
        end

      end
    end

    it "should sort referrals by patient name", cpp: true do
      # creating referrals out of order to ensure that default sorting is not used
      second_referral = create(:referral, :with_veteran, veteran:
                          create(:veteran, first_name: 'TEST_DEF', last_name: 'second vet'))
      third_referral = create(:referral, :with_veteran, veteran:
                          create(:veteran, first_name: 'TEST_GHI', last_name: 'third vet'))
      first_referral = create(:referral, :with_veteran, veteran:
                          create(:veteran, first_name: 'TEST_ABC', last_name: 'first vet'))
      referral_ids = [first_referral.id, second_referral.id, third_referral.id]

      asc_ordering = {'column' => Referral::COLUMNS[:patient_name], 'dir' => 'asc'}
      filtered_ordered_results_asc = @perform_filtering_func.call(asc_ordering, referral_ids)

      # ensuring that result values come back sorted by patient name
      expect(filtered_ordered_results_asc.count).to eq 3
      expect(filtered_ordered_results_asc[0][:patient_name]).to start_with('TEST_ABC')
      expect(filtered_ordered_results_asc[1][:patient_name]).to start_with('TEST_DEF')
      expect(filtered_ordered_results_asc[2][:patient_name]).to start_with('TEST_GHI')

      desc_ordering = {'column' => Referral::COLUMNS[:patient_name], 'dir' => 'desc'}
      filtered_ordered_results_desc = @perform_filtering_func.call(desc_ordering, referral_ids)

      expect(filtered_ordered_results_desc.count).to eq 3
      expect(filtered_ordered_results_desc[0][:patient_name]).to start_with('TEST_GHI')
      expect(filtered_ordered_results_desc[1][:patient_name]).to start_with('TEST_DEF')
      expect(filtered_ordered_results_desc[2][:patient_name]).to start_with('TEST_ABC')
    end

    it "should sort referrals by referral type", cpp: true do
      # creating referrals out of order to ensure that default sorting is not used
      second_referral = create(:referral, referral_type:
                          ReferralType.where(title: 'Outpatient Surgery').first)
      first_referral = create(:referral, referral_type:
                          ReferralType.where(title: 'Home Care Visit').first)
      third_referral = create(:referral, referral_type:
                          ReferralType.where(title: 'Radiology').first)
      referral_ids = [first_referral.id, second_referral.id, third_referral.id]

      asc_ordering = {'column' => Referral::COLUMNS[:referral], 'dir' => 'asc'}
      filtered_ordered_results_asc = @perform_filtering_func.call(asc_ordering, referral_ids)

      # ensuring that result values come back sorted by referral type
      expect(filtered_ordered_results_asc.count).to eq 3
      expect(filtered_ordered_results_asc[0][:referral]).to eq('Home Care Visit')
      expect(filtered_ordered_results_asc[1][:referral]).to eq('Outpatient Surgery')
      expect(filtered_ordered_results_asc[2][:referral]).to eq('Radiology')

      desc_ordering = {'column' => Referral::COLUMNS[:referral], 'dir' => 'desc'}
      filtered_ordered_results_desc = @perform_filtering_func.call(desc_ordering, referral_ids)

      expect(filtered_ordered_results_desc.count).to eq 3
      expect(filtered_ordered_results_desc[0][:referral]).to eq('Radiology')
      expect(filtered_ordered_results_desc[1][:referral]).to eq('Outpatient Surgery')
      expect(filtered_ordered_results_desc[2][:referral]).to eq('Home Care Visit')
    end

    it "should sort referrals by patient SSN", cpp: true do
      # creating referrals out of order to ensure that default sorting is not used
      third_referral = create(:referral, :with_veteran, veteran:
                          create(:veteran, ssn: '789789001'))
      first_referral = create(:referral, :with_veteran, veteran:
                          create(:veteran, ssn: '123123001'))
      second_referral = create(:referral, :with_veteran, veteran:
                          create(:veteran, ssn: '456456001'))
      referral_ids = [first_referral.id, second_referral.id, third_referral.id]

      asc_ordering = {'column' => Referral::COLUMNS[:ssn], 'dir' => 'asc'}
      filtered_ordered_results_asc = @perform_filtering_func.call(asc_ordering, referral_ids)
      filtered_ordered_results_asc = @perform_filtering_func.call({'column' => '2', 'dir' => 'asc'}, referral_ids)

      # ensuring that result values come back with ascending SSNs.  Only checking first
      # three digits because we don't care about how SSNs are formatted, just that they are
      # in the correct order. (and all other values have been filtered out)
      expect(filtered_ordered_results_asc.count).to eq 3
      expect(filtered_ordered_results_asc[0][:ssn]).to start_with('123')
      expect(filtered_ordered_results_asc[1][:ssn]).to start_with('456')
      expect(filtered_ordered_results_asc[2][:ssn]).to start_with('789')

      desc_ordering = {'column' => Referral::COLUMNS[:ssn], 'dir' => 'desc'}
      filtered_ordered_results_desc = @perform_filtering_func.call(desc_ordering, referral_ids)
      filtered_ordered_results_desc = @perform_filtering_func.call({'column' => '2', 'dir' => 'desc'}, referral_ids)

      expect(filtered_ordered_results_desc.count).to eq 3
      expect(filtered_ordered_results_desc[0][:ssn]).to start_with('789')
      expect(filtered_ordered_results_desc[1][:ssn]).to start_with('456')
      expect(filtered_ordered_results_desc[2][:ssn]).to start_with('123')
    end

    it "should sort referrals by date created", cpp: true do
      # this time, referrals need to be created in order
      first_referral = create(:referral)
      second_referral = create(:referral)
      third_referral = create(:referral)
      referral_ids = [first_referral.id, second_referral.id, third_referral.id]

      asc_ordering = {'column' => Referral::COLUMNS[:date], 'dir' => 'asc'}
      filtered_ordered_results_asc = @perform_filtering_func.call(asc_ordering, referral_ids)
      filtered_ordered_results_asc = @perform_filtering_func.call({'column' => '3', 'dir' => 'asc'}, referral_ids)

      # ensuring that result values come back sorted by referral type
      expect(filtered_ordered_results_asc.count).to eq 3
      expect(filtered_ordered_results_asc[0][:id]).to eq(first_referral.id)
      expect(filtered_ordered_results_asc[1][:id]).to eq(second_referral.id)
      expect(filtered_ordered_results_asc[2][:id]).to eq(third_referral.id)

      desc_ordering = {'column' => Referral::COLUMNS[:date], 'dir' => 'desc'}
      filtered_ordered_results_desc = @perform_filtering_func.call(desc_ordering, referral_ids)
      filtered_ordered_results_desc = @perform_filtering_func.call({'column' => '3', 'dir' => 'desc'}, referral_ids)

      expect(filtered_ordered_results_desc.count).to eq 3
      expect(filtered_ordered_results_desc[0][:id]).to eq(third_referral.id)
      expect(filtered_ordered_results_desc[1][:id]).to eq(second_referral.id)
      expect(filtered_ordered_results_desc[2][:id]).to eq(first_referral.id)
    end

    it "should sort referrals by referral status", cpp: true do
      # creating referrals out of order to ensure that default sorting is not used
      third_referral = create(:referral, :review_pending)
      first_referral = create(:referral, :complete)
      second_referral = create(:referral, :new)
      referral_ids = [first_referral.id, second_referral.id, third_referral.id]

      asc_ordering = {'column' => Referral::COLUMNS[:status], 'dir' => 'asc'}
      filtered_ordered_results_asc = @perform_filtering_func.call(asc_ordering, referral_ids)

      # ensuring that result values come back sorted by referral status
      expect(filtered_ordered_results_asc.count).to eq 3
      expect(filtered_ordered_results_asc[0][:status]).to eq('Complete')
      expect(filtered_ordered_results_asc[1][:status]).to eq('New')
      expect(filtered_ordered_results_asc[2][:status]).to eq('Review Pending')

      desc_ordering = {'column' => Referral::COLUMNS[:status], 'dir' => 'desc'}
      filtered_ordered_results_desc = @perform_filtering_func.call(desc_ordering, referral_ids)

      expect(filtered_ordered_results_desc.count).to eq 3
      expect(filtered_ordered_results_desc[0][:status]).to eq('Review Pending')
      expect(filtered_ordered_results_desc[1][:status]).to eq('New')
      expect(filtered_ordered_results_desc[2][:status]).to eq('Complete')
    end

    it "should sort referrals by coordinator name", cpp: true do
      # creating referrals out of order to ensure that default sorting is not used
      second_referral = create(:referral, coordinator:
                          create(:vha_user, first_name: 'Sortable2'))
      first_referral = create(:referral, coordinator:
                          create(:vha_user, first_name: 'Sortable1'))
      third_referral = create(:referral, coordinator:
                          create(:vha_user, first_name: 'Sortable3'))
      referral_ids = [first_referral.id, second_referral.id, third_referral.id]

      asc_ordering = {'column' => Referral::COLUMNS[:coordinator], 'dir' => 'asc'}
      filtered_ordered_results_asc = @perform_filtering_func.call(asc_ordering, referral_ids)

      # ensuring that result values come back sorted by referral type
      expect(filtered_ordered_results_asc.count).to eq 3
      expect(filtered_ordered_results_asc[0][:coordinator]).to start_with('Sortable1')
      expect(filtered_ordered_results_asc[1][:coordinator]).to start_with('Sortable2')
      expect(filtered_ordered_results_asc[2][:coordinator]).to start_with('Sortable3')

      desc_ordering = {'column' => Referral::COLUMNS[:coordinator], 'dir' => 'desc'}
      filtered_ordered_results_desc = @perform_filtering_func.call(desc_ordering, referral_ids)

      expect(filtered_ordered_results_desc.count).to eq 3
      expect(filtered_ordered_results_desc[0][:coordinator]).to start_with('Sortable3')
      expect(filtered_ordered_results_desc[1][:coordinator]).to start_with('Sortable2')
      expect(filtered_ordered_results_desc[2][:coordinator]).to start_with('Sortable1')
    end

    it "should sort referrals by provider name when all referrals have specified providers", cpp: true do
      # creating referrals out of order to ensure that default sorting is not used
      second_referral = create(:referral, provider:
                          create(:provider, name: 'SortingProvider2'))
      third_referral = create(:referral, provider:
                          create(:provider, name: 'SortingProvider3'))
      first_referral = create(:referral, provider:
                          create(:provider, name: 'SortingProvider1'))
      referral_ids = [first_referral.id, second_referral.id, third_referral.id]

      asc_ordering = {'column' => Referral::COLUMNS[:provider], 'dir' => 'asc'}
      filtered_ordered_results_asc = @perform_filtering_func.call(asc_ordering, referral_ids)

      # ensuring that result values come back sorted by referral type
      expect(filtered_ordered_results_asc.count).to eq 3
      expect(filtered_ordered_results_asc[0][:provider]).to start_with('SortingProvider1')
      expect(filtered_ordered_results_asc[1][:provider]).to start_with('SortingProvider2')
      expect(filtered_ordered_results_asc[2][:provider]).to start_with('SortingProvider3')

      desc_ordering = {'column' => Referral::COLUMNS[:provider], 'dir' => 'desc'}
      filtered_ordered_results_desc = @perform_filtering_func.call(desc_ordering, referral_ids)

      expect(filtered_ordered_results_desc.count).to eq 3
      expect(filtered_ordered_results_desc[0][:provider]).to start_with('SortingProvider3')
      expect(filtered_ordered_results_desc[1][:provider]).to start_with('SortingProvider2')
      expect(filtered_ordered_results_desc[2][:provider]).to start_with('SortingProvider1')
    end

    it "should sort referrals by provider name when some referrals use their ordering provider, " +
       "and referral providers should have priority over ordering providers", cpp: true do
      # creating referrals out of order to ensure that default sorting is not used
      # I. referral with specified provider
      second_referral = create(:referral, provider:
                          create(:provider, name: 'SortableMixedProvider2'))
      # II. referral with only ordering providers
      first_referral = create(:referral, :with_ordering_provider,
                          ordering_provider: create(:provider, name: 'SortableMixedProvider1'))
      fifth_referral = create(:referral, :with_ordering_provider,
                          ordering_provider: create(:provider, name: 'SortableMixedProvider5'))
      # II. referral with both specified and ordering providers
      third_referral = create(:referral, :with_ordering_provider,
                          provider: create(:provider, name: 'SortableMixedProvider3'),
                          ordering_provider: create(:provider, name: 'DO_NOT_USE_THIS_PROVIDER'))
      fourth_referral = create(:referral, :with_ordering_provider,
                          provider: create(:provider, name: 'SortableMixedProvider4'),
                          ordering_provider: create(:provider, name: 'DO_NOT_USE_THIS_PROVIDER2'))

      referral_ids = [first_referral.id, second_referral.id, third_referral.id, fourth_referral.id, fifth_referral.id]

      asc_ordering = {'column' => Referral::COLUMNS[:provider], 'dir' => 'asc'}
      filtered_ordered_results_asc = @perform_filtering_func.call(asc_ordering, referral_ids)

      # ensuring that result values come back sorted by referral type
      expect(filtered_ordered_results_asc.count).to eq 5
      expect(filtered_ordered_results_asc[0][:provider]).to start_with('SortableMixedProvider1')
      expect(filtered_ordered_results_asc[1][:provider]).to start_with('SortableMixedProvider2')
      expect(filtered_ordered_results_asc[2][:provider]).to start_with('SortableMixedProvider3')
      expect(filtered_ordered_results_asc[3][:provider]).to start_with('SortableMixedProvider4')
      expect(filtered_ordered_results_asc[4][:provider]).to start_with('SortableMixedProvider5')

      desc_ordering = {'column' => Referral::COLUMNS[:provider], 'dir' => 'desc'}
      filtered_ordered_results_desc = @perform_filtering_func.call(desc_ordering, referral_ids)

      expect(filtered_ordered_results_desc.count).to eq 5
      expect(filtered_ordered_results_desc[0][:provider]).to start_with('SortableMixedProvider5')
      expect(filtered_ordered_results_desc[1][:provider]).to start_with('SortableMixedProvider4')
      expect(filtered_ordered_results_desc[2][:provider]).to start_with('SortableMixedProvider3')
      expect(filtered_ordered_results_desc[3][:provider]).to start_with('SortableMixedProvider2')
      expect(filtered_ordered_results_desc[4][:provider]).to start_with('SortableMixedProvider1')
    end

    it "should sort referrals by VISN and facility name", cpp: true do
      # I. second VISN when sorting. (creating referrals with VISN:Facility out of order)
      fourth_referral = create(:referral, provider: create(:provider, :with_facility_and_visn_number,
                          facility_name: 'SortingTestFacility2', visn_number: 15 ))
      third_referral = create(:referral, provider: create(:provider, :with_facility_and_visn_number,
                          facility_name: 'SortingTestFacility1', visn_number: 15 ))
      # II. first VISN when sorting; using separate single-digit numbers
      first_referral = create(:referral, provider: create(:provider, :with_facility_and_visn_number,
                          facility_name: 'SortingTestFacilityX', visn_number: 2 ))
      second_referral = create(:referral, provider: create(:provider, :with_facility_and_visn_number,
                          facility_name: 'SortingTestFacilityY', visn_number: 5 ))
      referral_ids = [first_referral.id, second_referral.id, third_referral.id, fourth_referral.id]

      asc_ordering = {'column' => Referral::COLUMNS[:visn_facility], 'dir' => 'asc'}
      filtered_ordered_results_asc = @perform_filtering_func.call(asc_ordering, referral_ids)

      # ensuring that result values come back sorted by VISN:Facility -- first testing that correct
      # VISN appears in beginning of string, then correct facility should appear at end.
      expect(filtered_ordered_results_asc.count).to eq 4
      expect(filtered_ordered_results_asc[0][:visn_facility]).to start_with('2')
      expect(filtered_ordered_results_asc[1][:visn_facility]).to start_with('5')
      expect(filtered_ordered_results_asc[2][:visn_facility]).to start_with('15')
      expect(filtered_ordered_results_asc[3][:visn_facility]).to start_with('15')

      expect(filtered_ordered_results_asc[2][:visn_facility]).to end_with('SortingTestFacility1')
      expect(filtered_ordered_results_asc[3][:visn_facility]).to end_with('SortingTestFacility2')

      desc_ordering = {'column' => Referral::COLUMNS[:visn_facility], 'dir' => 'desc'}
      filtered_ordered_results_desc = @perform_filtering_func.call(desc_ordering, referral_ids)

      expect(filtered_ordered_results_desc.count).to eq 4
      expect(filtered_ordered_results_desc[0][:visn_facility]).to start_with('15')
      expect(filtered_ordered_results_desc[1][:visn_facility]).to start_with('15')
      expect(filtered_ordered_results_desc[2][:visn_facility]).to start_with('5')
      expect(filtered_ordered_results_desc[3][:visn_facility]).to start_with('2')

      expect(filtered_ordered_results_desc[0][:visn_facility]).to end_with('SortingTestFacility2')
      expect(filtered_ordered_results_desc[1][:visn_facility]).to end_with('SortingTestFacility1')
    end

  end # context "when sorting filtered referrals"


end
