# == Schema Information
#
# Table name: providers
#
#  id             :integer          not null, primary key
#  npi            :string           not null
#  name           :string           not null
#  physician_name :string           not null
#  content        :json
#  created_at     :datetime         not null
#  updated_at     :datetime         not null
#  facility_id    :integer
#

require 'rails_helper'

RSpec.describe Provider, :type => :model do

  context "associations" do
    it "should have one facility", cpp: true do
      provider = Provider.reflect_on_association(:facility)
      expect(provider.macro).to eq :belongs_to
    end

    it "should have one visn", cpp: true do
      provider = Provider.reflect_on_association(:visn)
      expect(provider.macro).to eq :has_one
    end

    it "should have and belongs to many medical specialities", cpp: true do
      provider = Provider.reflect_on_association(:medical_specialties)
      expect(provider.macro).to eq :has_and_belongs_to_many
    end

    it "should have many referrals", cpp: true do
      provider = Provider.reflect_on_association(:referrals)
      expect(provider.macro).to eq :has_many
    end

    it "should have many consultations", cpp: true do
      provider = Provider.reflect_on_association(:consultations)
      expect(provider.macro).to eq :has_many
    end

    it "should have many providers_users", cpp: true do
      provider = Provider.reflect_on_association(:providers_users)
      expect(provider.macro).to eq :has_many
    end

    it "should have many users through provider_users", cpp: true do
      provider = Provider.reflect_on_association(:users)
      expect(provider.macro).to eq :has_many
    end
  end

  context "methods" do
    let (:address_attributes) {
      {"city"=>"SomeCity",
       "address_2"=>"",
       "telephone_number"=>"333-555-0000",
       "fax_number"=>"333-556-0000",
       "state"=>"MD",
       "postal_code"=>"21045",
       "address_1"=>"123 SOME PLACE",
       "country_code"=>"US",
       "country_name"=>"United States",
       "address_type"=>"DOM",
       "address_purpose"=>"LOCATION"}
    }
    let(:address) {[address_attributes]}
    let(:address_with_zip_plus_4) {[address_attributes.merge("postal_code"=>"210456666")]}
    let(:taxonomy) {[{"state"=>"MD",
        "code"=>"0000C0700X",
        "primary"=>true,
        "license"=>"XX#0000",
        "desc"=>"SomeDescription"}]}
    let(:visn) { Visn.first }
    let(:facility) { create(:facility, visn: visn) }
    # facility2 must have different VISN for visn_id params query test to pass
    let(:facility2) { create(:facility, visn: Visn.second) }
    let!(:provider)  { create(:provider, content: {addresses: address, taxonomies: taxonomy}) }
    let!(:provider_1_in_second_visn) { create(:provider, content: {addresses: address, taxonomies: taxonomy}, facility: facility2) }
    let!(:provider_2_in_second_visn) { create(:provider, content: {addresses: address, taxonomies: taxonomy}, facility: facility2) }
    let!(:provider_with_zip_plus_4)  { create(:provider, content: {addresses: address_with_zip_plus_4, taxonomies: taxonomy}, facility: facility2) }

    it "should return a hash with address parameters", cpp: true do
      expect(provider.address_attrs).to include(address: "123 SOME PLACE ", city: "SomeCity", state: "MD", zip: "21045")
    end

    it "should return address", cpp: true do
      expect(provider.address).to start_with address.first["address_1"]
      expect(provider.address).to end_with address.first["address_2"]
    end

    it "should return formatted address", cpp: true do
      expect(provider.formatted_address_string).to eq "123 SOME PLACE , SomeCity, MD 21045"
    end

    it "should return city", cpp: true do
      expect(provider.city).to eq address.first["city"]
    end

    it "should return state", cpp: true do
      expect(provider.state).to eq address.first["state"]
    end

    it "should return zip", cpp: true do
      expect(provider.zip).to eq address.first["postal_code"]
    end

    it "should return formatted zip", cpp: true do
      expect(provider_with_zip_plus_4.zip).to eq "21045-6666"
    end

    it "should return phone no", cpp: true do
      expect(provider.phone).to eq address.first["telephone_number"]
    end

    it "should return fax no", cpp: true do
      expect(provider.fax).to eq address.first["fax_number"]
    end

    it "should return medical speciality", cpp: true do
      expect(provider.medical_specialty_names).to eq taxonomy.first["desc"]
    end

    # SMELL: goverly generic names 'provider1', etc. make the logic of this test hard to understand at first glance.
    it "should return filters based on visn_id params query", cpp: true do
      result1 = Provider.filter({visn_id: provider.facility.visn_id})
      result2 = Provider.filter({visn_id: facility2.visn_id})
      
      expect(result1.first).to eq(provider)
      expect(result2).to match_array([provider_1_in_second_visn, provider_2_in_second_visn, provider_with_zip_plus_4])
    end

    it "should return filters based on name params query", cpp: true do
      result = Provider.filter({name: provider_1_in_second_visn.name})
      expect(result.count).to eq(1)
      expect(result.first).to eq(provider_1_in_second_visn)
    end

    it "should return filters based on physician_name params query", cpp: true do
      result = Provider.filter({physician_name: provider_2_in_second_visn.physician_name})
      expect(result.count).to eq(1)
      expect(result.first).to eq(provider_2_in_second_visn)
    end

    it "should return all providers without filter", cpp: true do
      expect(Provider.filter).to match_array([provider, provider_1_in_second_visn, provider_2_in_second_visn,
                                              provider_with_zip_plus_4])
    end
  end
end
