require 'rails_helper'
require 'vbms'

RSpec.describe VBMSJob, type: :job do
  def disable_external_rendering!(evaluation)
    allow(evaluation).to receive(:temp_pdf_path).and_return "/tmp/out.pdf"
    allow(subject).to receive(:output_to_temp_file) # do not actually write a file
    allow_any_instance_of(Evaluation).to receive(:to_pdf) { "" }
  end
  before do
    allow_any_instance_of(Evaluation).to receive(:signed?).and_return(true)
  end

  context "connect_vbms error handling" do
    let(:evaluation) { create(:evaluation, completed_at: DateTime.current.utc) }
    let(:client) { instance_double("VBMS::Client") }

    before do
      disable_external_rendering!(evaluation)
      expect(VBMS::Client).to receive(:from_env_vars) { client }
      expect(VBMS::Requests::UploadDocumentWithAssociations).to receive(:new)
    end

    it "should handle an HTTPError" do
      expect(client).to receive(:send).and_raise(VBMS::HTTPError.new(500, "test error"))
      expect { subject.perform(evaluation) }.to raise_exception VBMS::HTTPError

      log = EvaluationLog.last
      expect(log.message).to eq "VBMS error code 500"
      expect(log.response_body).to eq "test error"
    end

    it "should handle an ExecutionError" do
      expect(client).to receive(:send).and_raise(VBMS::ExecutionError.new("command that failed", "command output"))
      expect { subject.perform(evaluation) }.to raise_exception VBMS::ExecutionError

      log = EvaluationLog.last
      expect(log.message).to eq "Execution error, unable to submit evaluation"
      expect(log.submitted_data).to eq "Error executing command `command that failed`"
      expect(log.response_body).to eq "command output"
    end

    it "should handle a ClientError" do
      expect(client).to receive(:send).and_raise(VBMS::ClientError.new("error message"))
      expect { subject.perform(evaluation) }.to raise_exception VBMS::ClientError

      log = EvaluationLog.last
      expect(log.message).to eq "VBMS error VBMS::ClientError: error message"
    end
  end

  describe "#perform" do
    context "when an evaluation is not complete" do
      before do
        user = create(:user)
        evaluation_spec = create(:diabetes_spec)
        claim = create(:claim, file_number: "784449089")
        claim.contentions << create(:contention, history: "This or That")
        @evaluation = Evaluation.new(claim: claim,
                                     evaluation_spec: evaluation_spec,
                                     doc: build(:diabetes_doc).doc,
                                     user: user)
        claim.contentions.first.evaluations << @evaluation
      end

      it "does not output or send a file if the evaluation is not complete" do
        expect(subject).to_not receive(:output_to_temp_file)
        expect(subject).to_not receive(:send_to_vbms)
        subject.perform(@evaluation)
      end

      it "creates an EvaluationLog object" do
        expect {subject.perform(@evaluation)}.to change {EvaluationLog.count}.by(1)
      end

      it "has the evaluation id properly set" do
        subject.perform(@evaluation)
        expect(EvaluationLog.last.evaluation_id).to eq @evaluation.id
      end
    end

    context "when an evaluation is complete" do
      before do
        user = create(:user)
        evaluation_spec = create(:diabetes_spec)
        claim = create(:claim, file_number: "784449089")
        claim.contentions << create(:contention, history: "This or That")
        @evaluation = Evaluation.new(claim: claim,
                                     evaluation_spec: evaluation_spec,
                                     doc: build(:diabetes_doc).doc,
                                     completed_at: Time.now,
                                     user: user)
        claim.contentions.first.evaluations << @evaluation

        disable_external_rendering!(@evaluation)
        client = instance_double("VBMS::Client")
        expect(VBMS::Client).to receive(:from_env_vars) { client }
        @doc = Nokogiri::XML("<test></test>")
        expect(client).to receive(:send) { @doc }
      end

      it "logs to the external activity logs table" do
        expect(VBMS::Requests::UploadDocumentWithAssociations).to receive(:new)
        expect{subject.perform(@evaluation)}.to change{EvaluationLog.count}.by(1)
      end

      it "calls the trigger to the specified external service" do
        expect(VBMS::Requests::UploadDocumentWithAssociations).to receive(:new).with(
          "784449089",                    # file_number
          kind_of(Time),                  # received_dt
          /\w+/,                          # first_name
          nil,                            # middle_name
          /\w+/,                          # last_name
          /VA Disability Assessment \w+/, # subject
          '/tmp/out.pdf',                 # pdf
          '356',                          # doctype
          'VHA_CUI',                      # source
          true,                           # new_mail
        )
        subject.perform(@evaluation)
      end

      it "returns the response document" do
        expect(VBMS::Requests::UploadDocumentWithAssociations).to receive(:new)
        expect(subject.perform(@evaluation)).to eq @doc
      end
    end

    context "send errors" do
      let(:evaluation) { Evaluation.new(claim: create(:claim),
                                        evaluation_spec: create(:diabetes_spec),
                                        doc: build(:diabetes_doc).doc,
                                        completed_at: Time.now,
                                        user: create(:user)) }

      before do
        disable_external_rendering!(evaluation)
        @client = instance_double("VBMS::Client")
        expect(VBMS::Client).to receive(:from_env_vars) { @client }
      end

      context "with HTTPError" do
        let(:error) { VBMS::HTTPError.new(500, "error") }

        before do
          expect(@client).to receive(:send).and_raise(error)
        end

        it "logs properly" do
          expect {subject.perform(evaluation)}.to raise_exception(error.class)
          expect(EvaluationLog.last.evaluation_id).to eq evaluation.id
          expect(EvaluationLog.last.message).to eq "VBMS error code 500"
        end
      end

      context "with ExecutionError" do
        let(:error) { VBMS::ExecutionError.new("echo", "error") }

        before do
          expect(@client).to receive(:send).and_raise(error)
        end

        it "logs properly" do
          expect {subject.perform(evaluation)}.to raise_exception(error.class)
          expect(EvaluationLog.last.evaluation_id).to eq evaluation.id
          expect(EvaluationLog.last.message).to eq "Execution error, unable to submit evaluation"
        end
      end

      context "with ClientError" do
        # SOAPError is a subclass of ClientError
        let(:error) { VBMS::SOAPError.new("Kilroy was here") }

        before do
          expect(@client).to receive(:send).and_raise(error)
        end

        it "logs properly" do
          expect {subject.perform(evaluation)}.to raise_exception(error.class)
          expect(EvaluationLog.last.evaluation_id).to eq evaluation.id
          expect(EvaluationLog.last.message).to eq "VBMS error VBMS::SOAPError: Kilroy was here"
        end
      end

      context "with anything else" do
        let(:error) { "yada yada yada" }

        before do
          expect(@client).to receive(:send).and_raise(error)
        end

        it "logs properly" do
          expect {subject.perform(evaluation)}.to raise_exception(RuntimeError)
          expect(EvaluationLog.last.evaluation_id).to eq evaluation.id
          expect(EvaluationLog.last.message).to eq "VBMS threw unexpected exception RuntimeError: yada yada yada"
        end
      end
    end
  end

  context "PDF job handling" do
    before do
      @user = create(:user)
      @evaluation_spec = create(:diabetes_spec)
      @claim = create(:claim, file_number: "123")
      @claim.contentions << create(:contention, history: "This or That")
      @evaluation = Evaluation.new(claim: @claim,
                                   evaluation_spec: @evaluation_spec,
                                   doc: build(:diabetes_doc).doc,
                                   completed_at: Time.now,
                                   user: @user)
      @claim.contentions.first.evaluations << @evaluation
      @evaluation.complete! @user
      allow(@evaluation).to receive(:render_html).and_return "<b>nope</b>"
      ENV['VBMS_UPLOADER_ENV'] = "test"
    end

    it "calls out to WickedPdf for the conversion" do
      wicked = double("WickedPdf")
      expect(WickedPdf).to receive(:new) { wicked }
      expect(wicked).to receive(:pdf_from_string).with("<b>nope</b>").and_return(kind_of(String))
      expect(subject).to receive(:send_to_vbms)
      subject.perform(@evaluation)
    end

    it "outputs a file" do
      disable_external_rendering!(@evaluation)
      expect(subject).to receive(:output_to_temp_file).with(kind_of(String), "/tmp/out.pdf")
      expect(subject).to receive(:send_to_vbms)
      subject.perform(@evaluation)
    end
  end

  context "PDF HTML" do
    before do
      @user = create(:user)
      @evaluation_spec = create(:diabetes_spec)
      @claim = create(:claim, file_number: "784449089")
      @evaluation = Evaluation.new(claim: @claim,
                                   evaluation_spec: @evaluation_spec,
                                   doc: build(:diabetes_doc).doc,
                                   vha_user_vista_access_code: "1234",
                                   vha_user_vista_verify_code: "1234",
                                   vha_user_electronic_signature: "1234",
                                   user: @user)
      @evaluation.complete! @user
      @first_contention = create(:contention, name: "First Contention", evaluations: [@evaluation])
      @second_contention = create(:contention, name: "Second Contention", evaluations: [@evaluation])
      @claim.contentions = [@first_contention, @second_contention]
      @evaluation.contentions = [@first_contention, @second_contention]

      ENV['VBMS_UPLOADER_ENV'] = "test"

      @wicked = double("WickedPdf")
      expect(WickedPdf).to receive(:new) { @wicked }

      client = instance_double("VBMS::Client")
      expect(VBMS::Client).to receive(:from_env_vars) { client }
      expect(VBMS::Requests::UploadDocumentWithAssociations).to receive(:new)
      expect(client).to receive(:send)
    end

    it "includes the evaluation title" do
      title = @evaluation_spec.title
      version = @evaluation_spec.version
      expect(@wicked).to receive(:pdf_from_string).with(/#{title} \(Version: #{version}\)/)
      subject.perform(@evaluation)
    end

    it "lists all contention titles at the top" do
      expect(@wicked).to receive(:pdf_from_string) { |html|
        expect(html).to match /<li>#{@first_contention.name}<\/li>/
        expect(html).to match /<li>#{@second_contention.name}<\/li>/
      }
      subject.perform(@evaluation)
    end

    it "includes the provider signature and date the exam was signed and completed by the provider." do
      expect(@wicked).to receive(:pdf_from_string) { |html|
        expect(html).to match /Printed name of provider: #{@user.first_name} #{@user.last_name}/
        expect(html).to match /#{@evaluation.completed_at.utc.to_s}/
      }
      subject.perform(@evaluation)
    end
  end
end
