namespace :exam_requests do

  # NOTE: yep, there is only one in the array; setting a pattern to
  #       to be used in a set of common support tasks to be used by other tasks.
  REQUIRED_UTILITIES = %w[ curl ]

  desc "Copy existing exam_request data to claims;"
  task copy_existing_data_to_claims: :environment do
    ExamRequest.where.not(claim_id: nil).each do |exam_request|
      begin
        if claim = Claim.find(exam_request.claim_id)
          if claim.exam_request_data.blank?
            if claim.update_attributes!(exam_request_data:  Hash.from_xml(exam_request.xml))
            puts "Adding exam_request_data to claim: #{claim.id}  (ExamRequest: #{exam_request.id.to_s})" if Rails.env.production? || Rails.env.development?
            end
          end
        end
      rescue Exception => e
        puts "#{e.message} (ExamRequest: #{exam_request.id.to_s})" if Rails.env.production? || Rails.env.development?
        Rails.logger.info "Rescued #{e.inspect} -- ExamRequest #{exam_request.id} has a dangling claim id."
      end
    end

  end # task copy_existing_data_to_claims: :environment do


  #############################################
  # exam_requests:seed
  # Assumes: 1) you are running Puma, not any other server
  #          2) your machine recognizes the curl command
  #
  desc "Seed a development environment by sending a standard set of exam requests to a running CUI app"
  task seed: %w[ environment exam_requests:check_required_utilities ] do

    unless Rails.env.development?  &&  DBSEED
      STDERR.puts <<~EOM

        As a safety precaution, this task can only run when DBSEED is true
        and RAILS_ENV is development.

        Current values are:
          DBSEED:    #{DBSEED}
          RAILS_ENV: #{Nenv.RAILS_ENV}

      EOM
      exit(-1)
    end

    pid_path    = Rails.root + "tmp" + "pids" + "puma.pid"
    seed_path   = Rails.root + "db"  + "exam_requests"

    begin
      pid = File.read(pid_path).to_i
      Process.getpgid pid  # NOTE: Will throw an exception if the pid is not running

      # NOTE: This command intentionally targets the localhost as a safety
      #       measure to help prevent polluting non-development databases
      #       __filename__ is a placeholder that will be replaced with the
      #       full path to the file to be uploaded.
      command = %Q[
        curl -S -s --data @__filename__ \
             --header "Content-Type: application/xml" \
             --header "Accept: application/xml" \
             -X POST #{Nenv.PUMA_PROTOCOL.gsub('tcp','http')}localhost:#{Nenv.PUMA_PORT}#{Nenv.CLAIMS_API_URL}
      ] # TODO: check the puma protocol for https it might be ssl://

      seed_path.children.
        select{|a_path| !a_path.directory? && '.xml' == a_path.extname}.
        each do |filename|
          print "Seeding exam request: #{filename.basename} ... "
            `#{command.gsub('__filename__', filename.to_s)}`
          puts "done"
        end # seed_path.children

    rescue Errno::ESRCH, Errno::ENOENT
      STDERR.puts <<~EOM

        ERROR:  The CUI is not running on this machine.
                Please start the application in the development environment
                then attempt to seed your exam requests again.

      EOM
      exit(-1)
    end

  end # task seed: :environment do


  # TODO: checking to see that the requred command line utilities is a common
  #       thing to do across several rake tasks; consider extract this kind
  #       of task into a common core of utility tasks and just pass
  #       an array of the specific tasks required utilities to the common task.
  task :check_required_utilities do
    missing_utilities = []
    REQUIRED_UTILITIES.each do |cli_utility|
      # TODO: Find out if the Windoze environment supports 'which' as a standard
      #       shell command
      missing_utilities << cli_utility if `which #{cli_utility}`.empty?
    end

    unless missing_utilities.empty?
      STDERR.puts <<~EOM

        ERROR:  The following command line utilities are required but
                are not found on this machine:

                #{missing_utilities.join(', ')}

      EOM
      exit(-1)
    end
  end # task :check_required_utilities do

end # namespace :exam_requests do