# Trouble Shooting Environment Variables

The doc/system_environment_variables.md file attempts to document all of the environment variables used by this web application and its many pieces.

This document's goal is to help the software developer or DevOps engineer focus on resolving a problem which may be a result of environment variable values not being as expected.

## Context

We have at least three major contexts in which we develop and operate this system.  Those contexts can generically be classified as AWS/EC2, docker containers/images and native host machines.  The goal of this project is to have a system that operates in all contexts the same way.  The way we have chosen to configure our system is through the use of system environment variables.

The system environment variables can change from context to context in order to accomplish the same behavior of the system.  Sometimes the values of the variables for a given context will cause a problem which may not be easy to track down.

The values of system environment variables are set in three different ways: 1) `.env*` files - some configuration managed and some not; 2) `config/docker/*_service.yml` files which are all configuration managed; and, 3) manually set within a terminal/console session.

The follow sections discuss the common problems that may occur with the value of variables being set via these methods.

### `.env*` Files.

As explained in `doc/system_environment_variables.md` the `.env*` files are processed in two different ways depending upon whether they are being used within a `bash` shell script or within the Rails application of `rake` task.

The `.env*.local` files are not configuration managed.  They are intended as local over-riders for use by the software developer on their individual workstations.  Its possible that the DevOps engineer may also file these files useful in providing some host-specific configuration.  This use case is not recommended.  The better practice for host-specific configuration is to make use of the `DEPLOY_TYPE` system environment variable to direct the loading of a host-specific `.env.#{DEPLOY_TYPE}` file.

It is possible that most problems caused by environment variables being wrong are a result of a setting in a `.env*.local` file.

The next most likely cause of problems is an environment varialbe being set in the wrong file.  There are three basic layers of `.env*` files.  The "top" later is the base (aka. default) layer implemented with the `.env` file.  In this file should only go value assignments that are common to all subsequent layers - environment and deployment type.

Setting `APP_URL` to a host-specific value is an invalid use of the `.env` file.

The next layer is the environment layer directed by the `RAILS_ENV` value.  The values in this layer adds new or makes changes to environment variables specific to a behavioural environment.  So far the only environments that have been defined are the common Rails environments of development, test and production.

The final layer is the deployment (host-specific) layer directed by the `DEPLOY_TYPE` environment variable value.  This layer adds new or changes the value of environment variables that have have come before in the other two layers.  This is the proper layer for setting the values of host-specific variables.

The following sections discuss how these different layers are processed by the different contexts of the system.

#### Rails and `rake` Tasks

The good news is that `rake` tasks and the Rails application process the `.env*` the same way - the first value set is the official value - as described by the `dotenv` gem documentation.  In fact `rake` tasks use `config/application.rb` to process the `.env*` the same way as the Rails application.

Refer to the diagram in `doc/env_file_relationship.dot.pdf` where you will notice that the Rails application starts at the bottom and goes up in terms of the order of which `.env*` is processed.

A common problem occurs when there is an environment variable that has a dependency on some other environment variable.  Consider this common configuration:

```
# File 1
export PROTOCOL="http://"
export HOST="localhost"
export PORT="1234"
export URL=$PROTOCOL$HOST:$PORT

# File 2
export PORT="4321"
```

When these two files are processed top-to-bottom (like is done in `bash` shell scripts) the value of `url` with be `http://localhost:1234` which is different when the files are processed bottom-to-top as is done by the Rails application and `rake` tasks.  That processing order will generate a value for `URL` as `http://localhost:4321` because the `dotenv` behavior is that the *first* value set wins.

In the bottom-to-top order `PORT` was initially set o 4321.  The subsequent assignment of `PORT` to 1234 is ignored by `dotenv` processing.


### Manually Set Within a Terminal/Console Session

As a developer it is common practice to have multiple terminal/console windows open on your workstation.  You may be working in the same feature branch in each window.

There are three system environment variables that must be manually set within a terminal/console window: `RAILS_ENV`, `RACK_ENV` and `DEPLOY_ENV`.  The value of these three environment variables direct which `.env*` files are processed.  If their values are not the same in each terminal/console window you may get different results when running the application.  The application may work in one window and not in another.

That is a condition that can drive any developer crazy.


### `config/docker/*_service.yml` Files

The final context is docker.  We use the `docker-compose` utility to build docker containers from docker images.  We get three officially maintained images from the docker hub: Ruby version 2.3.3, PostgreSQL version 9.3 and Redis version 3.2.8.  From these images `docker-compose` builds the containers which we use.

The script `bin/docker_cui_create.sh` is responsible for setting the environment within which the `docker-compose` utility is executed.  The script makes use of `bin/setup_environment` to ensure that the `.env*` files are processed (sourced) in the proper order.  This is the top-to-bottom order that is shown in the `doc/env_file_relationship.dot.pdf` diagram.

We use version 3 of the `docker-compose.yml` file format.  The version allows for the use of environment variables within the `docker-compose.yml` file.

Docker uses the term `service` to reference "things" which may or may not really be consider "containers."  The docker vocabulary is confusing.  Let use agree that "service" and "container" mean the same thing.  If we are wrong someone more educated will enlighten us.

There are three services defined for this system:  db, redis and web.  Each server/container can be executed on a developer's workstation or through some magic means yet unexplained be executed on an AWS/EC2 instance.

Each service has an associated `config/docker/#{service}_docker-compose.yml` file which uses an `m4` text macro `include()` to import a corresponding `config/docker/#{service}_service.yml` file.

> `m4` is a common *NIX command line utility.  If your workstation does not have this utility as part of its normal distribution you can get it through you primary package manager.  For example on the Macintosh OSX platform you can use `brew install m4` to get the utility.  The `awk` common *NIX command line utility is also used.  If your workstation does not have it you can get it the same way that you used to get the `m4` utility.

There are two places within the `*_service.yml` file in which environment variables are used.

#### ports Element

The `ports` element of the `*_service.yml` file is used to define the mapping of a port internal to the container (service) to the host machine - a developer's workstation or a deployed AWS/EC2 instance.  The following table shows the environment names used by each service to implement this mapping.

| Servuce | Variable Name for Host Port | Variable Name for Container Port |
| ------- | --------------------------- | -------------------------------- |
| db      | DOCKER_HOST_DB_PORT         | DBPORT |
| redis   | DOCKER_HOST_REDIS_PORT      | REDIS_PORT |
| web     | DOCKER_HOST_WEB_PORT        | PUMA_PORT |

As a developer if you have a PostgreSQL or Redis server running natively on your workstation AND you attempt to start up a similar docker container, you may have port conflicts.  The service may refuse to start in its docker container because the port is already in use.  In this case you have two choices: 1) terminate your native applications; or, 2) change the port mapping using an appropriate `.env*.local` file.

If you chose to change the port mapping, the common choice for the `DOCKER_HOST_*_PORT` is 10000 + the value of `*PORT` like so:

```
# .env.docker.local file
export DOCKER_HOST_DB_PORT="1$DBPORT"
```

> In order to get the `.env.docker.local` file to be processed the value of `DEPLOY_TYPE` must be `docker`.

PostgreSQL has some special environment variables that are used within the `db_service.yml` file and by the program itself.  These are explained in more detail in the following sections.


#### environment Element

The format of the `docker-compose.yml` files allows for various ways in which the values of environment variables can be fed into the container.  The way in which our system uses is the `environment` element of the YAML file format.

The format of the `environment` YAML element is discussed at [https://docs.docker.com/compose/compose-file/#environment](https://docs.docker.com/compose/compose-file/#environment)

We use the array syntax so that we can pass existing environment variables as well as assigne values to other environment variables.

The [documentation for the PostgreSQL v9.3 docker image](https://hub.docker.com/_/postgres/) explains how the variables `POSTGRES_USER`, `POSTGRES_PASSWORD` and `POSTGRES_DB` are used during initialization of the container to create the role `POSTGRES_USER` and to create the database `POSTGRES_DB` automatically.

These three special environment are not used anywhere else in our system so we have use of the ability of `docker-compose` to pass in variables with assignments using existing variables within our existin `.env*` files.

```
- POSTGRES_USER=$DBUSER
- POSTGRES_PASSWORD=$DBPASS
- POSTGRES_DB=${DBNAME}_${RAILS_ENV}
```

Notice how the `POSTGRES_DB` value is composed of both the `DBNAME` and `RAILS_ENV` the same way that it is within the `config/database.yml` file.

PostgreSQL (and other programs) gave special environment variables which are used in various stages of that initialization and startup.  These environment variables can be thought os as being invisible because you don't necessary know that they exist unless you have a good uderstanding of the program/utility that they support.

The next section discusses the environment variables used by PostgreSQL.


## Invisible Variables

Most *NIX programs/utilities make use of command line parameters, configuration files and/or environment variables for their configuration.  Some use all three.  In the case of PostgreSQL there are several environment variables that define its configuration which may result in problems within our application.  These environment varibles for PostgreSQL are documented at [https://www.postgresql.org/docs/9.3/static/libpq-envars.html](https://www.postgresql.org/docs/9.3/static/libpq-envars.html).

As a developer it is likely that you are running PostgreSQL natively and may have some of these environment variables set in one of your `~/.bashrc` files.  Some of the more common ones are listed in the table below.  Notice how the variable names are "namespaced" using `PG` as a prefix.  The "PG" of course refers to PostgreSQL.

| Variable Name | How Used by PostgreSQL |
| ------------- | ---------------------- |
| PGHOST | behaves the same as the host connection parameter. |
| PGHOSTADDR | behaves the same as the hostaddr connection parameter. This can be set instead of or in addition to PGHOST to avoid DNS lookup overhead. |
| PGPORT | behaves the same as the port connection parameter.|
| PGDATABASE | behaves the same as the dbname connection parameter. |
| PGUSER | behaves the same as the user connection parameter. |
| PGPASSWORD | behaves the same as the password connection parameter. Use of this environment variable is not recommended for security reasons, as some operating systems allow non-root users to see process environment variables via ps; instead consider using the ~/.pgpass file (see Section 31.15). |
| PGPASSFILE | specifies the name of the password file to use for lookups. If not set, it defaults to ~/.pgpass (see Section 31.15). |

If you have any of these `PG*` environment variables defined in your terminal/console window you may have problems.

Our system has chosen to use "DB" as a prefix for environment variables associated with the database service.  This is consistent with the Rails/ActiveRecord gnosticism with regard to the database management system.  It is possible to create a Rails-based application that can make use of any database management system.  We chose PostgreSQL as our initial component.  We may change our slection in the future to some other database management system.

The `.env*` files only define `DB*` environment variables.  Having `PGPORT` defined in the terminal/console window and having `DBPORT` defined in the `.env` file will result in confusion.  To mitigate this confusion we make use of the ability of the `docker-compose` utility to set the value of some environment variables.

In the `config/docker/db_service.yml` file within the `environment` YAML element we have this configuration defined:

```
- PGPORT=$DBPORT
```

