# db/seeds.rb
# This file is executed during `rake db:reset` and `db:setup` and `db:seed` tasks

# If this file is executed when the RAILS_ENV is development it will
# only seed the static text files unless the DBRESET environment variable is true
# WHen DBRESET is true and RAILS_ENV is development, the database will have
# been reset to a blank condition prior to this file being executed and the
# this file will populate the entire development dataset.

require 'seeder.rb'

require 'csv_seeder.rb'
include CsvSeeder

require 'sql_seeder.rb'
include SqlSeeder


#########################################################
## seed static text tables from csv files located in db/csv

puts "\nLoading static text tables."

[
  CareCategory,
  ConsultationOrder,
  ConsultationStatus,
  ConsultationType,
  MedicalSpecialty,
  OtherHealthInsurance,
  ReferralDocumentType,
  ReferralReason,
  ReferralType,

].each do |model_class_name|
  load_static_text_table_from_csv(klass: model_class_name,
    common_fields: { deleted_at: nil },
    column_mappings: ['sequence','title'],
    use_copy_from: false
  )
end

unless seed_from_sql(Visn, reset_only: false)
  load_static_text_table_from_csv(
    klass:            Visn,
    column_mappings: ['sequence','deleted_at','region','name'],
    use_copy_from: false
  )
end

unless seed_from_sql(Site, reset_only: true)
  load_static_text_table_from_csv(
    klass:            Site,
    column_mappings:  ['name','address','city','state','zip_code'],
    common_fields:    {country: 'USA'},
    use_copy_from:    false
  )
end

unless seed_from_sql(DiagnosisCode, reset_only: true)
  load_static_text_table_from_csv(
    klass:  DiagnosisCode,
    column_mappings: ['version_code','description'],
    use_copy_from:    false
  )
end

unless seed_from_sql( Clinic, reset_only: true)
  load_static_text_table_from_csv(
    klass:            Clinic,
    column_mappings:  ['site_id','clinic_id','name'],
    use_copy_from:    false
  )
end


### Load issue categories for DOM-127 Request for Support ###
%w[
  Performance
  Security
  Functional
  Access
  Forgot_Password
  Other
].each do |category|
  SupportRequestCategory.find_or_create_by(name: category.gsub('_', ' ').titlecase)
end

org1 = SupportRequestOrganization.create(short_name: "C&P", long_name:"Compensation and Pension")

org2 = SupportRequestOrganization.create(short_name: "CPP", long_name:"Community Provider Portal")

#############################################################


#########################################################
## define the ReferralStatus

print "Loading static text for ReferralStatus ... "

[
  {
    status:               "new",
    update_description:   "",
    referral_queue:       "vha_cc",   # Only users who have one of these roles will have this status in the referral_queue
    filterable_by_roles:  "vha_cc"    # Only users who have one of these roles will have this status in the detail filter pane
  },
  {
    status:               "prep",
    update_description:   "The Referral is being prepared for the Community Provider.",
    referral_queue:       "vha_cc",
    filterable_by_roles:  "vha_cc"
  },
  {
    status:               "assigned",
    update_description:   "The Referral contains all medical information needed for the first visit and has been assigned to a Community Provider. <b>Note:</b> The Communtiy Provider will receive a system-generated notification email as soon as the <b>Submit</b> button below is clicked.",
    referral_queue:       "vha_cc, non_vha",
    filterable_by_roles:  "vha_cc, non_vha"
  },
  {
    status:               "accepted",
    update_description:   "The Community Provider has successfully returned medical records for the first appointment. Additional appointments and medical documents are expected.",
    referral_queue:       "non_vha",
    filterable_by_roles:  "vha_cc, non_vha"
  },
  {
    status:               "review_pending",
    update_description:   "The Referral has been sent to the VA for approval.",
    referral_queue:       "vha_cc, non_vha",
    filterable_by_roles:  "vha_cc, non_vha"
  },
  {
    status:               "information_needed",
    update_description:   "The Referral has been reviewed and returned to the Community Provider with a request for additional information.",
    referral_queue:       "vha_cc, non_vha",
    filterable_by_roles:  "vha_cc, non_vha"
  },
  {
    status:               "complete",
    update_description:   "The Referral is complete and all medical document have been received.",
    referral_queue:       "",
    filterable_by_roles:  "vha_cc, non_vha"
  }
].each do |entry|
  ReferralStatus.find_or_create_by(
    name:                 entry[:status].gsub('_', ' ').titlecase,
    code:                 entry[:status].upcase,
    update_description:   entry[:update_description],
    referral_queue:       entry[:referral_queue].downcase,
    filterable_by_roles:  entry[:filterable_by_roles].downcase
  )
end

puts "done"

#########################################################
## define the test ExaminationState

print "Loading static text for ExaminationState ... "

# FIXME: spelling error: Canceled not Cancelled
%w[
  Pending
  Ready_to_be_Scheduled
  Scheduled
  In_Progress
  Completed
  Reviewed
  Submitted
  Rejected
  Pending_Cancellation
  Cancelled
  Reschedule
].each do |status|
  ExaminationState.find_or_create_by(name: status.gsub('_', ' ').titlecase, code: status.upcase)
end

puts "done"


#########################################################
## define the test ExamRequestState

print "Loading static text for ExamRequestState ... "

# FIXME: spelling error Canceled not Cancelled
%w[
  New
  In_Progress
  Pending_Reported
  Pending Cancellation
  Cancelled
  Canceled by MAS
  Canceled by RO
  Completed
  Completed Printed by RO
  Released to RO Not Printed
  Re-routed New
  Re-routed accepted
  Re-routed rejected
].each do |status|
  ExamRequestState.find_or_create_by(name: status.gsub('_', ' ').titlecase, code: status.upcase)
end

puts "done"


#########################################################
## define the test ClarificationType

print "Loading static text for ClarificationType ... "

%w[
  Request
  Response
].each do |status|
  ClarificationType.find_or_create_by(name: status.gsub('_', ' ').titlecase, code: status.upcase)
end

# BEGIN
# FIX-ME: These below rows need to be there until the above three sections (ExaminationState, ExamRequestState) are fixed
#         We are moving them our of the if statement (that runs if you are deploying in development mode), so that the application
#         can be deployed in 'production' mode
  ExaminationState.find_or_create_by(:name => "Pending", :code => "PENDING")
  ExaminationState.find_or_create_by(:name => "Ready to be Scheduled", :code => "READY_TO_BE_SCHEDULED")
  ExaminationState.find_or_create_by(:name => "Scheduled", :code => "SCHEDULED")
  ExaminationState.find_or_create_by(:name => "In Progress", :code => "IN_PROGRESS")
  ExaminationState.find_or_create_by(:name => "Completed", :code => "COMPLETED")
  ExaminationState.find_or_create_by(:name => "Reviewed", :code => "REVIEWED")
  ExaminationState.find_or_create_by(:name => "Submitted", :code => "SUBMITTED")
  ExaminationState.find_or_create_by(:name => "Rejected", :code => "REJECTED")
  ExaminationState.find_or_create_by(:name => "Pending Cancellation", :code => "PENDING_CANCELLATION")
  ExaminationState.find_or_create_by(:name => "Cancelled", :code => "CANCELLED")
  ExaminationState.find_or_create_by(:name => "Reschedule", :code => "RESCHEDULE")

  ExamRequestState.find_or_create_by(:name => "New", :code => "NEW")
  ExamRequestState.find_or_create_by(:name => "In Progress", :code => "IN_PROGRESS")
  ExamRequestState.find_or_create_by(:name => "Pending, Reported", :code => "PENDING_REPORTED")
  ExamRequestState.find_or_create_by(:name => "Pending Cancellation", :code => "PENDING_CANCELLATION")
  ExamRequestState.find_or_create_by(:name => "Cancelled", :code => "CANCELLED")
  ExamRequestState.find_or_create_by(:name => "Canceled by MAS", :code => "CANCELED_BY_MAS")
  ExamRequestState.find_or_create_by(:name => "Canceled by RO", :code => "CANCELED_BY_RO")
  ExamRequestState.find_or_create_by(:name => "Completed", :code => "COMPLETED")
  ExamRequestState.find_or_create_by(:name => "Completed, Printed by RO", :code => "COMPLETED_PRINTED_BY_RO")
  ExamRequestState.find_or_create_by(:name => "Released to RO, Not Printed", :code => "RELEASED_TO_RO_NOT_PRINTED")
  ExamRequestState.find_or_create_by(:name => "Re-routed, New", :code => "REROUTED_NEW")
  ExamRequestState.find_or_create_by(:name => "Re-routed, accepted", :code => "REROUTED_ACCEPTED")
  ExamRequestState.find_or_create_by(:name => "Re-routed, rejected", :code => "REROUTED_REJECTED")

  ClarificationType.find_or_create_by(:name => "Request", :code => "REQUEST")
  ClarificationType.find_or_create_by(:name => "Response", :code => "RESPONSE")
# END

puts "done"


#########################################################
## Loading the data from db/ecm_seeds.rb via
## rake eval_con_man:seed
##
## The following models are impacted:
##      DiagnosisModifier
##      Diagnosis
##      DmAssignment
##      Symptom
##      MinorSystem
##      MajorSystem
##      EvaluationTemplate


print "Invoking rake task eval_con_man:seed ... "
unless EvaluationTemplate.exists? 
  Rake::Task["eval_con_man:seed"].invoke

  puts "done"

  print "Invoking rake task eval_con_man:load_html ..."

  Rake::Task["eval_con_man:load_html"].invoke

  puts "done"
end

#########################################################
## Loading the data from db/vbms_r_fg_init.rb via
## rake vbms_r_fg:init
##
## The following models are impacted:
##      VbmsRFactGroup

print "Invoking rake task vbms_r_fg:init ... "

Rake::Task["vbms_r_fg:init"].invoke

puts "done"

  ###############################################################################
  #### Sites - The below 5 sites were provided earlier in the project
  #### These sites have to be created outside the 'development' test data section
  #### as they are valid sites
  #### (These sites cannot be moved to the sites csv file as they are being used
  #### (See below for details on these sites usage)
  ###############################################################################

  print "Creating the 7 additional sites ... "

  site1 = Site.find_or_create_by(
              name: "Chicago Medical Center",
              address: "1111 Silver Avenue",
              city: "Chicago",
              state: "IL",
              zip_code: "60611",
              country: "USA")

  site2 = Site.find_or_create_by(
              name: "Dallas VA Clinic",
              address: "2222 Rush Avenue",
              city: "Dallas",
              state: "TX",
              zip_code: "75418",
              country: "USA")

  site3 = Site.find_or_create_by(
              name: "QTC",
              facility_id: "QTC VBA Vendor",
              address: "21700 Copley Dr., Ste 200",
              city: "Diamond Bar",
              state: "CA",
              zip_code: "91765-2219",
              country: "USA")

  site4 = Site.find_or_create_by(
              name: "VES",
              facility_id: "VES VBA Vendor",
              address: "2707 North Loop W., Suite 1000",
              city: "Houston",
              state: "TX",
              zip_code: "77008",
              country: "USA")

  site5 = Site.find_or_create_by(
              name: "VetFed",
              facility_id: "VetFed VBA Vendor",
              address: "21700 Copley Dr., Ste 200",
              city: "Diamond Bar",
              state: "CA",
              zip_code: "91765-2219",
              country: "USA")

  site6 = Site.find_or_create_by(
              name: "Atlanta VA Medical Center",
              facility_id: "508",
              address: "1670 Clairmont Road",
              city: "Decatur",
              state: "GA",
              zip_code: "30033",
              country: "USA")

  site7 = Site.find_or_create_by(
              name: "CSRA ITC",
              facility_id: "777",
              address: "6300 Texas Ave.",
              city: "Bossier City",
              state: "LA",
              zip_code: "71111",
              country: "USA")

  puts "done"
#########################################################
## make up some test data for development

if Rails.env.development?  &&  DBRESET

  puts "\nLoading development test data."


  #########################################################
  ## define test facilities and providers

  unless seed_from_sql( Facility, Provider, reset_only: true )
    puts "\n... Facilities table."

    50.times do |facility_number|
      facility_record = Seeder::FacilitySeed.new(facility_number)
    end

    puts "\n... Provider table from NPI Registry with zip codes ..."

    zipcodes = %w[20002 23233 71111 73501]

    zipcodes.each do |postal_code|
      printf "%s ", postal_code
      NpiRegistry.filter('postal_code' => postal_code)
    end
    puts "done"
  end



  #########################################################
  ## define test users

  puts "... User table."

  medical_assistant = User.create(
              email: "team@adhocteam.us",
              password: "Sms!123456789",
              first_name: "Medical",
              last_name: "Assistant",
              authorization_state: 'authorized',
              roles: ['medical_assistant']) unless User.exists?(email: "team@adhocteam.us")

  supervisor = User.create(
              email: "supervisor@adhocteam.us",
              password: "Sms!123456789",
              first_name: "Super",
              last_name: "Visor",
              authorization_state: 'authorized',
              roles: ['supervisor']) unless User.exists?(email: "supervisor@adhocteam.us")

  examiner = User.create(
              email: "examiner@adhocteam.us",
              password: "Sms!123456789",
              first_name: "Exam",
              last_name: "Iner",
              authorization_state: 'authorized',
              roles: ["examiner"]) unless User.exists?(email: "")

  app_admin = User.create(
              email: "samantha@smith.com",
              password: "Sms!123456789",
              authorization_state: "authorized",
              first_name: "Samantha",
              last_name: "Smith",
              # FIXME: at the end of CPP sprint 15 these two lines were removed
              #        from the integration branch.  So if they are truly not needed,
              #        come back here and delete this dead code in a future sprint.
              # authorization_state: 'authorized',
              roles: ["app_admin"]) unless User.exists?(email: "samantha@smith.com")

  site_user1 = User.create(
              email: "mike@wallace.com",
              password: "Sms!123456789",
              authorization_state: "authorized",
              first_name: "Mike",
              last_name: "Wallace") unless User.exists?(email: "mike@wallace.com")

  site_user2 = User.create(
              email: "eric@wilson.com",
              password: "Sms!123456789",
              authorization_state: "authorized",
              first_name: "Eric",
              last_name: "Wilson") unless User.exists?(email: "eric@wilson.com")

  all_roles_QTC_user = User.create(
              email: "all@qtc.com",
              password: "Sms!123456789",
              authorization_state: "authorized",
              first_name: "All",
              last_name: "Qtc") unless User.exists?(email: "all@qtc.com")

  admin_QTC_user = User.create(
              email: "admin@qtc.com",
              password: "Sms!123456789",
              authorization_state: "authorized",
              first_name: "Admin",
              last_name: "Qtc") unless User.exists?(email: "admin@qtc.com")

  triage_QTC_user = User.create(
              email: "triage@qtc.com",
              password: "Sms!123456789",
              authorization_state: "authorized",
              first_name: "Triage",
              last_name: "Qtc") unless User.exists?(email: "triage@qtc.com")

  scheduling_QTC_user = User.create(
              email: "scheduling@qtc.com",
              password: "Sms!123456789",
              authorization_state: "authorized",
              first_name: "Scheduling",
              last_name: "Qtc") unless User.exists?(email: "scheduling@qtc.com")

  clinician_QTC_user = User.create(
              email: "clinician@qtc.com",
              password: "Sms!123456789",
              authorization_state: "authorized",
              first_name: "Clinician",
              last_name: "Qtc") unless User.exists?(email: "clinician@qtc.com")

  sclinician_QTC_user = User.create(
              email: "super_clinician@qtc.com",
              password: "Sms!123456789",
              authorization_state: "authorized",
              first_name: "Super",
              last_name: "Qtc") unless User.exists?(email: "super_clinician@qtc.com")

  qa_QTC_user = User.create(
              email: "qa@qtc.com",
              password: "Sms!123456789",
              authorization_state: "authorized",
              first_name: "Qa",
              last_name: "Qtc") unless User.exists?(email: "qa@qtc.com")

  # CPP-specific test users

  cheryl_howard_npis = %w(
    1043303563
    1083089809
    1225182579
    1225340383
    1295124063
    1295862415
    1366499295
    1598199259
    1659696045
    1710342811
  )

  vha_cc = User.create(
              email: "vha_cc@example.com",
              password: "Sms!123456789",
              first_name: "VHA CC",
              last_name: "TestUser",
              authorization_state: 'authorized',
              roles: ["vha_cc"]) unless User.exists?(email: "vha_cc@example.com")

  non_vha = User.create(
              email: "non_vha@example.com",
              password: "Sms!123456789",
              first_name: "Non VHA",
              last_name: "TestUser",
              authorization_state: 'authorized',
              npi: cheryl_howard_npis.pop,
              roles: ["non_vha"]) unless User.exists?(email: "non_vha@example.com")

  vha_cc2 = User.create(
              email: "vhacc.cpptestuser@gmail.com",
              password: "Sms!123456789",
              first_name: "VHA CC2",
              last_name: "TestUser",
              authorization_state: 'authorized',
              roles: ["app_admin","vha_cc"]) unless User.exists?(email: "vhacc.cpptestuser@gmail.com")

  non_vha2 = User.create(
              email: "nonvha.cpptestuser@gmail.com",
              password: "Sms!123456789",
              first_name: "Non VHA2",
              last_name: "TestUser",
              authorization_state: 'authorized',
              npi: cheryl_howard_npis.pop(2),
              roles: ["non_vha"]) unless User.exists?(email: "nonvha.cpptestuser@gmail.com")

  vha_cc3 = User.create(
              email: "PII                 ",
              password: "Sms!123456789",
              first_name: "Amy",
              last_name: "Rosenthal",
              authorization_state: 'authorized',
              roles: ["vha_cc"]) unless User.exists?(email: "PII                 ")

  non_vha3 = User.create(
              email: "arosenthal@meetveracity.com",
              password: "Sms!123456789",
              first_name: "Amy",
              last_name: "Rosenthal",
              authorization_state: 'authorized',
              npi: cheryl_howard_npis.pop(2),
              roles: ["non_vha"]) unless User.exists?(email: "arosenthal@meetveracity.com")

  non_vha4 = User.create(
              email: "tmcmurdo@meetveracity.com",
              password: "Sms!123456789",
              first_name: "Theresa",
              last_name: "McMurdo",
              authorization_state: 'authorized',
              npi: cheryl_howard_npis.pop(2),
              roles: ["non_vha"]) unless User.exists?(email: "tmcmurdo@meetveracity.com")

  non_vha5 = User.create(
              email: "jrobinson@meetveracity.com",
              password: "Sms!123456789",
              first_name: "Jeff",
              last_name: "Robinson",
              authorization_state: 'authorized',
              npi: cheryl_howard_npis.pop(2),
              roles: ["non_vha"]) unless User.exists?(email: "jrobinson@meetveracity.com")


  # Adding customer test accounts for CPP
  customer_test_accounts = %w[
    PII                      
    PII                    
    PII              
  ].each do |email|
    name = email.split('@').first.split('.')
    User.create(
              email:                email,
              password:             "Sms!123456789",
              first_name:           name.first,
              last_name:            name.last,
              authorization_state:  'authorized',
              roles:                ["vha_cc"]) unless User.exists?(email: email)

  end


  # add empty UserPreference object to each user, with time zone
  # corresponding to index
  US_TIME_ZONES = ActiveSupport::TimeZone.us_zones.map &:name
  User.all.each_with_index do |user, index|
    time_zone_string     = US_TIME_ZONES[index % US_TIME_ZONES.count]
    user.user_preference = UserPreference.new(time_zone: time_zone_string)
    if user.is_vha_cc?
      user.user_preference.vista_duz       = 'mLb+ZHYXCWt7R9WvfFhWHBwdOKeT5ZsuZJOmjLIU8ko='
      user.user_preference.vista_user_name = 'WASH,PAUL E'
      user.user_preference.vista_site_id   = '516'
      user.user_preference.save(validation: false)
    end
    user.save(validation: false)
  end

  #########################################################
  ## Updating test user's site and roles

  puts "... Adding site and site roles to CUI-users."

  app_admin.present? ? (app_admin.add_site(site3); app_admin.get_site_roles(site3).update(admin: true, triage: true, scheduling: true, clinician: true, super_clinician: true, qa: true)) : false
  app_admin.present? ? (app_admin.add_site(site6); app_admin.get_site_roles(site6).update(admin: true, triage: true, scheduling: true, clinician: true, super_clinician: true, qa: true)) : false
  site_user1.present? ? (site_user1.add_site(site3); site_user1.get_site_roles(site3).update(admin: true, triage: true, scheduling: true, clinician: true, super_clinician: true, qa: true)) : false
  site_user2.present? ? (site_user2.add_site(site3); site_user2.get_site_roles(site3).update(admin: true, triage: true, scheduling: true, clinician: true, super_clinician: true, qa: true)) : false
  all_roles_QTC_user.present? ? (all_roles_QTC_user.add_site(site3); all_roles_QTC_user.get_site_roles(site3).update(admin: true, triage: true, scheduling: true, clinician: true, super_clinician: true, qa: true)) : false
  admin_QTC_user.present? ? (admin_QTC_user.add_site(site3); admin_QTC_user.get_site_roles(site3).update(admin: true)) : false
  triage_QTC_user.present? ? (triage_QTC_user.add_site(site3); triage_QTC_user.get_site_roles(site3).update(triage: true)) : false
  scheduling_QTC_user.present? ? (scheduling_QTC_user.add_site(site3); scheduling_QTC_user.get_site_roles(site3).update(scheduling: true)) : false
  clinician_QTC_user.present? ? (clinician_QTC_user.add_site(site3); clinician_QTC_user.get_site_roles(site3).update(clinician: true)) : false
  sclinician_QTC_user.present? ? (sclinician_QTC_user.add_site(site3); sclinician_QTC_user.get_site_roles(site3).update(super_clinician: true)) : false
  qa_QTC_user.present? ? (qa_QTC_user.add_site(site3); qa_QTC_user.get_site_roles(site3).update(qa: true)) : false


  #########################################################
  ## Adding the Evaluation specs

  puts "... EvaluationSpec table."

  #########################################################
  ## Seed the Evaluation Specs

  seed_specs_log = File.open(LOG_DIR+'seed_specs.log','w')

  print "... Executing EvaluationSpec.sync_specs ... "
  EvaluationSpec.sync_specs(nil, "app/exams/*.yml", seed_specs_log)
  puts "done"

  seed_specs_log.close

  ##########################################################
  ## Send exam requests to CUI

  puts "... Seeding exam requests in CUI"

  seed_from_sql( Claim, ExamRequest, Contention, ContentionDetail, ContentionsDbqInformation, ContentionObject, RequestObject, reset_only: true )

  puts "Done"


  #########################################################
  ## define test Veteran, consultation, referral
  ## TODO: add test data for the sub-referral tables

  puts "\n... Veteran, Consultation, Referral tables."

  RANDOMIZED_VETERAN_COUNT = 10

  # PP-496: adding more test veterans named after famous U.S. Generals who died in WWII
  US_GENERALS = [
    {last_name: 'Andersen', first_name: 'James', middle_name: 'Roy', gender: 'M', date_of_birth: '1/1/1904'},
    {last_name: 'Anderson', first_name: 'Alexander', middle_name: 'Edward', gender: 'M', date_of_birth: '1/1/1889'},
    {last_name: 'Andrews', first_name: 'Frank', middle_name: 'Maxwell', gender: 'M', date_of_birth: '1/1/1884'},
    {last_name: 'Barth Jr.', first_name: 'Charles', middle_name: 'Henry', gender: 'M', date_of_birth: '1/1/1903'},
    {last_name: 'Berman', first_name: 'Morris', middle_name: '', gender: 'M', date_of_birth: '1/1/1891'},
    {last_name: 'Boudinot', first_name: 'Truman', middle_name: 'Everett', gender: 'M', date_of_birth: '1/1/1895'},
    {last_name: 'Brann', first_name: 'Donald', middle_name: 'Weldon', gender: 'M', date_of_birth: '1/1/1895'},
    {last_name: 'Braun Jr.', first_name: 'Gustav', middle_name: 'Joseph', gender: 'M', date_of_birth: '1/1/1895'},
    {last_name: 'Buckner Jr.', first_name: 'Simon', middle_name: 'Bolivar', gender: 'M', date_of_birth: '1/1/1886'},
    {last_name: 'Carrington', first_name: 'Gordon', middle_name: 'de Launey', gender: 'M', date_of_birth: '1/1/1894'},
    {last_name: 'Castle', first_name: 'Frederick', middle_name: 'Walker', gender: 'M', date_of_birth: '1/1/1908'},
    {last_name: 'Chamberlin', first_name: 'Harry', middle_name: 'Dwight', gender: 'M', date_of_birth: '1/1/1887'},
    {last_name: 'Craig', first_name: 'Malin', middle_name: '', gender: 'M', date_of_birth: '1/1/1875'},
    {last_name: 'Dalton II', first_name: 'James', middle_name: 'Leo', gender: 'M', date_of_birth: '1/1/1910'},
    {last_name: 'Darby', first_name: 'William', middle_name: 'Orlando', gender: 'M', date_of_birth: '1/1/1911'},
    {last_name: 'Dargue', first_name: 'Herbert', middle_name: 'Arthur', gender: 'M', date_of_birth: '1/1/1886'},
    {last_name: 'Davis', first_name: 'Dwight', middle_name: 'Filley', gender: 'M', date_of_birth: '1/1/1879'},
    {last_name: 'Davison', first_name: 'Donald', middle_name: 'Angus', gender: 'M', date_of_birth: '1/1/1892'},
    {last_name: 'De Jesus', first_name: 'Simeon', middle_name: '', gender: 'M', date_of_birth: '1/1/1894'},
    {last_name: 'Duncan', first_name: 'Asa', middle_name: 'North', gender: 'M', date_of_birth: '1/1/1892'},
    {last_name: 'Easley', first_name: 'Claudius', middle_name: 'Miller', gender: 'M', date_of_birth: '1/1/1891'},
    {last_name: 'Eaton', first_name: 'William', middle_name: 'Herbert', gender: 'M', date_of_birth: '1/1/1906'},
    {last_name: 'Emery', first_name: 'Ambrose', middle_name: 'Robert', gender: 'M', date_of_birth: '1/1/1883'},
    {last_name: 'Forrest', first_name: 'Nathan', middle_name: 'Bedford', gender: 'M', date_of_birth: '1/1/1905'},
    {last_name: 'Fort', first_name: 'Guy', middle_name: 'Osborne', gender: 'M', date_of_birth: '1/1/1879'},
    {last_name: 'Gardner', first_name: 'John', middle_name: 'Henry', gender: 'M', date_of_birth: '1/1/1893'},
    {last_name: 'Garlington', first_name: 'Creswell', middle_name: '', gender: 'M', date_of_birth: '1/1/1887'},
    {last_name: 'George', first_name: 'Harold', middle_name: 'Huston', gender: 'M', date_of_birth: '1/1/1892'},
    {last_name: 'Gibbons', first_name: 'Lloyd', middle_name: 'Henry', gender: 'M', date_of_birth: '1/1/1895'},
    {last_name: 'Godfrey', first_name: 'Stuart', middle_name: 'Chapin', gender: 'M', date_of_birth: '1/1/1886'},
    {last_name: 'Goodrich', first_name: 'Donald', middle_name: 'Reuben', gender: 'M', date_of_birth: '1/1/1894'},
    {last_name: 'Graves', first_name: 'Davis', middle_name: 'Dunbar', gender: 'M', date_of_birth: '1/1/1903'},
    {last_name: 'Harmon Jr.', first_name: 'Millard', middle_name: 'Fillmore', gender: 'M', date_of_birth: '1/1/1888'},
    {last_name: 'Harms', first_name: 'Henry', middle_name: 'William', gender: 'M', date_of_birth: '1/1/1887'},
    {last_name: 'Holland', first_name: 'Thomas', middle_name: 'Leroy', gender: 'M', date_of_birth: '1/1/1879'},
    {last_name: 'Hyde', first_name: 'James', middle_name: 'Francis Clark', gender: 'M', date_of_birth: '1/1/1894'},
    {last_name: 'Jackson', first_name: 'Stonewall', middle_name: '', gender: 'M', date_of_birth: '1/1/1891'},
    {last_name: 'Johnson', first_name: 'Herbert', middle_name: 'Thomas', gender: 'M', date_of_birth: '1/1/1872'},
    {last_name: 'Keerans Jr.', first_name: 'Charles', middle_name: 'Leslie', gender: 'M', date_of_birth: '1/1/1899'},
    {last_name: 'Lackland', first_name: 'Frank', middle_name: 'Dorwin', gender: 'M', date_of_birth: '1/1/1884'},
    {last_name: 'Layman', first_name: 'Walter', middle_name: 'Gilbert', gender: 'M', date_of_birth: '1/1/1888'},
    {last_name: 'Lim', first_name: 'Vicente', middle_name: '', gender: 'M', date_of_birth: '1/1/1889'},
    {last_name: 'Loutzenheiser', first_name: 'Joe', middle_name: 'L.', gender: 'M', date_of_birth: '1/1/1899'},
    {last_name: 'Lyman', first_name: 'Albert', middle_name: 'Kuali Brickwood', gender: 'M', date_of_birth: '1/1/1885'},
    {last_name: 'Lyon', first_name: 'Alfred', middle_name: 'Jefferson', gender: 'M', date_of_birth: '1/1/1892'},
    {last_name: 'Mahin', first_name: 'Frank', middle_name: 'Cadle', gender: 'M', date_of_birth: '1/1/1887'},
    {last_name: 'Maynard', first_name: 'John', middle_name: 'Blackwell', gender: 'M', date_of_birth: '1/1/1887'},
    {last_name: 'McBride', first_name: 'Allan', middle_name: 'Clay', gender: 'M', date_of_birth: '1/1/1885'},
    {last_name: 'McCornack', first_name: 'Condon', middle_name: 'Carlton', gender: 'M', date_of_birth: '1/1/1880'},
    {last_name: 'McDaniel', first_name: 'Arthur', middle_name: 'Bee', gender: 'M', date_of_birth: '1/1/1895'},
    {last_name: 'McNair', first_name: 'Lesley', middle_name: 'James', gender: 'M', date_of_birth: '1/1/1883'},
    {last_name: 'Naiden', first_name: 'Earl', middle_name: 'Larue', gender: 'M', date_of_birth: '1/1/1894'},
    {last_name: 'Netherwood', first_name: 'Douglas', middle_name: 'Blackshaw', gender: 'M', date_of_birth: '1/1/1885'},
    {last_name: 'Newgarden', first_name: 'Paul', middle_name: 'Woolever', gender: 'M', date_of_birth: '1/1/1892'},
    {last_name: 'Olds', first_name: 'Robert', middle_name: '', gender: 'M', date_of_birth: '1/1/1896'},
    {last_name: 'Patch', first_name: 'Alexander', middle_name: 'McCarrell', gender: 'M', date_of_birth: '1/1/1889'},
    {last_name: 'Patrick', first_name: 'Edwin', middle_name: 'Davies', gender: 'M', date_of_birth: '1/1/1894'},
    {last_name: 'Patton Jr.', first_name: 'George', middle_name: 'Smith', gender: 'M', date_of_birth: '1/1/1885'},
    {last_name: 'Penington', first_name: 'Carlos', middle_name: 'Alden', gender: 'M', date_of_birth: '1/1/1878'},
    {last_name: 'Powell', first_name: 'William', middle_name: 'Dan', gender: 'M', date_of_birth: '1/1/1893'},
    {last_name: 'Pratt', first_name: 'Don', middle_name: 'Forrester', gender: 'M', date_of_birth: '1/1/1892'},
    {last_name: 'Ramey', first_name: 'Howard', middle_name: 'Knox', gender: 'M', date_of_birth: '1/1/1896'},
    {last_name: 'Roderick', first_name: 'Thomas', middle_name: 'Edison', gender: 'M', date_of_birth: '1/1/1892'},
    {last_name: 'Roosevelt Jr.', first_name: 'Theodore', middle_name: '', gender: 'M', date_of_birth: '1/1/1887'},
    {last_name: 'Rose', first_name: 'Maurice', middle_name: '', gender: 'M', date_of_birth: '1/1/1899'},
    {last_name: 'Russell', first_name: 'Clinton', middle_name: 'Warden', gender: 'M', date_of_birth: '1/1/1891'},
    {last_name: 'Santos', first_name: 'Paulino', middle_name: 'Torres', gender: 'M', date_of_birth: '1/1/1890'},
    {last_name: 'Searby', first_name: 'Edmund', middle_name: 'Wilson', gender: 'M', date_of_birth: '1/1/1896'},
    {last_name: 'Segundo Y Ventura', first_name: 'Fidel', gender: 'M', date_of_birth: '1/1/1894'},
    {last_name: 'Taylor', first_name: 'Victor', middle_name: 'Vaughan', gender: 'M', date_of_birth: '1/1/1893'},
    {last_name: 'Taylor', first_name: 'Willis', middle_name: 'Ratcliffe', gender: 'M', date_of_birth: '1/1/1897'},
    {last_name: 'Tinker', first_name: 'Clarence', middle_name: 'Leonard', gender: 'M', date_of_birth: '1/1/1887'},
    {last_name: 'Walker', first_name: 'Kenneth', middle_name: 'Newton', gender: 'M', date_of_birth: '1/1/1898'},
    {last_name: 'Walker', first_name: 'Nelson', middle_name: 'Macy', gender: 'M', date_of_birth: '1/1/1891'},
    {last_name: 'Wash', first_name: 'Carlyle', middle_name: 'Hilton', gender: 'M', date_of_birth: '1/1/1889'},
    {last_name: 'Watson', first_name: 'Edwin', middle_name: 'Martin', gender: 'M', date_of_birth: '1/1/1883'},
    {last_name: 'Weaver', first_name: 'Walter', middle_name: 'Reed', gender: 'M', date_of_birth: '1/1/1885'},
    {last_name: 'Weed', first_name: 'Frank', middle_name: 'Watkins', gender: 'M', date_of_birth: '1/1/1881'},
    {last_name: 'Wharton', first_name: 'James', middle_name: 'Edward', gender: 'M', date_of_birth: '1/1/1894'},
    {last_name: 'Wilson', first_name: 'Russell', middle_name: 'Alger', gender: 'M', date_of_birth: '1/1/1905'},
    {last_name: 'Wing', first_name: 'Leonard', middle_name: 'Fish', gender: 'M', date_of_birth: '1/1/1893'}
  ]

  TOTAL_VETERAN_COUNT = (RANDOMIZED_VETERAN_COUNT + US_GENERALS.count)

  RANDOMIZED_VETERAN_COUNT.times do |veteran_number|

    puts "Fake Veteran #{veteran_number+1} of #{RANDOMIZED_VETERAN_COUNT}."

    veteran_record  = Seeder::VeteranSeed.new(veteran_number).veteran
    veteran_id      = veteran_record.id

    5.times do |consultation_number|
      consultation_record = Seeder::ConsultationSeed.new(veteran_record, consultation_number).consultation
      consultation_id     = consultation_record.id

      20.times do |referral_number|
        referral_seed_record  = Seeder::ReferralSeed.new(veteran_id, consultation_id, referral_number)
        referral_id           = referral_seed_record.referral.id

        2.times do |referral_note_number|
          Seeder::ReferralNoteSeed.new(veteran_id, consultation_id, referral_id, referral_note_number)
        end

        2.times do |referral_appointment_number|
          Seeder::ReferralAppointmentSeed.new(veteran_id, consultation_id, referral_id, referral_appointment_number)
        end

        2.times do |referral_approval_number|
          Seeder::ReferralApprovalSeed.new(veteran_id, consultation_id, referral_id, referral_approval_number)
        end

        2.times do |referral_document_number|
          Seeder::ReferralDocumentSeed.new(veteran_id, consultation_id, referral_id, referral_document_number)
        end

      end # 20.times do |referral_number|
    end # 5.times do |consultation_number|
  end # RANDOMIZED_VETERAN_COUNT.times do |veteran_number|


  print "Adding World War II heroes with fake SSNs ... "
  US_GENERALS.each_with_index do |us_general_data, usg_index|
    vet_number = RANDOMIZED_VETERAN_COUNT + usg_index + 1
    Seeder::VeteranSeed.new((vet_number), us_general_data)
  end

  puts "Done."

end # if Rails.env == "development"
