# app/via_api/scheduling_service.rb
module VIA_API
  module SchedulingService

    class << self

      # constant values that are the same for all appointments
      PURPOSE                  = 'SCHEDULED VISIT'
      APPOINTMENT_LENGTH       = 60

      # default value for appointment fields that have either default or
      # user-inputted values, depending on caller.
      DEFAULT_APPOINTMENT_TYPE = 'REGULAR'


      ##############################################################
      def get_appointment_types(vista_session)

        VIA_API.validate_vista_session(vista_session)
        request  = SCHEDULING_CLIENT.build_request(:get_appointment_types, message: appointment_type_request(vista_session)).body
        request  = VIA_API.add_ser_prefix(request)
        response  = SCHEDULING_CLIENT.call(:get_appointment_types, xml: request, soap_action: false)

        data = response.body

        appointment_types_raw_data = data.dig(:get_appointment_types_response, :appointment_types_array,
                                        :appointment_types, :appointment_types_to)
        appointment_types_raw_data_count = data.dig(:get_appointment_types_response, :appointment_types_array, :count)
        fault_message = data.dig(:get_appointment_types_response, :appointment_types_array, :fault)

        if fault_message.present?
          raise ViaApiFaultError.new(fault_message)
        end

        return data  # FIXME: should return VIA_API::Models::AppointmentType objects from appointment_types_raw_data
      end


      def appointment_type_request(vista_session)

        request = {}

        request[:queryBean] = {
          provider: {
            userId:         vista_session[:duz],
            name:           vista_session[:user_name],
            loginSiteCode:  vista_session[:site_id]
          },
          requestingApp: "requestingApp here"  # FIXME: don't hardcode stuff
        }

        return request
      end


      ##############################################################
      # Saves appointment to VISTA.
      # patient - VIA_API::Models::Veteran object representing patient to be saved.
      # clinic_id - clinic id used to determine where the appointment is taking place.
      # appointment_type - appointment type text, must be a value contained in VIA API
      #    method get_appointment_types above.  Defaults to DEFAULT_APPOINTMENT_TYPE if falsy.
      # timestamp - date and time of appointment, as Ruby object
      # notes - text entered by user
      ##############################################################
      def make_appointment(patient, clinic_id, appointment_type, timestamp, notes, vista_session)

        VIA_API.validate_vista_session(vista_session)
        request  = SCHEDULING_CLIENT.build_request(:make_appointment,
          message: make_appointment_request(patient, clinic_id, appointment_type, timestamp, notes, vista_session)
        ).body

        request = VIA_API.add_ser_prefix(request)
        response  = SCHEDULING_CLIENT.call(:make_appointment, xml: request, soap_action: false)

        data = response.body

        appointment_raw_data = data.dig(:make_appointment_response, :appointment_to)
        fault_message = data.dig(:make_appointment_response, :appointment_to, :fault)

        if fault_message.present?
          raise ViaApiFaultError.new(fault_message)
        end

        appointment = [VIA_API::Models::Appointment.new(appointment_raw_data)]

        return appointment
      end


      def make_appointment_request(patient, clinic_id, appointment_type, timestamp, notes, vista_session)

        request = {}

        request[:queryBean]= {
          provider: {
            userId: vista_session[:duz],
            name: vista_session[:user_name],
            loginSiteCode: vista_session[:site_id]
          },
          patient: {
            localPid: patient.local_pid,
            localSiteId: vista_session[:site_id],
            mpiPid: patient.mpi_pid
          },
          requestingApp:        ENV['VIA_REQ_APP'],
          consumingAppToken:    ENV['VIA_CONS_APP_TOKEN'],
          consumingAppPassword: ENV['VIA_CONS_APP_PASS']
        }

        request[:appointment] = {
          purpose: PURPOSE,
          clinicId: clinic_id,
          timestamp: VIA_API::encode_vista_date(timestamp),
          appointmentLength: APPOINTMENT_LENGTH,
          appointmentType: appointment_type || DEFAULT_APPOINTMENT_TYPE,
          bookNotes: notes
        }

        return request
      end # def make_appointment_request


      ##############################################################
      def cancel_appointment(timestamp, patient, clinic_id, cancel_code, cancel_reason, remarks, vista_session)

        VIA_API.validate_vista_session(vista_session)

        request = SCHEDULING_CLIENT.build_request(:cancel_appointment,
         message: cancel_appointment_request(timestamp, patient, clinic_id, cancel_code,
                  cancel_reason, remarks, vista_session)).body

        request = VIA_API.add_ser_prefix(request)

        response = SCHEDULING_CLIENT.call(:cancel_appointment, xml: request, soap_action: false)

        data = response.body

        appointment_raw_data = data.dig(:cancel_appointment_response, :cancel_appointment_to)
        fault_message = data.dig(:cancel_appointment_response, :cancel_appointment_to, :fault)

        if fault_message.present?
          raise ViaApiFaultError.new(fault_message)
        end

        appointment = [VIA_API::Models::Appointment.new(appointment_raw_data)]

        return appointment
      end

      def cancel_appointment_request(timestamp, patient, clinic_id, cancel_code, cancel_reason, remarks, vista_session)

        request = {}

        request[:queryBean] = {
          provider: {
            userId:         vista_session[:duz],
            name:           vista_session[:user_name],
            loginSiteCode:  vista_session[:site_id]
          },
          patient: {
            localPid:    patient.local_pid,
            localSiteId:  vista_session[:site_id]
          },
          requestingApp:         ENV['VIA_REQ_APP'],
          consumingAppToken:     ENV['VIA_CONS_APP_TOKEN'],
          consumingAppPassword:  ENV['VIA_CONS_APP_PASS']
        }

        request[:appointment] = {
          clinicId:             clinic_id,
          timestamp:            timestamp,
          cancelCode:           cancel_code,
          cancelReason:         cancel_reason,
          remarks:              remarks
        }

        request
      end

      ##############################################################
      #Purpose: page through the list of clinics
      #Start with target field at 0 -- the response is the first 44 clinics
      #For next api call, set target to last clinic name from the first call.
      #The response will be the next 44 clinics.

      def get_clinics(target_clinic, vista_session)

        VIA_API.validate_vista_session(vista_session)

        request = SCHEDULING_CLIENT.build_request(:get_clinics, message: get_clinics_request(target_clinic, vista_session)).body
        request = VIA_API.add_ser_prefix(request)

        response = SCHEDULING_CLIENT.call(:get_clinics, xml: request, soap_action: false)

        data = response.body

        clinics_raw_data = data.dig(:get_clinics_response, :tagged_hospital_location_array, :locations, :hospital_location_to)
        clinics_raw_data_count = data.dig(:get_clinics_response, :tagged_hospital_location_array, :count)
        fault_message = data.dig(:get_clinics_response, :tagged_hospital_location_array, :fault)

        if fault_message.present?
          raise ViaApiFaultError.new(fault_message)
        end

        clinics = []

        if clinics_raw_data_count.to_i > 1
          clinics_raw_data.each do |clinic|
            clinics << VIA_API::Models::Clinic.new(clinic)
          end
        else
          clinics = [VIA_API::Models::Clinic.new(clinics_raw_data)]
        end

        return clinics
      end

      def get_clinics_with_specialties(target_clinic, vista_session)

        VIA_API.validate_vista_session(vista_session)
        request = SCHEDULING_CLIENT.build_request(:get_clinics_with_specialties, message: get_clinics_request(target_clinic, vista_session)).body
        request = VIA_API.add_ser_prefix(request)

        response = SCHEDULING_CLIENT.call(:get_clinics_with_specialties, xml: request, soap_action: false)

        response.body
      end

      def get_clinics_request(target_clinic, vista_session)

        request = {}

        request[:queryBean] = {
          provider: {
            name:           vista_session[:user_name],
            userId:         vista_session[:duz],
            loginSiteCode:  vista_session[:site_id]
          },
          recordSiteCode:        "XXX",
          requestingApp:         ENV['VIA_REQ_APP'],
          consumingAppToken:     ENV['VIA_CONS_APP_TOKEN'],
          consumingAppPassword:  ENV['VIA_CONS_APP_PASS'],
          target:                target_clinic
        }

        request
      end

    end # class << self

  end # module SchedulingService
end # module VIA_API
