require_rel 'via_api/**/*.rb'

module VIA_API

  # datetime format used by VISTA when sending/receiving datetime data
  # (appointment times, etc.)
  VISTA_DATETIME_FORMAT = '%Y%m%d.%H%M%S'

  # EMR SERVICE
  VIA_EMR_WSDL         = File.join(ENV['VIA_API_URL'], 'EmrService?wsdl')
  VIA_EMR_ENDPOINT_URL = File.join(ENV['VIA_API_URL'], 'EmrService')
  VIA_EMR_NAMESPACE    = ENV['VIA_API_ROOT_ENDPOINT']

  # SCHEDULING SERVICE
  VIA_SCHEDULING_WSDL         = File.join(ENV['VIA_API_URL'], 'SchedulingService?wsdl')
  VIA_SCHEDULING_ENDPOINT_URL = File.join(ENV['VIA_API_URL'], 'SchedulingService')
  VIA_SCHEDULING_NAMESPACE    = ENV['VIA_API_ROOT_ENDPOINT']

  # FINDPATIENT SERVICE
  VIA_FINDPATIENT_WSDL         = File.join(ENV['VIA_API_URL'], 'FindPatientService?wsdl')
  VIA_FINDPATIENT_ENDPOINT_URL = File.join(ENV['VIA_API_URL'], 'FindPatientService')
  VIA_FINDPATIENT_NAMESPACE    = ENV['VIA_API_ROOT_ENDPOINT']

  EMR_CLIENT = Savon.client(
   # wsdl: VIA_EMR_WSDL,
   endpoint: VIA_EMR_ENDPOINT_URL,
   namespace: VIA_EMR_NAMESPACE,
   env_namespace: :soapenv,
   namespace_identifier: :ser,
   namespaces: {
    # NOTE: xmlns:xsd and xmlns:xsi are being added by Savon by default which
    #       are not present in the actual request/response xml obtained using SoapUI
    'xmlns:xsd'     => '',
    'xmlns:xsi'     => '',
    'xmlns:soapenv' => 'http://schemas.xmlsoap.org/soap/envelope/',
    'xmlns:ser'     => 'http://DNS        .URL       /'
   },
   log: true,
   logger: Rails.logger,
   pretty_print_xml: true
  )

  SCHEDULING_CLIENT = Savon.client(
   # wsdl: VIA_SCHEDULING_WSDL,
   endpoint: VIA_SCHEDULING_ENDPOINT_URL,
   namespace: VIA_SCHEDULING_NAMESPACE,
   env_namespace: :soapenv,
   namespace_identifier: :ser,
   namespaces: {
    'xmlns:xsd'     => '',
    'xmlns:xsi'     => '',
    'xmlns:soapenv' => 'http://schemas.xmlsoap.org/soap/envelope/',
    'xmlns:ser'     => 'http://DNS        .URL       /'
   },
   log: true,
   logger: Rails.logger,
   pretty_print_xml: true
  )

  FINDPATIENT_CLIENT = Savon.client(
   # wsdl: VIA_FINDPATIENT_WSDL,
   endpoint: VIA_FINDPATIENT_ENDPOINT_URL,
   namespace: VIA_FINDPATIENT_NAMESPACE,
   namespaces: {
    'xmlns:xsd'     => '',
    'xmlns:xsi'     => '',
    'xmlns:soapenv' => 'http://schemas.xmlsoap.org/soap/envelope/',
    'xmlns:ser'     => 'http://DNS        .URL       /'
   },
   env_namespace: :soapenv,
   namespace_identifier: :ser,
   log: true,
   logger: Rails.logger,
   pretty_print_xml: true
  )

  # Base error class for all VIA API errors.  If we add multiple VIA API exception
  # types we can rescue all API errors under one base type.
  class ViaApiError < RuntimeError; end

  # Error returned from VISTA when messages contain <fault> tags.
  class ViaApiFaultError < ViaApiError; end

  # Error returned from VISTA when loginVIA authentication contains <fault> tags.
  class ViaApiAuthenticationError < ViaApiError; end


  class << self

    def login_via_request(params)
      request = {}
      request[:siteCode]    = params[:site_code]
      request[:accessCode]  = params[:access_code]
      request[:verifyCode]  = params[:verify_code]
      request[:queryBean]= {
       patient: {
        localSiteId: params[:site_code]
       },
       requestingApp: ENV['VIA_REQ_APP'],
       consumingAppToken: ENV['VIA_CONS_APP_TOKEN'],
       consumingAppPassword: ENV['VIA_CONS_APP_PASS']
      }

      return request
    end

    # To convert a noko element to hash while parsing a xml
    def x2h(noko_element)
      Hash.from_xml(noko_element.to_s)
    end

    def authenticate(params)

      # NOTE: The duz returned by the loginVIA operation can be used interchangeably within
      #       the VIA web-services like FindPatientService, SchedulingService, EmrService etc.

      login_via_request = EMR_CLIENT.build_request(:login_v_i_a, message: login_via_request(params)).body

      # Adding 'ser:' prefix as necessary to the xml elements.

      login_via_request = add_ser_prefix(login_via_request)

      login_via_response = EMR_CLIENT.call(:login_v_i_a, xml: login_via_request, soap_action: false).body

      # For more complex XML structures,
      # you can pass any other object that is not a Hash and responds to #to_s
      # login_via_response = EMR_CLIENT.call(:login_via, message: login_via_request(params)).body
      fault_message = login_via_response.dig(:login_via_response, :user_to, :fault)

      if fault_message.present?
        raise ViaApiAuthenticationError.new(fault_message)
      end

      {
        duz:       login_via_response[:login_via_response][:user_to][:duz],
        user_name: login_via_response[:login_via_response][:user_to][:name],
        site_id:   login_via_response[:login_via_response][:user_to][:site_id]
      }

    end # def authenticate

    def validate_vista_session(vista_session)

      if  vista_session[:duz].nil?       ||
          vista_session[:user_name].nil? ||
          vista_session[:site_id].nil?
        raise ViaApiAuthenticationError.new("Authentication needed")
      end
    end

    # Convert date in format "YYYYMMDD.HHMMSS" to Ruby Time object.
    # TODO: handle time zones in the future.
    def parse_vista_date(via_api_date)
      Time.strptime(via_api_date, VISTA_DATETIME_FORMAT)
    end

    # Convert date from Ruby Time object to "YYYYMMDD.HHMMSS" format.
    # TODO: handle time zones in the future.
    def encode_vista_date(datetime)
      datetime.strftime(VISTA_DATETIME_FORMAT)
    end

    def add_ser_prefix(xml_request)
      nodes = ["//accessCode", "//verifyCode", "//siteCode", "//queryBean", "//appointment"]
      doc = Nokogiri::XML(xml_request)
      nodes.each do |node|
        unless doc.at_xpath(node).nil?
          doc.at_xpath(node).name = "ser:" + doc.at_xpath(node).name
        end
      end
      return doc.to_xml
    end
  end # class << self
end # module VIA_API
