
# handles indirection for our messy Provider business logic:
# 1. when accessing Referral#provider object data-table:
#   use Referral provider if it exists, otherwise use Consultation provider,
#   and if that does not exist either use Consultation internal static data.
# 2. when accessing Consultation#ordering_provider object data-table:
#   use Consultation provider, and if that does not exist use Consultation
#   internal static data.
class ProviderDataPresenter

  METHODS = [:id, :npi, :name, :physician_name, :address, :city, :state,
    :zip, :phone, :fax, :visn, :facility, :referrals, :consultations,
    :medical_specialty_names, :formatted_address_string]

  # Consultation fields that contain VHA provider data when both Consultation
  # Provider and Referral Provider are null.
  CONSULTATION_METHOD_NAMES = {
    name: :requesting_provider_contact_name,
    physician_name: :ordering_physician_name,
    phone: :requesting_provider_telephone,
    fax: :requesting_provider_fax
  }

  # 'obj' parameter takes Provider, Referral, or Consultation objects.
  # if not Provider: get Provider from either Referral or Consultation object,
  # if present, with referral provider having precedence.  If neither object has a
  # provider then save consultation internally instead and use consultation data
  # and consultation attributes (requesting_provider_contact_name, etc.)
  def initialize(obj)
    if obj.is_a? Provider
      @provider = obj
    end
    if obj.is_a? Referral
      @provider = obj.provider
      obj = obj.consultation unless @provider.present?
    end
    if obj.is_a? Consultation
      @provider = obj.ordering_provider
      @consultation = obj unless @provider.present?
    end
  end

  # create accessors for all method names in METHODS constant
  METHODS.each do |method_name|
    define_method(method_name.to_s) { get_value method_name.to_s }
  end

private

  def get_value(attr)
    return @provider.send(attr) if @provider.present?
    if CONSULTATION_METHOD_NAMES.key?(attr.to_sym)
      return @consultation.send(CONSULTATION_METHOD_NAMES[attr.to_sym])
    end
    return nil   # not necessary but adds clarity
  end

end
