# == Schema Information
#
# Table name: providers
#
#  id             :integer          not null, primary key
#  npi            :string           not null
#  name           :string           not null
#  physician_name :string           not null
#  content        :json
#  created_at     :datetime         not null
#  updated_at     :datetime         not null
#  facility_id    :integer
#

class Provider < ActiveRecord::Base
  belongs_to :facility
  has_one :visn, through: :facility
  has_and_belongs_to_many :medical_specialties
  has_many :referrals
  has_many :consultations

  has_many :providers_users, :dependent => :destroy
  has_many :users, through: :providers_users

  # autowire_content_fields :address, :city, :state, :zip, :phone, :fax

  autowire_content_fields   :taxonomies,
                            :addresses,
                            :created_epoch,
                            :identifiers,
                            :other_names,
                            :last_updated_epoch,
                            :basic,
                            :enumeration_type

  ## Eigen Class
  class << self

    def filter(params={})
      query = all

      if params.key? :visn_id
        query = query.includes(:facility)
                     .where('facilities.visn_id = ?', params[:visn_id])
                     .references(:facilities)
      end

      if params.key? :name
        query = query.where("providers.name ilike ?", "#{params[:name]}%")
      end

      if params.key? :physician_name
        query = query.where("providers.physician_name ilike ?", "#{params[:physician_name]}%")
      end

      return query

    end

  end # class << self


  # Hash with address parameters, some of which are called from instance methods.
  def address_attrs
    { address: address, city: city, state: state, zip: zip }
  end


  def address
    addresses.nil? ? 'unknown' : addresses.first['address_1'] + ' ' + addresses.first['address_2']
  end


  def city
    addresses.nil? ? '' : addresses.first['city']
  end


  def state
    addresses.nil? ? '' : addresses.first['state']
  end


  def zip
    postal_code = addresses.nil? ? '' : addresses.first['postal_code']
    if postal_code.size > 5
      postal_code = postal_code[0,5] + '-' + postal_code[5,99] # format as zip+4
    end
    return postal_code
  end


  def medical_specialty_names
    taxonomies.nil? ? '' : taxonomies.map{|tax| tax['desc']}.join(', ')
  end


  # RWP TEMP: using helper method to make address formatting consistent in the short term.
  # In the long term, find a better way to DRY up address formatting without accessing helper methods.
  def formatted_address_string
    ApplicationController.helpers.format_address address_attrs
  end

  def phone
    addresses.nil? ? '' : addresses.first["telephone_number"]
  end

  def fax
    addresses.nil? ? '' : addresses.first["fax_number"]
  end

end
