class SymptomsController < ApplicationController
  include DataTablePaginator
  before_action :set_symptom, only: [:show, :edit, :update, :destroy, :dbq_informations_sort]

  # GET /symptoms
  # GET /symptoms.json
  def index
    @symptoms = Symptom.all.order("name ASC")
  end

  # GET /symptoms/1
  # GET /symptoms/1.json
  def show
  end

  # GET /symptoms/new
  def new
    @symptom = Symptom.new
  end

  # GET /symptoms/1/edit
  def edit
    if params[:delete_diagnosis_id].present?
      d = @symptom.diagnoses.find_by_id(params[:delete_diagnosis_id])
      @symptom.diagnoses.delete(d)
      flash.now[:notice] = "Diagnosis successfully removed."
    end

    if params[:dbq_information_id].present?
      dbq = @symptom.dbq_informations.find_by_id(params[:dbq_information_id])
      @symptom.dbq_informations.delete(dbq)
      flash.now[:notice] = "DBQ Information successfully removed."
    end

    set_diagnosis_not_added @symptom
    set_dbq_informations_not_added @symptom

    @generated_html = @symptom.build_html()
  end

  def generated_html
    @symptom = Symptom.find(params[:symptom_id])
    @generated_html = @symptom.build_html()
    render partial: 'evaluation_templates/displayed_result'
  end

  # POST /symptoms
  # POST /symptoms.json
  def create
    @symptom = Symptom.new(symptom_params)

    respond_to do |format|
      if @symptom.save
        format.html { redirect_to @symptom, notice: 'Symptom was successfully created.' }
        format.json { render action: 'show', status: :created, location: @symptom }
      else
        format.html { render action: 'new' }
        format.json { render json: @symptom.errors, status: :unprocessable_entity }
      end
    end
  end

  # PATCH/PUT /symptoms/1
  # PATCH/PUT /symptoms/1.json
  def update
    if params[:identifier].present?
      redirect_to edit_symptom_path(@symptom, :identifier => params[:identifier])
    else
      respond_to do |format|
        if @symptom.update(symptom_params)
          if params[:symptom] && params[:symptom][:diagnosis_ids]
            params[:symptom][:diagnosis_ids].each do |d|
              diagnosis = Diagnosis.find_by_id(d)
              if diagnosis
                @symptom.diagnoses << diagnosis
              end
            end
          end
          if params[:symptom] && params[:symptom][:dbq_information_ids]
            params[:symptom][:dbq_information_ids].each do |d|
              dbq_info = DbqInformation.find_by_id(d)
              if dbq_info
                @symptom.dbq_informations << dbq_info
              end
            end
          end
          @symptom.save
          format.html { redirect_to edit_symptom_path(@symptom), notice: 'Symptom was successfully updated.' }
          format.json { head :no_content }
        else
          format.html { render action: 'edit' }
          format.json { render json: @symptom.errors, status: :unprocessable_entity }
        end
      end
    end
  end

  # DELETE /symptoms/1
  # DELETE /symptoms/1.json
  def destroy
    @symptom.destroy
    respond_to do |format|
      format.html { redirect_to symptoms_url }
      format.json { head :no_content }
    end
  end

  def paged_list
    symptom_to_fields_func = -> (symptom) do
      {
        id:                 symptom.id,
        name:               symptom.name.try(:humanize),
        row_label:          symptom.label.try(:humanize),
        edit:               edit_symptom_path(symptom),
        destroy:            symptom_path(symptom)
      }
    end
    ordered_symptoms = apply_ordering_to_query(Symptom.filter(params), Symptom::FIELDS_FOR_RESULTSET, params['order'])
    return_data = datatables_json_for_query_using_func(ordered_symptoms, symptom_to_fields_func)

    render json: return_data
  end

  def dbq_informations_sort
    params[:dbq_information].each_with_index do |id, index|
      @symptom.symptoms_dbq_informations
        .where(dbq_information_id: id)
        .update_all({position: index+1})
    end
    render nothing: true
  end

  private
    # Use callbacks to share common setup or constraints between actions.
    def set_symptom
      @symptom = Symptom.find(params[:id])
    end

    def set_dbq_informations_not_added info
      @dbq_info_not_yet_added = DbqInformation.all.order("identifier ASC")
      info.dbq_informations.each do |dbq|
        @dbq_info_not_yet_added = @dbq_info_not_yet_added.reject{|i| i == dbq}
      end
    end

    def set_diagnosis_not_added info
      @diagnoses_not_yet_added = Diagnosis.all.order("name ASC")
      @symptom.diagnoses.each do |d|
        @diagnoses_not_yet_added = @diagnoses_not_yet_added.reject{|diag| diag == d}
      end
    end

    # Never trust parameters from the scary internet, only allow the white list through.
    def symptom_params
      params.require(:symptom).permit(:name, :label)
    end
end
