class FullBodyEvaluationsController < ApplicationController
  before_action :set_full_body_evaluation, only: [:show, :edit, :update, :destroy, :review]
  before_action :check_ability

  check_authorization
  authorize_resource

  # GET /full_body_evaluations
  # GET /full_body_evaluations.json
  def index
    @full_body_evaluations = FullBodyEvaluation.all
  end

  # GET /full_body_evaluations/1
  # GET /full_body_evaluations/1.json
  def show
    @claim = @full_body_evaluation.claim
    @form = @full_body_evaluation.html_document_fragment
    respond_to do |format|
      format.html

      format.xml do
        render :xml => @full_body_evaluation.to_xml
      end

      format.text do
        render :text => @full_body_evaluation.to_txt
      end

      format.pdf do
        params[:show_all].present? ? @show_all = true : @show_all = false
        render :pdf => "cui-#{@full_body_evaluation.guid}.pdf",
               :layout => "pdf.html",
               :locals => {claim: @claim, form: @form, evaluation: @full_body_evaluation}
      end
    end

  end

  # GET /full_body_evaluations/new
  def new
    @full_body_evaluation = FullBodyEvaluation.new
  end

  # GET /full_body_evaluations/1/edit
  def edit
    if(@full_body_evaluation.completed_at.nil?)
      @the_html_to_display = @full_body_evaluation.html_for_diagnoses
    else
      @the_html_to_display = "Empty"
      redirect_to full_body_evaluation_path(@full_body_evaluation)
    end
  end

  # POST /full_body_evaluations
  # POST /full_body_evaluations.json
  def create
    @full_body_evaluation = FullBodyEvaluation.new(full_body_evaluation_params)

    respond_to do |format|
      if @full_body_evaluation.save
        format.html { redirect_to @full_body_evaluation, notice: 'Full body evaluation was successfully created.' }
        format.json { render action: 'show', status: :created, location: @full_body_evaluation }
      else
        format.html { render action: 'new' }
        format.json { render json: @full_body_evaluation.errors, status: :unprocessable_entity }
      end
    end
  end

  # PATCH/PUT /full_body_evaluations/1
  # PATCH/PUT /full_body_evaluations/1.json
  def update
    the_local_params = params
    @full_body_evaluation.update!(full_body_evaluation_params)
    case params[:commit]

      when "Review"
      begin
        authorize! :review, @full_body_evaluation
        the_evaluation_was_saved = @full_body_evaluation.update_all_contents( params )   #cs merge

        @full_body_evaluation.review!

        redirect_to review_full_body_evaluation_path(@full_body_evaluation)

      rescue DependencyError => errors
        @errors = errors
        flash.now[:alert] = errors.errors[0].message
        render :edit
      rescue CredentialError
        # we only care about this exception on submissions
        flash[:notice] = "Please review your evaluation, check for errors, and hit complete when you are satisfied."
        redirect_to review_full_body_evaluation_path(@full_body_evaluation)
      end

      when "Complete"
        authorize! :complete, @full_body_evaluation
        @claim = @full_body_evaluation.claim
        @form = @full_body_evaluation.html_document_fragment
        the_local_hash = {"vha_user_electronic_signature" => params["full_body_evaluation"]["vha_user_electronic_signature"],
                          "vha_user_vista_access_code"    => params["full_body_evaluation"]["vha_user_vista_access_code"],
                          "vha_user_vista_verify_code"    => params["full_body_evaluation"]["vha_user_vista_verify_code"]     }
        @full_body_evaluation.update_attributes(the_local_hash)
        begin
          @full_body_evaluation.complete!(current_user)
        rescue DependencyError => errors
          @errors = errors
          flash.now[:alert] = errors.errors[0].message
          return render :edit
        rescue CredentialError => error
          flash.now[:alert] = error.message
          @errors = errors
          return render :review
        end
        flash[:notice] = "Evaluation completed"
        @exam = @full_body_evaluation.examination
        @exam.examination_state = ExaminationState.find_by_code("COMPLETED")
        @exam.examination_state_start_date = DateTime.current
        @exam.save
        # Send Appointment completed event
        @exam_request = ExamRequest.find_by_id(@exam.exam_request_id)
        appointmentCompletedEventNotification = AppointmentCompletedEvent.new
        eventId = appointmentCompletedEventNotification.build_notification(@exam_request.claim_id, :unused, @exam_request.site_id)
        VlerDASJob.sendNotification(appointmentCompletedEventNotification, eventId, @exam_request.claim_id, "Appointment Completed Event Notification")

        flash[:notice] = "Examination is now ready for review by supervising clinician."
        redirect_to edit_examination_path(@full_body_evaluation.examination), notice: 'Evaluation is complete.'

      else # Save, Save as Update
        the_evaluation_was_saved = @full_body_evaluation.update_all_contents( params )   #cs merge

        @the_html_to_display = @full_body_evaluation.html_for_diagnoses
        respond_to do |format|
          if the_evaluation_was_saved
            format.html { render action: 'edit' }
            format.json { head :no_content }
          else
            format.html { render action: 'edit' }
            format.json { render json: @full_body_evaluation.errors, status: :unprocessable_entity }
          end
        end
    end

  end

  def review
    @claim = @full_body_evaluation.claim
    @form = @full_body_evaluation.html_document_fragment
  end


  # DELETE /full_body_evaluations/1
  # DELETE /full_body_evaluations/1.json
  def destroy
    @full_body_evaluation.destroy
    respond_to do |format|
      format.html { redirect_to full_body_evaluations_url }
      format.json { head :no_content }
    end
  end

  private

  def check_ability
    user = @current_user
    if (!user.has_this_role_on_any_site?('Clinician') and
        !user.has_this_role_on_any_site?('Supervisory Clinician') and
        !user.has_this_role_on_any_site?('Quality Assurance')
       )
      redirect_to dashboards_path
    end
  end

  # Use callbacks to share common setup or constraints between actions.
  def set_full_body_evaluation
    @full_body_evaluation = FullBodyEvaluation.find(params[:id])
  end

  # Never trust parameters from the scary internet, only allow the white list through.
  def full_body_evaluation_params
    params.permit(:full_body_evaluation, :full_body_evaluation_spec_id, :evaluation_spec_id, :vha_user_vista_access_code, :vha_user_vista_verify_code,
                                                 :vha_user_electronic_signature, :user_id)
  end
end
