require 'rails_helper'

RSpec.describe "Contentions", type: :request do
  before do
    @user = create(:examiner)
    @supervisor = create(:supervisor)
    @claim = create :claim
    @contention = create(:contention, history: "Cthulhu", claim_folder_reviewed: true)
    @claim.contentions << @contention
  end

  describe "GET /claims/:claim_id/rejections/new" do
    it "does not allow a non-supervisor to access" do
      login_as @user
      visit new_claim_contention_rejection_path(@claim, @contention)
      expect(page.current_path).to eq root_path
      expect(page.text).to match "not authorized"
    end

    it "does not allow a supervisor to reject a claim that's not complete" do
      login_as @supervisor
      visit new_claim_contention_rejection_path(@claim, @contention)
      expect(page.current_path).to eq claims_path
      expect(page.text).to match "Cannot reject a contention that is not yet resolved"
    end

    context "with a completed claim" do
      before do
        evaluation_spec = create(:evaluation_spec)
        @evaluation = create(:evaluation, user: @user, evaluation_spec: evaluation_spec, doc: build(:diabetes_doc).doc)
        allow_any_instance_of(Evaluation).to receive(:signed?).and_return(true)
        @evaluation.complete! @user
        @claim.evaluations << @evaluation
        @contention.evaluations << @evaluation
        @contention.resolve!(@user)
      end

      it "does not allow a supervisor to reject without an explanation" do
        login_as @supervisor
        visit new_claim_contention_rejection_path(@claim, @contention)
        click_button("Reject")

        expect(page.current_path).to eq claim_contention_rejections_path(@claim, @contention)
        expect(page.text).to match "Reason can't be blank"
      end

      it "allows a supervisor to reject" do
        login_as @supervisor
        visit new_claim_contention_rejection_path(@claim, @contention)
        fill_in "rejection_reason", with: "Cthulhu"
        click_button("Reject")

        expect(page.current_path).to eq claims_path
        expect(page.text).to match "Contention has been unsubmitted and user notified"

        logout(@supervisor)
        login_as(@user)
        visit edit_claim_evaluation_path(@claim, @evaluation)
        expect(page).to have_button("Save as draft")

        visit claim_contention_path(@claim, @contention)
        expect(page.text).to match /Rejected at .*? Cthulhu/
      end

      it "shows the proper rejection if rejected multiple times" do

        login_as @supervisor
        visit new_claim_contention_rejection_path(@claim, @contention)
        fill_in "rejection_reason", with: "Cthulhu"
        click_button("Reject")

        logout(@supervisor)
        login_as(@user)
        visit review_claim_evaluation_path(@claim, @evaluation)
        click_button "Complete"
        # don't try to send to VBMS
        expect(VBMSJob).to receive(:perform_later)
        visit claim_contention_path(@claim, @contention)
        click_link "Send to Administrator for review"

        login_as @supervisor
        visit new_claim_contention_rejection_path(@claim, @contention)
        fill_in "rejection_reason", with: "Fnorg"
        click_button("Reject")

        logout(@supervisor)
        login_as(@user)
        visit edit_claim_evaluation_path(@claim, @evaluation)
        expect(page).to have_button("Save as draft")

        visit claim_contention_path(@claim, @contention)
        expect(page.text).to match /Rejected at .*? Fnorg/
      end
    end
  end
end
