require 'rails_helper'

RSpec.describe "Claims", type: :request do
  before do
    @user = create(:examiner)
    login_as @user
    @evaluation_spec = create(:evaluation_spec)
    allow_any_instance_of(Evaluation).to receive(:signed?).and_return(true)
  end

  describe "GET /claims" do
    context "when there are no incomplete claims" do
      before do
        create :claim, first_name: 'Joe', last_name: 'Veteran', completed_at: Time.now
      end

      context "when the user is a medical assistant" do
        before do
          @user.update_attributes(roles: ["medical_assistant"])
        end

        it "should show them all the claims in the system under the proper heading" do
          visit root_path
          expect(page).to have_content 'All claims'
          expect(page).to have_no_content 'Your unfinished claims'
          expect(page).to have_content 'Joe Veteran'
          expect(page).to_not have_content "Contentions ready for review"
        end
      end

      context "when the user is not a medical assistant" do
        it "should show no claims under the proper heading" do
          visit root_path
          expect(page).to have_no_content 'Your unfinished claims'
          expect(page).to have_no_content 'All claims'
          expect(page).to have_content "No incomplete claims"
        end
      end
    end

    context "when there are incomplete claims" do
      before do
        create :claim, first_name: 'Joe', last_name: 'Veteran'
        create :claim, first_name: 'Jane', last_name: 'Veteran'
        create :claim, first_name: 'Jack', last_name: 'Veteran', completed_at: Time.now
      end

      context "when the user is a medical assistant" do
        before do
          @user.update_attributes(roles: ["medical_assistant"])
          visit root_path
        end

        it "should show them all the claims in the system under the proper heading" do
          expect(page).to have_content 'All claims'
          expect(page).to have_no_content 'Your unfinished claims'
          expect(page).to have_content 'Joe Veteran'
          expect(page).to have_content 'Jane Veteran'
          expect(page).to have_content 'Jack Veteran'
        end

        it "should show a link to create a new claim" do
          expect(page).to have_link 'Create new claim', href: new_claim_path
        end
      end

      context "when the user is a supervisor (but not a medical assistant) and there are resolved contentions" do
        before do
          @supervisor = create(:supervisor)
          login_as @supervisor
        end

        after do
          login_as @user
        end

        it "should show them all the claims in the system under the proper heading" do
          visit root_path
          expect(page).to have_content 'All claims'
          expect(page).to have_no_content 'Your unfinished claims'
          expect(page).to have_content 'Joe Veteran'
          expect(page).to have_content "Contentions ready for review"
        end

        context "and there are resolved contentions" do
          before do
            @claim = create(:claim, first_name: "Joe", last_name: "Resolved")
            @contention = create(:contention, assigner: @supervisor, history: "Patient History",
                                 claim_folder_reviewed: true)
            @claim.contentions << @contention
            @evaluation_spec = create(:diabetes_spec)
            @evaluation = Evaluation.new(claim: @claim, evaluation_spec: @evaluation_spec, user: create(:user))
            @contention.evaluations << @evaluation
            @evaluation.complete! @supervisor
            @contention.resolve!(@supervisor)
            visit root_path
          end

          it "should show the reviewable contentions table" do
            expect(page).to have_content "Contentions ready for review"
            expect(page).to have_link @contention.name, href: claim_contention_path(@claim, @contention)
          end

          it "should show them all the claims in the system under the proper heading" do
            expect(page).to have_content 'All claims'
            expect(page).to have_no_content 'Your unfinished claims'
            expect(page).to have_content 'Joe Veteran'
            expect(page).to have_content 'Jane Veteran'
            expect(page).to have_content 'Jack Veteran'
          end

          it "should allow the supervisor to view the claim's contention" do
            click_link 'Joe Veteran'
            expect(page).to have_content 'Contentions'
            expect(page).to have_content 'Create new contention'
          end

          it "should show a link to create a new claim" do
            expect(page).to have_link 'Create new claim', href: new_claim_path
          end
        end
      end

      context "when one claim has an unfinished evaluation" do
        before do
          @evaluation = Evaluation.new(claim: @claim, evaluation_spec: @evaluation_spec, user: @user)
          Claim.first.evaluations << @evaluation
        end

        it "should show the user only the claim with that evaluation" do
          visit root_path
          expect(page).to have_content 'Your unfinished claims'
          expect(page).to have_no_content 'All claims'
          expect(page).to have_link 'Joe Veteran'
          expect(page).to have_no_content 'Jane Veteran'
          expect(page).to have_no_content 'Jack Veteran'
        end

        it "should not show any claims to a user without any evaluations" do
          login_as create(:user)
          visit root_path
          expect(page).to have_no_content 'Your unfinished claims'
          expect(page).to have_no_content 'All claims'
          expect(page).to have_no_content 'Joe Veteran'
          expect(page).to have_no_content 'Jane Veteran'
          expect(page).to have_no_content 'Jack Veteran'

          # return the login to @user
          login_as @user
        end
      end
    end

    context "when the user is not a medical assistant" do
      it "should not show a link to create a new claim" do
        visit root_path
        expect(page).to have_no_link 'Create new claim', href: new_claim_path
      end
    end

    context "when the user is a medical assistant" do
      before do
        @user.roles = ["medical_assistant"]
      end

      it "should show a link to create a new claim" do
        visit root_path
        expect(page).to have_link 'Create new claim', href: new_claim_path
      end
    end
  end

  describe "GET /claims/new" do
    context "when the user is not a medical assistant" do
      it "should not show the user a link to create a new claim or let them visit the new claim page" do
        visit new_claim_path
        expect(page).to have_content "You are not authorized to access this page."
      end
    end

    context "when the user is a medical assistant" do
      before do
        @user.roles = ["medical_assistant"]
      end

      context "when all fields are filled in" do
        it "should allow the user to create a new claim" do
          visit new_claim_path
          fill_in 'First name', with: 'Joe'
          fill_in 'Last name', with: 'Veteran'
          fill_in 'Patient Social Security Number', with: '123-45-6789'
          fill_in 'File number', with: '123456789'
          click_button 'Create claim'
          expect(page).to have_content 'Claim created.'
          expect(page).to have_content 'Claim: Joe Veteran'
        end
      end

      context "when required information is missing" do
        it "should display an error message to the user" do
          visit new_claim_path
          fill_in 'First name', with: "Joe"
          fill_in 'Last name', with: "Veteran"
          click_button 'Create claim'
          expect(page).to have_content "Error creating claim. Patient ssn can't be blank File number can't be blank"
        end
      end

      context "when adding a contention along with the new claim" do
#        it "should allow the user to add any number of contentions", js: true do
#          visit new_claim_path
#          fill_in 'First name', with: 'Joe'
#          fill_in 'Last name', with: 'Veteran'
#          fill_in 'Patient Social Security Number', with: '123456789'
#          fill_in 'File number', with: '123456789'
#          fill_in 'claim_contentions_attributes_0_name', with: 'My head hurts'
#          click_link 'Add another contention'
#          fill_in 'claim_contentions_attributes_1_name', with: 'My face hurts'
#          click_button 'Create claim'
#          expect(page).to have_content 'Claim created'
#          expect(page).to have_content 'Claim: Joe Veteran'
#          expect(page).to have_content 'My head hurts'
#          expect(page).to have_content 'My face hurts'
#        end

        it "should require the user to input a name when adding a contention" do
          visit new_claim_path
          fill_in 'First name', with: 'Joe'
          fill_in 'Last name', with: 'Veteran'
          fill_in 'Patient Social Security Number', with: '123456789'
          fill_in 'File number', with: '123456789'
          fill_in 'claim_contentions_attributes_0_name', with: 'My head hurts'
          click_button 'Create claim'
          expect(page).to have_content 'Claim created'
          expect(page).to have_content 'My head hurts'
        end
      end
    end
  end

  describe "GET /claims/:id" do

    context "when the user is an examiner" do
      before do
        @claim = create :claim
        @claim_without_exam_request_data = create :claim_without_exam_request_data
        @other_user = create(:examiner, email: "examiner@DNS   ", first_name: "An", last_name: "Examiner")
      end

      context "when there is exam request data for the claim" do

        it "should show a link to see the exam request data" do
          visit claim_path(@claim)
          expect(page).to have_link 'Show Exam Request Data', href: details_claim_path(@claim)
        end

        context "when the examiner clicks on the Show Exam Request Data link" do
          it "should show the exam request data" do
            visit claim_path(@claim)
            click_link 'Show Exam Request Data'
            expect(page).to have_content "I am exam request data"
          end
        end
      end

      context "when there is no exam request data for the claim" do
        it "should not show a link to exam request data, but it should show the message This Claim Has No Exam Request Data" do
          visit claim_path(@claim_without_exam_request_data)
          expect(page).to have_no_link 'Show Exam Request Data'
          expect(page).to have_content "This claim has no exam request data."
        end
      end
    end

    context "when the user is a medical assistant" do
      before do
        @claim = create :claim
        @claim_without_exam_request_data = create :claim_without_exam_request_data
        @other_user = create(:medical_assistant, email: "medical_assistant@DNS   ", first_name: "Medical", last_name: "Assistant")
      end

      context "when there is exam request data for the claim" do
        it "should show a link to see the exam request data" do
          visit claim_path(@claim)
          expect(page).to have_link 'Show Exam Request Data', href: details_claim_path(@claim)
        end

        context "when the medical_assistant clicks on the Show Exam Request Data link" do
          it "should show the exam request data" do
            visit claim_path(@claim)
            click_link 'Show Exam Request Data'
            expect(page).to have_content "I am exam request data"
          end
        end
      end

      context "when there is no exam request data for the claim" do
        it "should not show a link to exam request data, but should show the message This Claim Has No Exam Request Data" do
          visit claim_path(@claim_without_exam_request_data)
          expect(page).to have_no_link 'Show Exam Request Data'
          expect(page).to have_content "This claim has no exam request data."
        end
      end
    end

    context "when a claim is unresolved and has contentions" do
      before do
        @other_user = create(:examiner, email: "otherdoc@DNS   ", first_name: "Doc", last_name: "Blue")
        @claim = create :claim
        @claim.contentions << create(:contention)
        @claim.contentions << create(:contention, name: "My arm hurts.", resolved_at: Time.now, vba_diagnostic_code: "123")
      end

      context "when the contentions have evaluations" do
        before do
          @claim.contentions.first.evaluations << Evaluation.new(evaluation_spec: @evaluation_spec, claim: @claim, user: @user)
        end

        it "should show a link to edit the evaluation" do
          visit claim_path(@claim)
          expect(page).to have_link 'Simple Yes/No Evaluation', href: edit_claim_evaluation_path(@claim, @claim.contentions.first.evaluations.first)
        end

        it "should show a link to evaluation activity" do
          visit claim_path(@claim)
          expect(page).to have_link 'activity_log', href: activity_claim_evaluation_path(@claim, @claim.evaluations.first)
        end

        context "when the evaluation has been completed" do
          before do
            @claim.contentions.first.evaluations.first.complete! @other_user
          end

          it "should show a link to a non-editable version of the evaluation" do
            visit claim_path(@claim)
            expect(page).to have_link 'Simple Yes/No Evaluation', href: claim_evaluation_path(@claim, @claim.contentions.first.evaluations.first)
          end
        end
      end

      it "should show the claim, and link to all contentions" do
        visit claim_path(@claim)
        expect(page).to have_content "Claim: #{@claim.patient_name} #{@claim.created_at.to_date}"
        expect(page).to have_content "SSN: #{@claim.masked_ssn}"
        expect(page).to have_content "Contentions"
        expect(page).to have_link @claim.contentions.first.name, href: claim_contention_path(@claim, @claim.contentions.first)
        expect(page).to have_content "Exam Iner"
        expect(page).to have_content "Unresolved"
        expect(page).to have_link @claim.contentions.last.name, href: claim_contention_path(@claim, @claim.contentions.last)
        expect(page).to have_content "Resolved"
      end
    end
  end
end
